"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.discoverTaskCommands = void 0;
const chalk = require("chalk");
const logging = require("../logging");
/**
 * Reads .projen/tasks.json and adds CLI commands for all tasks.
 * @param ya yargs
 */
function discoverTaskCommands(runtime, ya) {
    var _a, _b;
    const tasks = (_a = runtime.manifest.tasks) !== null && _a !== void 0 ? _a : {};
    for (const task of Object.values(tasks)) {
        ya.command(task.name, (_b = task.description) !== null && _b !== void 0 ? _b : '', taskCommandHandler(task.name));
    }
    function taskCommandHandler(taskName) {
        return (args) => {
            args.option('inspect', { alias: 'i', desc: 'show all steps in this task' });
            const argv = args.argv;
            if (argv.inspect) {
                return inspectTask(taskName);
            }
            else {
                try {
                    runtime.runTask(taskName);
                }
                catch (e) {
                    logging.error(e.message);
                    process.exit(1);
                }
            }
        };
    }
    function inspectTask(name, indent = 0) {
        var _a, _b;
        const writeln = (s) => console.log(' '.repeat(indent) + s);
        const task = runtime.tryFindTask(name);
        if (!task) {
            throw new Error(`${name}: unable to resolve subtask with name "${name}"`);
        }
        if (task.description) {
            writeln(`${chalk.underline('description')}: ${task.description}`);
        }
        for (const [k, v] of Object.entries((_a = task.env) !== null && _a !== void 0 ? _a : {})) {
            writeln(`${chalk.underline('env')}: ${k}=${v}`);
        }
        for (const step of (_b = task.steps) !== null && _b !== void 0 ? _b : []) {
            if (step.spawn) {
                writeln(`- ${chalk.bold(step.spawn)}`);
                inspectTask(step.spawn, indent + 2);
            }
            else if (step.exec) {
                writeln(`- exec: ${step.exec}`);
            }
            else if (step.builtin) {
                writeln(`- builtin: ${step.builtin}`);
            }
        }
    }
}
exports.discoverTaskCommands = discoverTaskCommands;
//# sourceMappingURL=data:application/json;base64,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