"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const fs = require("fs-extra");
const yargs = require("yargs");
const inventory = require("../../inventory");
const option_hints_1 = require("../../option-hints");
const projects_1 = require("../../projects");
const util_1 = require("../../util");
const macros_1 = require("../macros");
const util_2 = require("../util");
class Command {
    constructor() {
        this.command = 'new [PROJECT-TYPE-NAME] [OPTIONS]';
        this.describe = 'Creates a new projen project';
    }
    builder(args) {
        var _a;
        args.positional('PROJECT-TYPE-NAME', { describe: 'optional only when --from is used and there is a single project type in the external module', type: 'string' });
        args.option('synth', { type: 'boolean', default: true, desc: 'Synthesize after creating .projenrc.js' });
        args.option('comments', { type: 'boolean', default: true, desc: 'Include commented out options in .projenrc.js (use --no-comments to disable)' });
        args.option('from', { type: 'string', alias: 'f', desc: 'External jsii npm module to create project from. Supports any package spec supported by npm (such as "my-pack@^2.0")' });
        args.option('git', { type: 'boolean', default: true, desc: 'Run `git init` and create an initial commit (use --no-git to disable)' });
        args.example('projen new awscdk-app-ts', 'Creates a new project of built-in type "awscdk-app-ts"');
        args.example('projen new --from projen-vue@^2', 'Creates a new project from an external module "projen-vue" with the specified version');
        for (const type of inventory.discover()) {
            args.command(type.pjid, (_a = type.docs) !== null && _a !== void 0 ? _a : '', {
                builder: cargs => {
                    var _a, _b, _c;
                    cargs.showHelpOnFail(false);
                    for (const option of (_a = type.options) !== null && _a !== void 0 ? _a : []) {
                        if (option.simpleType !== 'string' && option.simpleType !== 'number' && option.simpleType !== 'boolean' && option.kind !== 'enum') {
                            continue; // we only support primitive and enum fields as command line options
                        }
                        let desc = [(_c = (_b = option.docs) === null || _b === void 0 ? void 0 : _b.replace(/\ *\.$/, '')) !== null && _c !== void 0 ? _c : ''];
                        const required = !option.optional;
                        let defaultValue;
                        if (option.default && option.default !== 'undefined') {
                            if (!required) {
                                // if the field is not required, just describe the default but don't actually assign a value
                                desc.push(`[default: ${option.default.replace(/^\ *-/, '').replace(/\.$/, '').trim()}]`);
                            }
                            else {
                                // if the field is required and we have a @default, then assign
                                // the value here so it appears in `--help`
                                defaultValue = renderDefault(process.cwd(), option.default);
                            }
                        }
                        const argType = option.kind === 'enum' ? 'string' : option.simpleType;
                        cargs.option(option.switch, {
                            group: required ? 'Required:' : 'Optional:',
                            type: argType,
                            description: desc.join(' '),
                            default: defaultValue,
                            required,
                        });
                    }
                    return cargs;
                },
                handler: argv => newProject(process.cwd(), type, argv),
            });
        }
        return args;
    }
    async handler(args) {
        // handle --from which means we want to first install a jsii module and then
        // create a project defined within this module.
        if (args.from) {
            return newProjectFromModule(process.cwd(), args.from, args);
        }
        // project type is defined but was not matched by yargs, so print the list of supported types
        if (args.projectTypeName) {
            console.log(`Invalid project type ${args.projectTypeName}. Supported types:`);
            for (const pjid of inventory.discover().map(x => x.pjid)) {
                console.log(`  ${pjid}`);
            }
            return;
        }
        // Handles the use case that nothing was specified since PROJECT-TYPE is now an optional positional parameter
        yargs.showHelp();
    }
}
/**
 * Given a value from "@default", processes macros and returns a stringified
 * (quoted) result.
 *
 * @returns a javascript primitive (could be a string, number or boolean)
 */
function renderDefault(cwd, value) {
    var _a;
    return (_a = macros_1.tryProcessMacro(cwd, value)) !== null && _a !== void 0 ? _a : JSON.parse(value);
}
/**
 * Converts yargs command line switches to project type props.
 * @param type Project type
 * @param argv Command line switches
 */
function commandLineToProps(cwd, type, argv) {
    var _a;
    const props = {};
    // initialize props with default values
    for (const prop of type.options) {
        if (prop.default && prop.default !== 'undefined' && !prop.optional) {
            props[prop.name] = renderDefault(cwd, prop.default);
        }
    }
    for (const [arg, value] of Object.entries(argv)) {
        for (const prop of type.options) {
            if (prop.switch === arg) {
                let curr = props;
                const queue = [...prop.path];
                while (true) {
                    const p = queue.shift();
                    if (!p) {
                        break;
                    }
                    if (queue.length === 0) {
                        curr[p] = value;
                    }
                    else {
                        curr[p] = (_a = curr[p]) !== null && _a !== void 0 ? _a : {};
                        curr = curr[p];
                    }
                }
            }
        }
    }
    return props;
}
/**
 * Generates a new project from an external module.
 *
 * @param spec The name of the external module to load
 * @param args Command line arguments (incl. project type)
 */
async function newProjectFromModule(baseDir, spec, args) {
    var _a, _b;
    const projenVersion = (_a = args.projenVersion) !== null && _a !== void 0 ? _a : 'latest';
    const installCommand = util_2.renderInstallCommand(baseDir, `projen@${projenVersion}`);
    if (args.projenVersion) {
        util_1.exec(installCommand, { cwd: baseDir });
    }
    else {
        // do not overwrite existing installation
        util_1.exec(`npm ls --prefix=${baseDir} --depth=0 --pattern projen || ${installCommand}`, { cwd: baseDir });
    }
    const moduleName = util_2.installPackage(baseDir, spec);
    // Find the just installed package and discover the rest recursively from this package folder
    const moduleDir = path.dirname(require.resolve(`${moduleName}/.jsii`, {
        paths: [
            baseDir,
        ],
    }));
    // Only leave projects from the main (requested) package
    const projects = inventory
        .discover(moduleDir)
        .filter(x => x.moduleName === moduleName); // Only list project types from the requested 'from' module
    if (projects.length < 1) {
        throw new Error(`No projects found after installing ${spec}. The module must export at least one class which extends projen.Project`);
    }
    const requested = args.projectTypeName;
    const types = projects.map(p => p.pjid);
    // if user did not specify a project type but the module has more than one, we need them to tell us which one...
    if (!requested && projects.length > 1) {
        throw new Error(`Multiple projects found after installing ${spec}: ${types.join(',')}. Please specify a project name.\nExample: npx projen new --from ${spec} ${types[0]}`);
    }
    // if user did not specify a type (and we know we have only one), the select it. otherwise, search by pjid.
    const type = !requested ? projects[0] : projects.find(p => p.pjid === requested);
    if (!type) {
        throw new Error(`Project type ${requested} not found. Found ${types.join(',')}`);
    }
    for (const option of (_b = type.options) !== null && _b !== void 0 ? _b : []) {
        if (option.simpleType !== 'string' && option.simpleType !== 'number' && option.simpleType !== 'boolean') {
            continue; // we don't support non-primitive fields as command line options
        }
        if (args[option.name] !== undefined) {
            if (option.simpleType === 'number') {
                args[option.name] = parseInt(args[option.name]);
                args[option.switch] = args[option.name];
            }
            else if (option.simpleType === 'boolean') {
                const raw = args[option.name];
                const safe = typeof raw === 'string' ? util_1.isTruthy(raw) : raw;
                args[option.name] = safe;
                args[option.switch] = safe;
            }
            continue; // do not overwrite passed arguments
        }
        if (option.default && option.default !== 'undefined') {
            if (!option.optional) {
                const defaultValue = renderDefault(baseDir, option.default);
                args[option.name] = defaultValue;
                args[option.switch] = defaultValue;
            }
        }
    }
    // include a dev dependency for the external module
    args.devDeps = [spec];
    args['dev-deps'] = [spec];
    await newProject(baseDir, type, args);
}
/**
 * Generates a new project.
 * @param type Project type
 * @param args Command line arguments
 * @param additionalProps Additional parameters to include in .projenrc.js
 */
async function newProject(baseDir, type, args) {
    var _a;
    // convert command line arguments to project props using type information
    const props = commandLineToProps(baseDir, type, args);
    projects_1.Projects.createProject({
        dir: (_a = props.outdir) !== null && _a !== void 0 ? _a : baseDir,
        projectFqn: type.fqn,
        projectOptions: props,
        optionHints: args.comments ? option_hints_1.NewProjectOptionHints.FEATURED : option_hints_1.NewProjectOptionHints.NONE,
        synth: args.synth,
        post: args.post,
    });
    if (fs.existsSync(path.join(baseDir, 'package.json')) && args.post) {
        util_1.exec('npm run eslint --if-present', { cwd: baseDir });
    }
    if (args.git) {
        const git = (cmd) => util_1.exec(`git ${cmd}`, { cwd: baseDir });
        git('init');
        git('add .');
        git('commit --allow-empty -m "chore: project created with projen"');
        git('branch -M main');
    }
}
module.exports = new Command();
//# sourceMappingURL=data:application/json;base64,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