"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaRuntime = exports.LambdaFunction = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path_1 = require("path");
const case_1 = require("case");
const component_1 = require("../component");
const file_1 = require("../file");
const javascript_1 = require("../javascript");
const source_code_1 = require("../source-code");
const internal_1 = require("./internal");
/**
 * (experimental) Generates a pre-bundled AWS Lambda function construct from handler code.
 *
 * To use this, create an AWS Lambda handler file under your source tree with
 * the `.lambda.ts` extension and add a `LambdaFunction` component to your
 * typescript project pointing to this entrypoint.
 *
 * This will add a task to your "compile" step which will use `esbuild` to
 * bundle the handler code into the build directory. It will also generate a
 * file `src/foo-function.ts` with a custom AWS construct called `FooFunction`
 * which extends `@aws-cdk/aws-lambda.Function` which is bound to the bundled
 * handle through an asset.
 *
 * @experimental
 * @example
 *
 * new LambdaFunction(myProject, {
 *   srcdir: myProject.srcdir,
 *   entrypoint: 'src/foo.lambda.ts',
 * });
 */
class LambdaFunction extends component_1.Component {
    /**
     * (experimental) Defines a pre-bundled AWS Lambda function construct from handler code.
     *
     * @param project The project to use.
     * @param options Options.
     * @experimental
     */
    constructor(project, options) {
        var _c, _d, _e;
        super(project);
        const bundler = javascript_1.Bundler.of(project);
        if (!bundler) {
            throw new Error('No bundler found. Please add a Bundler component to your project.');
        }
        const runtime = (_c = options.runtime) !== null && _c !== void 0 ? _c : LambdaRuntime.NODEJS_14_X;
        // allow Lambda handler code to import dev-deps since they are only needed
        // during bundling
        const eslint = javascript_1.Eslint.of(project);
        eslint === null || eslint === void 0 ? void 0 : eslint.allowDevDeps(options.entrypoint);
        const entrypoint = options.entrypoint;
        if (!entrypoint.endsWith(internal_1.TYPESCRIPT_LAMBDA_EXT)) {
            throw new Error(`${entrypoint} must have a ${internal_1.TYPESCRIPT_LAMBDA_EXT} extension`);
        }
        const basePath = path_1.join(path_1.dirname(entrypoint), path_1.basename(entrypoint, internal_1.TYPESCRIPT_LAMBDA_EXT));
        const constructFile = (_d = options.constructFile) !== null && _d !== void 0 ? _d : `${basePath}-function.ts`;
        if (path_1.extname(constructFile) !== '.ts') {
            throw new Error(`Construct file name "${constructFile}" must have a .ts extension`);
        }
        // type names
        const constructName = (_e = options.constructName) !== null && _e !== void 0 ? _e : case_1.pascal(path_1.basename(basePath)) + 'Function';
        const propsType = `${constructName}Props`;
        const bundle = bundler.addBundle(entrypoint, {
            target: runtime.esbuildTarget,
            platform: runtime.esbuildPlatform,
            externals: ['aws-sdk'],
            ...options.bundlingOptions,
        });
        // calculate the relative path between the directory containing the
        // generated construct source file to the directory containing the bundle
        // index.js by resolving them as absolute paths first.
        // e.g:
        //  - outfileAbs => `/project-outdir/assets/foo/bar/baz/foo-function/index.js`
        //  - constructAbs => `/project-outdir/src/foo/bar/baz/foo-function.ts`
        const outfileAbs = path_1.join(project.outdir, bundle.outfile);
        const constructAbs = path_1.join(project.outdir, constructFile);
        const relativeOutfile = path_1.relative(path_1.dirname(constructAbs), path_1.dirname(outfileAbs));
        const src = new source_code_1.SourceCode(project, constructFile);
        src.line(`// ${file_1.FileBase.PROJEN_MARKER}`);
        src.line('import * as path from \'path\';');
        src.line('import * as lambda from \'@aws-cdk/aws-lambda\';');
        src.line('import { Construct } from \'@aws-cdk/core\';');
        src.line();
        src.line('/**');
        src.line(` * Props for ${constructName}`);
        src.line(' */');
        src.open(`export interface ${propsType} extends lambda.FunctionOptions {`);
        src.close('}');
        src.line();
        src.line('/**');
        src.line(` * An AWS Lambda function which executes ${convertToPosixPath(basePath)}.`);
        src.line(' */');
        src.open(`export class ${constructName} extends lambda.Function {`);
        src.open(`constructor(scope: Construct, id: string, props?: ${propsType}) {`);
        src.open('super(scope, id, {');
        src.line(`description: '${convertToPosixPath(entrypoint)}',`);
        src.line('...props,');
        src.line(`runtime: lambda.Runtime.${runtime.functionRuntime},`);
        src.line('handler: \'index.handler\',');
        src.line(`code: lambda.Code.fromAsset(path.join(__dirname, '${convertToPosixPath(relativeOutfile)}')),`);
        src.close('});');
        src.close('}');
        src.close('}');
        this.project.logger.verbose(`${basePath}: construct "${constructName}" generated under "${constructFile}"`);
        this.project.logger.verbose(`${basePath}: bundle task "${bundle.bundleTask.name}"`);
        if (bundle.watchTask) {
            this.project.logger.verbose(`${basePath}: bundle watch task "${bundle.watchTask.name}"`);
        }
    }
}
exports.LambdaFunction = LambdaFunction;
_a = JSII_RTTI_SYMBOL_1;
LambdaFunction[_a] = { fqn: "projen.awscdk.LambdaFunction", version: "0.40.1" };
/**
 * (experimental) The runtime for the AWS Lambda function.
 *
 * @experimental
 */
class LambdaRuntime {
    constructor(
    /**
     * The aws-lambda.Runtime member name to use.
     */
    functionRuntime, 
    /**
     * The esbuild setting to use.
     */
    esbuildTarget) {
        this.functionRuntime = functionRuntime;
        this.esbuildTarget = esbuildTarget;
        /**
         * @experimental
         */
        this.esbuildPlatform = 'node';
    }
}
exports.LambdaRuntime = LambdaRuntime;
_b = JSII_RTTI_SYMBOL_1;
LambdaRuntime[_b] = { fqn: "projen.awscdk.LambdaRuntime", version: "0.40.1" };
/**
 * (experimental) Node.js 10.x.
 *
 * @experimental
 */
LambdaRuntime.NODEJS_10_X = new LambdaRuntime('NODEJS_10_X', 'node10');
/**
 * (experimental) Node.js 12.x.
 *
 * @experimental
 */
LambdaRuntime.NODEJS_12_X = new LambdaRuntime('NODEJS_12_X', 'node12');
/**
 * (experimental) Node.js 14.x.
 *
 * @experimental
 */
LambdaRuntime.NODEJS_14_X = new LambdaRuntime('NODEJS_14_X', 'node14');
/**
 * Converts the given path string to posix if it wasn't already.
 */
function convertToPosixPath(p) {
    return p.split(path_1.sep).join(path_1.posix.sep);
}
//# sourceMappingURL=data:application/json;base64,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