"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Monitoring = void 0;
const cw = require("@aws-cdk/aws-cloudwatch");
const core_1 = require("@aws-cdk/core");
const cdk_watchful_1 = require("cdk-watchful");
const web_canary_1 = require("./web-canary");
/**
 * Construct hub monitoring.
 *
 * This construct uses cdk-watchful to automatically generate a dashboard and a
 * set of standard alarms for common resources.
 *
 * Components should use the APIs of this module to add watchful monitors,
 * and add canaries and alarms as needed.
 */
class Monitoring extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.alarmActions = props.alarmActions;
        this.watchful = new cdk_watchful_1.Watchful(this, 'Watchful', {
            dashboardName: props.dashboardName,
            // alarms that come from watchful are all considered normal severity
            alarmActionArns: this.alarmActions.normalSeverity ? [this.alarmActions.normalSeverity] : [],
        });
        this.highSeverityDashboard = new cw.Dashboard(this, 'Dashboard', {
            dashboardName: `${props.dashboardName}-high-severity`,
        });
    }
    /**
     * Adds a high-severity alarm. If this alarm goes off, the action specified in `highSeverityAlarmActionArn`
     * @param alarm
     */
    addHighSeverityAlarm(title, alarm) {
        alarm.addAlarmAction({
            bind: () => ({ alarmActionArn: this.alarmActions.highSeverity }),
        });
        this.highSeverityDashboard.addWidgets(new cw.AlarmWidget({
            alarm,
            title,
            width: 24,
        }));
    }
    /**
     * Adds a canary that pings a certain URL and raises an alarm in case the URL
     * responds with an error over 80% of the times.
     *
     * Canary failures are always considered high-severity (at this point).
     *
     * @param name The name of the canary
     * @param url The URL to ping
     */
    addWebCanary(name, url) {
        const canary = new web_canary_1.WebCanary(this, `WebCanary${name.replace(/[^A-Z0-9]/ig, '')}`, {
            url,
            displayName: name,
        });
        this.addHighSeverityAlarm(`${name} Canary`, canary.alarm);
    }
}
exports.Monitoring = Monitoring;
//# sourceMappingURL=data:application/json;base64,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