"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const os = require("os");
const path = require("path");
const fs = require("fs-extra");
const docgen = require("jsii-docgen");
const aws = require("../shared/aws.lambda-shared");
const code_artifact_lambda_shared_1 = require("../shared/code-artifact.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const language_1 = require("../shared/language");
const ASSEMBLY_KEY_REGEX = new RegExp(`^${constants.STORAGE_KEY_PREFIX}((?:@[^/]+/)?[^/]+)/v([^/]+)${constants.ASSEMBLY_KEY_SUFFIX}$`);
// Capture groups:                                                    ┗━━━━━━━━━1━━━━━━━┛  ┗━━2━━┛
/**
 * This function receives an S3 event, and for each record, proceeds to download
 * the `.jsii` assembly the event refers to, transliterates it to Python, then
 * uploads the resulting `.jsii.python` object to S3.
 *
 * @param event   an S3 event payload
 * @param context a Lambda execution context
 *
 * @returns nothing
 */
function handler(event, context) {
    console.log(JSON.stringify(event, null, 2));
    // We'll need a writable $HOME directory, or this won't work well, because
    // npm will try to write stuff like the `.npmrc` or package caches in there
    // and that'll bail out on EROFS if that fails.
    return withFakeHome(async () => {
        var _a, _b;
        const endpoint = process.env.CODE_ARTIFACT_REPOSITORY_ENDPOINT;
        if (!endpoint) {
            console.log('No CodeArtifact endpoint configured - using npm\'s default registry');
        }
        else {
            console.log(`Using CodeArtifact registry: ${endpoint}`);
            const domain = env_lambda_shared_1.requireEnv('CODE_ARTIFACT_DOMAIN_NAME');
            const domainOwner = process.env.CODE_ARTIFACT_DOMAIN_OWNER;
            const apiEndpoint = process.env.CODE_ARTIFACT_API_ENDPOINT;
            await code_artifact_lambda_shared_1.logInWithCodeArtifact({ endpoint, domain, domainOwner, apiEndpoint });
        }
        const language = env_lambda_shared_1.requireEnv('TARGET_LANGUAGE');
        const created = new Array();
        const [, packageName, packageVersion] = (_a = event.assembly.key.match(ASSEMBLY_KEY_REGEX)) !== null && _a !== void 0 ? _a : [];
        if (packageName == null) {
            throw new Error(`Invalid object key: "${event.assembly.key}". It was expected to match ${ASSEMBLY_KEY_REGEX}!`);
        }
        const packageFqn = `${packageName}@${packageVersion}`;
        console.log(`Source Bucket:  ${event.bucket}`);
        console.log(`Source Key:     ${event.assembly.key}`);
        console.log(`Source Version: ${event.assembly.versionId}`);
        console.log(`Fetching assembly: ${event.assembly.key}`);
        const assemblyResponse = await aws.s3().getObject({ Bucket: event.bucket, Key: event.assembly.key }).promise();
        if (!assemblyResponse.Body) {
            throw new Error(`Response body for assembly at key ${event.assembly.key} is empty`);
        }
        const assembly = JSON.parse(assemblyResponse.Body.toString('utf-8'));
        const submodules = Object.keys((_b = assembly.submodules) !== null && _b !== void 0 ? _b : {}).map(s => s.split('.')[1]);
        if (language !== 'typescript' && assembly.targets[language] == null) {
            console.error(`Package ${assembly.name}@${assembly.version} does not support ${language}, skipping!`);
            console.log(`Assembly targets: ${JSON.stringify(assembly.targets, null, 2)}`);
            for (const submodule of [undefined, ...submodules]) {
                const key = event.assembly.key.replace(/\/[^/]+$/, constants.docsKeySuffix(language_1.DocumentationLanguage.fromString(language), submodule)) + constants.NOT_SUPPORTED_SUFFIX;
                const response = await uploadFile(context, event.bucket, key, event.assembly.versionId);
                created.push({ bucket: event.bucket, key, versionId: response.VersionId });
            }
            return created;
        }
        async function generateDocs(lang) {
            const uploads = new Map();
            const docs = await docgen.Documentation.forPackage(packageFqn, { language: docgen.Language.fromString(lang) });
            function renderAndDispatch(submodule) {
                console.log(`Rendering documentation in ${lang} for ${packageFqn} (submodule: ${submodule})`);
                const page = docs.render({ submodule, linkFormatter: linkFormatter(docs) }).render();
                const key = event.assembly.key.replace(/\/[^/]+$/, constants.docsKeySuffix(language_1.DocumentationLanguage.fromString(lang), submodule));
                console.log(`Uploading ${key}`);
                const upload = uploadFile(context, event.bucket, key, event.assembly.versionId, page);
                uploads.set(key, upload);
            }
            renderAndDispatch();
            for (const submodule of submodules) {
                renderAndDispatch(submodule);
            }
            for (const [key, upload] of uploads.entries()) {
                const response = await upload;
                created.push({ bucket: event.bucket, key, versionId: response.VersionId });
                console.log(`Finished uploading ${key} (Version ID: ${response.VersionId})`);
            }
        }
        await generateDocs(language);
        return created;
    });
}
exports.handler = handler;
async function withFakeHome(cb) {
    const fakeHome = await fs.mkdtemp(path.join(os.tmpdir(), 'fake-home'));
    const oldHome = process.env.HOME;
    try {
        process.env.HOME = fakeHome;
        return await cb();
    }
    finally {
        process.env.HOME = oldHome;
        await fs.remove(fakeHome);
    }
}
function uploadFile(context, bucket, key, sourceVersionId, body) {
    return aws.s3().putObject({
        Bucket: bucket,
        Key: key,
        Body: body,
        CacheControl: 'public',
        ContentType: 'text/markdown; charset=UTF-8',
        Metadata: {
            'Origin-Version-Id': sourceVersionId !== null && sourceVersionId !== void 0 ? sourceVersionId : 'N/A',
            'Lambda-Log-Group': context.logGroupName,
            'Lambda-Log-Stream': context.logStreamName,
            'Lambda-Run-Id': context.awsRequestId,
        },
    }).promise();
}
/**
 * A link formatter to make sure type links redirect to the appropriate package
 * page in the webapp.
 */
function linkFormatter(docs) {
    function _formatter(type) {
        const packageName = type.source.assembly.name;
        const packageVersion = type.source.assembly.version;
        // the webapp sanitizes anchors - so we need to as well when
        // linking to them.
        const hash = sanitize(type.fqn);
        if (docs.assembly.name === packageName) {
            // link to the same package - just add the hash
            return `#${hash}`;
        }
        // cross link to another package
        return `/packages/${packageName}/v/${packageVersion}?lang=${type.language.toString()}${type.submodule ? `&submodule=${type.submodule}` : ''}#${hash}`;
    }
    return _formatter;
}
function sanitize(input) {
    return input
        .toLowerCase()
        .replace(/[^a-zA-Z0-9 ]/g, '')
        .replace(/ /g, '-');
}
;
//# sourceMappingURL=data:application/json;base64,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