"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Orchestration = void 0;
const aws_sqs_1 = require("@aws-cdk/aws-sqs");
const aws_stepfunctions_1 = require("@aws-cdk/aws-stepfunctions");
const tasks = require("@aws-cdk/aws-stepfunctions-tasks");
const core_1 = require("@aws-cdk/core");
const catalog_builder_1 = require("../catalog-builder");
const language_1 = require("../shared/language");
const transliterator_1 = require("../transliterator");
const redrive_state_machine_1 = require("./redrive-state-machine");
const reprocess_all_1 = require("./reprocess-all");
const SUPPORTED_LANGUAGES = [language_1.DocumentationLanguage.PYTHON, language_1.DocumentationLanguage.TYPESCRIPT];
/**
 * Orchestrates the backend processing tasks using a StepFunctions State Machine.
 */
class Orchestration extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const catalogBuilder = new catalog_builder_1.CatalogBuilder(this, 'CatalogBuilder', props).function;
        const addToCatalog = (lang) => new tasks.LambdaInvoke(this, `Add to catalog.json (${lang})`, {
            lambdaFunction: catalogBuilder,
            resultPath: '$.catalogBuilderOutput',
            resultSelector: {
                'ETag.$': '$.Payload.ETag',
                'VersionId.$': '$.Payload.VersionId',
            },
        })
            // This has a concurrency of 1, so we want to aggressively retry being throttled here.
            .addRetry({ errors: ['Lambda.TooManyRequestsException'], interval: core_1.Duration.seconds(30), maxAttempts: 5 });
        this.deadLetterQueue = new aws_sqs_1.Queue(this, 'DLQ', {
            encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            retentionPeriod: core_1.Duration.days(14),
            visibilityTimeout: core_1.Duration.minutes(15),
        });
        const sendToDeadLetterQueue = new tasks.SqsSendMessage(this, 'Send to Dead Letter Queue', {
            messageBody: aws_stepfunctions_1.TaskInput.fromJsonPathAt('$'),
            queue: this.deadLetterQueue,
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD,
        });
        const definition = new aws_stepfunctions_1.Parallel(this, 'DocGen')
            .branch(...SUPPORTED_LANGUAGES.map((language) => {
            const task = new tasks.LambdaInvoke(this, `Generate ${language} docs`, {
                lambdaFunction: new transliterator_1.Transliterator(this, `DocGen-${language}`, { ...props, language }).function,
                resultPath: '$.docGenOutput',
                resultSelector: { [`${language}.$`]: '$.Payload' },
            }).addRetry({ interval: core_1.Duration.seconds(30) });
            return task.next(addToCatalog(language));
        }))
            .addCatch(sendToDeadLetterQueue.next(new aws_stepfunctions_1.Fail(this, 'Fail', {
            error: 'Failed',
            cause: 'Input was submitted to dead letter queue',
        })), { resultPath: '$._error' });
        this.stateMachine = new aws_stepfunctions_1.StateMachine(this, 'Resource', {
            definition,
            stateMachineType: aws_stepfunctions_1.StateMachineType.STANDARD,
            timeout: core_1.Duration.hours(1),
        });
        // This function is intended to be manually triggered by an operrator to
        // attempt redriving messages from the DLQ.
        this.redriveFunction = new redrive_state_machine_1.RedriveStateMachine(this, 'Redrive', {
            description: '[ConstructHub/Redrive] Manually redrives all messages from the backend dead letter queue',
            environment: {
                STATE_MACHINE_ARN: this.stateMachine.stateMachineArn,
                QUEUE_URL: this.deadLetterQueue.queueUrl,
            },
            memorySize: 1024,
            timeout: core_1.Duration.minutes(15),
        });
        this.stateMachine.grantStartExecution(this.redriveFunction);
        this.deadLetterQueue.grantConsumeMessages(this.redriveFunction);
        // This function is intended to be manually triggered by an operator to
        // reprocess all package versions currently in store through the back-end.
        const reprocessAll = new reprocess_all_1.ReprocessAll(this, 'ReprocessAll', {
            description: '[ConstructHub/ReprocessAll] Reprocess all package versions through the backend',
            environment: {
                BUCKET_NAME: props.bucket.bucketName,
                STATE_MACHINE_ARN: this.stateMachine.stateMachineArn,
            },
            memorySize: 1024,
            timeout: core_1.Duration.minutes(15),
        });
        props.bucket.grantRead(reprocessAll);
        this.stateMachine.grantStartExecution(reprocessAll);
    }
}
exports.Orchestration = Orchestration;
//# sourceMappingURL=data:application/json;base64,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