"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Inventory = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const core_1 = require("@aws-cdk/core");
const canary_1 = require("./canary");
const constants_1 = require("./constants");
/**
 * Periodically computes an inventory of all indexed packages into the storage
 * bucket, and produces metrics with an overview of the index' state.
 */
class Inventory extends core_1.Construct {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        const rate = (_a = props.scheduleRate) !== null && _a !== void 0 ? _a : core_1.Duration.minutes(5);
        this.canary = new canary_1.Canary(this, 'Resource', {
            description: '[ConstructHub/Inventory] A canary that periodically inspects the list of indexed packages',
            environment: { BUCKET_NAME: props.bucket.bucketName },
            logRetention: props.logRetention,
            memorySize: 10240,
            timeout: rate,
        });
        const grant = props.bucket.grantRead(this.canary);
        new aws_events_1.Rule(this, 'ScheduleRule', {
            schedule: aws_events_1.Schedule.rate(rate),
            targets: [new aws_events_targets_1.LambdaFunction(this.canary)],
        }).node.addDependency(grant);
        props.monitoring.watchful.watchLambdaFunction('Inventory Canary', this.canary);
    }
    metricMissingPackageMetadataCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "MissingPackageMetadataCount" /* MISSING_METADATA_COUNT */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    metricMissingAssemblyCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "MissingAssemblyCount" /* MISSING_ASSEMBLY_COUNT */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    metricPackageCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "PackageCount" /* PACKAGE_COUNT */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    metricPackageMajorCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "PackageMajorVersionCount" /* PACKAGE_MAJOR_COUNT */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    metricPackageVersionCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "PackageVersionCount" /* PACKAGE_VERSION_COUNT */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    metricMissingPythonDocsCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "MissingPythonDocsCount" /* MISSING_PYTHON_DOCS_COUNT */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    metricMissingTypeScriptDocsCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "MissingTypeScriptDocsCount" /* MISSING_TYPESCRIPT_DOCS_COUNT */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    metricMissingPackageTarballCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "MissingPackageTarballCount" /* MISSING_TARBALL_COUNT */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    metricSubmoduleCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "SubmoduleCount" /* SUBMODULE_COUNT */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    metricUnknownObjectCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "UnknownObjectCount" /* UNKNOWN_OBJECT_COUNT */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
}
exports.Inventory = Inventory;
//# sourceMappingURL=data:application/json;base64,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