"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const Environments_1 = require("aws-embedded-metrics/lib/environment/Environments");
const semver_1 = require("semver");
const aws = require("../shared/aws.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const language_1 = require("../shared/language");
const constants_1 = require("./constants");
aws_embedded_metrics_1.Configuration.environmentOverride = Environments_1.default.Lambda;
aws_embedded_metrics_1.Configuration.namespace = constants_1.METRICS_NAMESPACE;
async function handler(event, _context) {
    var _a;
    console.log('Event:', JSON.stringify(event, null, 2));
    const indexedPackages = new Map();
    const packageNames = new Set();
    const packageMajorVersions = new Set();
    const submoduleRegexes = {
        tsDocsPresent: submoduleKeyRegexp(language_1.DocumentationLanguage.TYPESCRIPT),
        pythonDocsPresent: submoduleKeyRegexp(language_1.DocumentationLanguage.PYTHON),
    };
    const bucket = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    for await (const key of relevantObjectKeys(bucket)) {
        const [, name, version] = constants.STORAGE_KEY_FORMAT_REGEX.exec(key);
        packageNames.add(name);
        packageMajorVersions.add(`${name}@${new semver_1.SemVer(version).major}`);
        const fullName = `${name}@${version}`;
        if (!indexedPackages.has(fullName)) {
            indexedPackages.set(fullName, {});
        }
        const status = indexedPackages.get(fullName);
        if (key.endsWith(constants.METADATA_KEY_SUFFIX)) {
            status.metadataPresent = true;
        }
        else if (key.endsWith(constants.PACKAGE_KEY_SUFFIX)) {
            status.tarballPresent = true;
        }
        else if (key.endsWith(constants.ASSEMBLY_KEY_SUFFIX)) {
            status.assemblyPresent = true;
        }
        else if (key.endsWith(constants.DOCS_KEY_SUFFIX_PYTHON) ||
            key.endsWith(constants.DOCS_KEY_SUFFIX_PYTHON + constants.NOT_SUPPORTED_SUFFIX)) {
            status.pythonDocsPresent = true;
        }
        else if (key.endsWith(constants.DOCS_KEY_SUFFIX_TYPESCRIPT) ||
            key.endsWith(constants.DOCS_KEY_SUFFIX_TYPESCRIPT + constants.NOT_SUPPORTED_SUFFIX)) {
            status.tsDocsPresent = true;
        }
        else {
            // If this is a submodule-doc key, add the relevant nested status entry.
            const matching = Object.entries(submoduleRegexes)
                .map(([statusKey, regexp]) => {
                const match = regexp.exec(key);
                if (match == null) {
                    return undefined;
                }
                return [statusKey, match[1]];
            })
                .find((item) => item != null);
            if (matching) {
                const [statusKey, submoduleName] = matching;
                const submoduleFqn = `${fullName}.${submoduleName}`;
                if (status.submodules == null) {
                    status.submodules = {};
                }
                if (status.submodules[submoduleFqn] == null) {
                    status.submodules[submoduleFqn] = {};
                }
                status.submodules[submoduleFqn][statusKey] = true;
            }
            else {
                status.unknownObjects = (_a = status.unknownObjects) !== null && _a !== void 0 ? _a : [];
                status.unknownObjects.push(key);
            }
        }
    }
    await aws_embedded_metrics_1.metricScope((metrics) => () => {
        var _a, _b, _c;
        // Clear out default dimensions as we don't need those. See https://github.com/awslabs/aws-embedded-metrics-node/issues/73.
        metrics.setDimensions();
        const missingMetadata = new Array();
        const missingAssembly = new Array();
        const missingPythonDocs = new Array();
        const missingTsDocs = new Array();
        const missingTarball = new Array();
        const unknownObjects = new Array();
        const submodules = new Array();
        for (const [name, status] of indexedPackages.entries()) {
            if (!status.metadataPresent) {
                missingMetadata.push(name);
            }
            if (!status.assemblyPresent) {
                missingAssembly.push(name);
            }
            if (!status.pythonDocsPresent) {
                missingPythonDocs.push(name);
            }
            if (!status.tsDocsPresent) {
                missingTsDocs.push(name);
            }
            if (!status.tarballPresent) {
                missingTarball.push(name);
            }
            if ((_b = (_a = status.unknownObjects) === null || _a === void 0 ? void 0 : _a.length) !== null && _b !== void 0 ? _b : 0 > 0) {
                unknownObjects.push(...status.unknownObjects);
            }
            for (const [submodule, subStatus] of Object.entries((_c = status.submodules) !== null && _c !== void 0 ? _c : {})) {
                submodules.push(submodule);
                if (!subStatus.tsDocsPresent) {
                    missingTsDocs.push(submodule);
                }
                if (!subStatus.pythonDocsPresent) {
                    missingPythonDocs.push(submodule);
                }
            }
        }
        metrics.setProperty('detail', { missingMetadata, missingAssembly, missingPythonDocs, missingTsDocs, missingTarball, unknownObjects });
        metrics.putMetric("MissingPackageMetadataCount" /* MISSING_METADATA_COUNT */, missingMetadata.length, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("MissingAssemblyCount" /* MISSING_ASSEMBLY_COUNT */, missingAssembly.length, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("MissingPythonDocsCount" /* MISSING_PYTHON_DOCS_COUNT */, missingPythonDocs.length, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("MissingTypeScriptDocsCount" /* MISSING_TYPESCRIPT_DOCS_COUNT */, missingTsDocs.length, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("MissingPackageTarballCount" /* MISSING_TARBALL_COUNT */, missingTarball.length, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("PackageCount" /* PACKAGE_COUNT */, packageNames.size, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("PackageMajorVersionCount" /* PACKAGE_MAJOR_COUNT */, packageMajorVersions.size, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("PackageVersionCount" /* PACKAGE_VERSION_COUNT */, indexedPackages.size, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("SubmoduleCount" /* SUBMODULE_COUNT */, submodules.length, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("UnknownObjectCount" /* UNKNOWN_OBJECT_COUNT */, unknownObjects.length, aws_embedded_metrics_1.Unit.Count);
    })();
}
exports.handler = handler;
async function* relevantObjectKeys(bucket) {
    var _a;
    const request = {
        Bucket: bucket,
        Prefix: constants.STORAGE_KEY_PREFIX,
    };
    do {
        const response = await aws.s3().listObjectsV2(request).promise();
        for (const { Key } of (_a = response.Contents) !== null && _a !== void 0 ? _a : []) {
            if (Key == null) {
                continue;
            }
            yield Key;
        }
        request.ContinuationToken = response.NextContinuationToken;
    } while (request.ContinuationToken != null);
}
/**
 * This function obtains a regular expression with a single capture group that
 * allows determining the submodule name from a submodule documentation key.
 */
function submoduleKeyRegexp(language) {
    // We use a placeholder to be able to insert the capture group once we have
    // fully quoted the key prefix for Regex safety.
    const placeholder = '<SUBMODULENAME>';
    // We obtain the standard key prefix.
    const keyPrefix = constants.docsKeySuffix(language, placeholder);
    // Make it regex-safe by quoting all "special meaning" characters.
    const regexpQuoted = keyPrefix.replace(/([+*.()?$[\]])/g, '\\$1');
    // Finally, assemble the regular expression with the capture group.
    return new RegExp(`.*${regexpQuoted.replace(placeholder, '(.+)')}$`);
}
//# sourceMappingURL=data:application/json;base64,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