"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const crypto_1 = require("crypto");
const path_1 = require("path");
const url_1 = require("url");
const zlib_1 = require("zlib");
const spec_1 = require("@jsii/spec");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const Environments_1 = require("aws-embedded-metrics/lib/environment/Environments");
const tar_stream_1 = require("tar-stream");
const aws = require("../shared/aws.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const integrity_lambda_shared_1 = require("../shared/integrity.lambda-shared");
const constants_1 = require("./constants");
aws_embedded_metrics_1.Configuration.environmentOverride = Environments_1.default.Lambda;
aws_embedded_metrics_1.Configuration.namespace = constants_1.METRICS_NAMESPACE;
exports.handler = aws_embedded_metrics_1.metricScope((metrics) => async (event, context) => {
    var _a, _b;
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    // Clear out the default dimensions, we won't need them.
    metrics.setDimensions();
    const BUCKET_NAME = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const STATE_MACHINE_ARN = env_lambda_shared_1.requireEnv('STATE_MACHINE_ARN');
    const result = new Array();
    for (const record of (_a = event.Records) !== null && _a !== void 0 ? _a : []) {
        const payload = JSON.parse(record.body);
        const tarballUri = new url_1.URL(payload.tarballUri);
        if (tarballUri.protocol !== 's3:') {
            throw new Error(`Unsupported protocol in URI: ${tarballUri}`);
        }
        const tarball = await aws.s3().getObject({
            // Note: we drop anything after the first `.` in the host, as we only care about the bucket name.
            Bucket: tarballUri.host.split('.')[0],
            // Note: the pathname part is absolute, so we strip the leading `/`.
            Key: tarballUri.pathname.replace(/^\//, ''),
            VersionId: (_b = tarballUri.searchParams.get('versionId')) !== null && _b !== void 0 ? _b : undefined,
        }).promise();
        const integrityCheck = integrity_lambda_shared_1.integrity(payload, Buffer.from(tarball.Body));
        if (payload.integrity !== integrityCheck) {
            throw new Error(`Integrity check failed: ${payload.integrity} !== ${integrityCheck}`);
        }
        const tar = await gunzip(Buffer.from(tarball.Body));
        let dotJsii;
        let licenseText;
        let packageJson;
        try {
            const extracted = await new Promise((ok, ko) => {
                let dotJsiiBuffer;
                let licenseTextBuffer;
                let packageJsonData;
                const extractor = tar_stream_1.extract({ filenameEncoding: 'utf-8' })
                    .once('error', (reason) => {
                    ko(reason);
                })
                    .once('finish', () => {
                    if (dotJsiiBuffer == null) {
                        ko(new Error('No .jsii file found in tarball!'));
                    }
                    else if (packageJsonData == null) {
                        ko(new Error('No package.json file found in tarball!'));
                    }
                    else {
                        ok({ dotJsii: dotJsiiBuffer, licenseText: licenseTextBuffer, packageJson: packageJsonData });
                    }
                })
                    .on('entry', (headers, stream, next) => {
                    const chunks = new Array();
                    if (headers.name === 'package/.jsii') {
                        return stream.on('data', (chunk) => chunks.push(Buffer.from(chunk)))
                            .once('error', ko)
                            .once('end', () => {
                            dotJsiiBuffer = Buffer.concat(chunks);
                            // Skip on next runLoop iteration so we avoid filling the stack.
                            setImmediate(next);
                        })
                            .resume();
                    }
                    else if (headers.name === 'package/package.json') {
                        return stream.on('data', (chunk) => chunks.push(Buffer.from(chunk)))
                            .once('error', ko)
                            .once('end', () => {
                            packageJsonData = Buffer.concat(chunks);
                            // Skip on next runLoop iteration so we avoid filling the stack.
                            setImmediate(next);
                        })
                            .resume();
                    }
                    else if (isLicenseFile(headers.name)) {
                        return stream.on('data', (chunk) => chunks.push(Buffer.from(chunk)))
                            .once('error', ko)
                            .once('end', () => {
                            licenseTextBuffer = Buffer.concat(chunks);
                            // Skip on next runLoop iteration so we avoid filling the stack.
                            setImmediate(next);
                        })
                            .resume();
                    }
                    // Skip on next runLoop iteration so we avoid filling the stack.
                    return setImmediate(next);
                });
                extractor.write(tar, (err) => {
                    if (err != null) {
                        ko(err);
                    }
                    extractor.end();
                });
            });
            dotJsii = extracted.dotJsii;
            licenseText = extracted.licenseText;
            packageJson = extracted.packageJson;
            metrics.putMetric("InvalidTarball" /* INVALID_TARBALL */, 0, aws_embedded_metrics_1.Unit.Count);
        }
        catch (err) {
            console.error(`Invalid tarball content: ${err}`);
            metrics.putMetric("InvalidTarball" /* INVALID_TARBALL */, 1, aws_embedded_metrics_1.Unit.Count);
            return;
        }
        const metadata = { date: payload.time, licenseText: licenseText === null || licenseText === void 0 ? void 0 : licenseText.toString('utf-8') };
        let packageLicense;
        let packageName;
        let packageVersion;
        try {
            const { license, name, version } = spec_1.validateAssembly(JSON.parse(dotJsii.toString('utf-8')));
            packageLicense = license;
            packageName = name;
            packageVersion = version;
            metrics.putMetric("InvalidAssembly" /* INVALID_ASSEMBLY */, 0, aws_embedded_metrics_1.Unit.Count);
        }
        catch (ex) {
            console.error(`Package does not contain a valid assembly -- ignoring: ${ex}`);
            metrics.putMetric("InvalidAssembly" /* INVALID_ASSEMBLY */, 1, aws_embedded_metrics_1.Unit.Count);
            return;
        }
        // Ensure the `.jsii` name, version & license corresponds to those in `package.json`
        const { name: packageJsonName, version: packageJsonVersion, license: packageJsonLicense } = JSON.parse(packageJson.toString('utf-8'));
        if (packageJsonName !== packageName || packageJsonVersion !== packageVersion || packageJsonLicense !== packageLicense) {
            console.log(`Ignoring package with mismatched name, version, and/or license (${packageJsonName}@${packageJsonVersion} is ${packageJsonLicense} !== ${packageName}@${packageVersion} is ${packageLicense})`);
            metrics.putMetric("MismatchedIdentityRejections" /* MISMATCHED_IDENTITY_REJECTIONS */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        metrics.putMetric("MismatchedIdentityRejections" /* MISMATCHED_IDENTITY_REJECTIONS */, 0, aws_embedded_metrics_1.Unit.Count);
        // Did we identify a license file or not?
        metrics.putMetric("FoundLicenseFile" /* FOUND_LICENSE_FILE */, licenseText != null ? 1 : 0, aws_embedded_metrics_1.Unit.Count);
        const assemblyKey = `${constants.STORAGE_KEY_PREFIX}${packageName}/v${packageVersion}${constants.ASSEMBLY_KEY_SUFFIX}`;
        console.log(`Writing assembly at ${assemblyKey}`);
        const packageKey = `${constants.STORAGE_KEY_PREFIX}${packageName}/v${packageVersion}${constants.PACKAGE_KEY_SUFFIX}`;
        console.log(`Writing package at  ${packageKey}`);
        const metadataKey = `${constants.STORAGE_KEY_PREFIX}${packageName}/v${packageVersion}${constants.METADATA_KEY_SUFFIX}`;
        console.log(`Writing metadata at  ${metadataKey}`);
        // we upload the metadata file first because the catalog builder depends on
        // it and is triggered by the assembly file upload.
        console.log(`${packageName}@${packageVersion} | Uploading package and metadata files`);
        const [pkg, storedMetadata] = await Promise.all([
            aws.s3().putObject({
                Bucket: BUCKET_NAME,
                Key: packageKey,
                Body: tarball.Body,
                CacheControl: 'public',
                ContentType: 'application/octet-stream',
                Metadata: {
                    'Lambda-Log-Group': context.logGroupName,
                    'Lambda-Log-Stream': context.logStreamName,
                    'Lambda-Run-Id': context.awsRequestId,
                },
            }).promise(),
            aws.s3().putObject({
                Bucket: BUCKET_NAME,
                Key: metadataKey,
                Body: JSON.stringify(metadata),
                CacheControl: 'public',
                ContentType: 'application/json',
                Metadata: {
                    'Lambda-Log-Group': context.logGroupName,
                    'Lambda-Log-Stream': context.logStreamName,
                    'Lambda-Run-Id': context.awsRequestId,
                },
            }).promise(),
        ]);
        // now we can upload the assembly.
        console.log(`${packageName}@${packageVersion} | Uploading assembly file`);
        const assembly = await aws.s3().putObject({
            Bucket: BUCKET_NAME,
            Key: assemblyKey,
            Body: dotJsii,
            CacheControl: 'public',
            ContentType: 'application/json',
            Metadata: {
                'Lambda-Log-Group': context.logGroupName,
                'Lambda-Log-Stream': context.logStreamName,
                'Lambda-Run-Id': context.awsRequestId,
            },
        }).promise();
        const created = {
            bucket: BUCKET_NAME,
            assembly: {
                key: assemblyKey,
                versionId: assembly.VersionId,
            },
            package: {
                key: packageKey,
                versionId: pkg.VersionId,
            },
            metadata: {
                key: metadataKey,
                versionId: storedMetadata.VersionId,
            },
        };
        console.log(`Created objects: ${JSON.stringify(created, null, 2)}`);
        const sfn = await aws.stepFunctions().startExecution({
            input: JSON.stringify(created),
            name: sfnExecutionNameFromParts(packageName, `v${packageVersion}`, context.awsRequestId),
            stateMachineArn: STATE_MACHINE_ARN,
        }).promise();
        console.log(`Started StateMachine execution: ${sfn.executionArn}`);
        result.push(sfn.executionArn);
    }
    return result;
});
function gunzip(data) {
    const chunks = new Array();
    return new Promise((ok, ko) => zlib_1.createGunzip()
        .once('error', ko)
        .on('data', (chunk) => chunks.push(Buffer.from(chunk)))
        .once('end', () => ok(Buffer.concat(chunks)))
        .end(data));
}
/**
 * Checks whether the provided file name corresponds to a license file or not.
 *
 * @param fileName the file name to be checked.
 *
 * @returns `true` IIF the file is named LICENSE and has the .MD or .TXT
 *          extension, or no extension at all. The test is case-insensitive.
 */
function isLicenseFile(fileName) {
    const ext = path_1.extname(fileName);
    const possibleExtensions = new Set(['', '.md', '.txt']);
    return possibleExtensions.has(ext.toLowerCase())
        && path_1.basename(fileName, ext).toUpperCase() === 'LICENSE';
}
/**
 * Creates a StepFunction execution request name based on the provided parts.
 * The result is guaranteed to be 80 characters or less and to contain only
 * characters that are valid for a StepFunction execution request name for which
 * CloudWatch Logging can be enabled. The resulting name is very likely to
 * be unique for a given input.
 */
function sfnExecutionNameFromParts(first, ...rest) {
    const parts = [first, ...rest];
    const name = parts
        .map((part) => part.replace(/[^a-z0-9_-]+/ig, '_'))
        .join('_')
        .replace(/^_/g, '')
        .replace(/_{2,}/g, '_');
    if (name.length <= 80) {
        return name;
    }
    const suffix = crypto_1.createHash('sha256')
        // The hash is computed based on input arguments, to maximize unicity
        .update(parts.join('_'))
        .digest('hex')
        .substring(0, 6);
    return `${name.substring(0, 80 - suffix.length - 1)}_${suffix}`;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5nZXN0aW9uLmxhbWJkYS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9iYWNrZW5kL2luZ2VzdGlvbi9pbmdlc3Rpb24ubGFtYmRhLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLG1DQUFvQztBQUNwQywrQkFBeUM7QUFDekMsNkJBQTBCO0FBQzFCLCtCQUFvQztBQUVwQyxxQ0FBOEM7QUFDOUMsK0RBQXdFO0FBQ3hFLG9GQUE2RTtBQUU3RSwyQ0FBcUM7QUFFckMsbURBQW1EO0FBQ25ELGlEQUFpRDtBQUNqRCxtRUFBeUQ7QUFFekQsK0VBQThEO0FBQzlELDJDQUE0RDtBQUU1RCxvQ0FBYSxDQUFDLG1CQUFtQixHQUFHLHNCQUFZLENBQUMsTUFBTSxDQUFDO0FBQ3hELG9DQUFhLENBQUMsU0FBUyxHQUFHLDZCQUFpQixDQUFDO0FBRS9CLFFBQUEsT0FBTyxHQUFHLGtDQUFXLENBQUMsQ0FBQyxPQUFPLEVBQUUsRUFBRSxDQUFDLEtBQUssRUFBRSxLQUFlLEVBQUUsT0FBZ0IsRUFBRSxFQUFFOztJQUMxRixPQUFPLENBQUMsR0FBRyxDQUFDLFVBQVUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQztJQUV4RCx3REFBd0Q7SUFDeEQsT0FBTyxDQUFDLGFBQWEsRUFBRSxDQUFDO0lBRXhCLE1BQU0sV0FBVyxHQUFHLDhCQUFVLENBQUMsYUFBYSxDQUFDLENBQUM7SUFDOUMsTUFBTSxpQkFBaUIsR0FBRyw4QkFBVSxDQUFDLG1CQUFtQixDQUFDLENBQUM7SUFFMUQsTUFBTSxNQUFNLEdBQUcsSUFBSSxLQUFLLEVBQVUsQ0FBQztJQUVuQyxLQUFLLE1BQU0sTUFBTSxVQUFJLEtBQUssQ0FBQyxPQUFPLG1DQUFJLEVBQUUsRUFBRTtRQUN4QyxNQUFNLE9BQU8sR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQW1CLENBQUM7UUFFMUQsTUFBTSxVQUFVLEdBQUcsSUFBSSxTQUFHLENBQUMsT0FBTyxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBQy9DLElBQUksVUFBVSxDQUFDLFFBQVEsS0FBSyxLQUFLLEVBQUU7WUFDakMsTUFBTSxJQUFJLEtBQUssQ0FBQyxnQ0FBZ0MsVUFBVSxFQUFFLENBQUMsQ0FBQztTQUMvRDtRQUNELE1BQU0sT0FBTyxHQUFHLE1BQU0sR0FBRyxDQUFDLEVBQUUsRUFBRSxDQUFDLFNBQVMsQ0FBQztZQUN2QyxpR0FBaUc7WUFDakcsTUFBTSxFQUFFLFVBQVUsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUNyQyxvRUFBb0U7WUFDcEUsR0FBRyxFQUFFLFVBQVUsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUM7WUFDM0MsU0FBUyxRQUFFLFVBQVUsQ0FBQyxZQUFZLENBQUMsR0FBRyxDQUFDLFdBQVcsQ0FBQyxtQ0FBSSxTQUFTO1NBQ2pFLENBQUMsQ0FBQyxPQUFPLEVBQUUsQ0FBQztRQUViLE1BQU0sY0FBYyxHQUFHLG1DQUFTLENBQUMsT0FBTyxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLElBQUssQ0FBQyxDQUFDLENBQUM7UUFDdEUsSUFBSSxPQUFPLENBQUMsU0FBUyxLQUFLLGNBQWMsRUFBRTtZQUN4QyxNQUFNLElBQUksS0FBSyxDQUFDLDJCQUEyQixPQUFPLENBQUMsU0FBUyxRQUFRLGNBQWMsRUFBRSxDQUFDLENBQUM7U0FDdkY7UUFFRCxNQUFNLEdBQUcsR0FBRyxNQUFNLE1BQU0sQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxJQUFLLENBQUMsQ0FBQyxDQUFDO1FBQ3JELElBQUksT0FBZSxDQUFDO1FBQ3BCLElBQUksV0FBK0IsQ0FBQztRQUNwQyxJQUFJLFdBQW1CLENBQUM7UUFDeEIsSUFBSTtZQUNGLE1BQU0sU0FBUyxHQUFHLE1BQU0sSUFBSSxPQUFPLENBQWlFLENBQUMsRUFBRSxFQUFFLEVBQUUsRUFBRSxFQUFFO2dCQUM3RyxJQUFJLGFBQWlDLENBQUM7Z0JBQ3RDLElBQUksaUJBQXFDLENBQUM7Z0JBQzFDLElBQUksZUFBbUMsQ0FBQztnQkFDeEMsTUFBTSxTQUFTLEdBQUcsb0JBQU8sQ0FBQyxFQUFFLGdCQUFnQixFQUFFLE9BQU8sRUFBRSxDQUFDO3FCQUNyRCxJQUFJLENBQUMsT0FBTyxFQUFFLENBQUMsTUFBTSxFQUFFLEVBQUU7b0JBQ3hCLEVBQUUsQ0FBQyxNQUFNLENBQUMsQ0FBQztnQkFDYixDQUFDLENBQUM7cUJBQ0QsSUFBSSxDQUFDLFFBQVEsRUFBRSxHQUFHLEVBQUU7b0JBQ25CLElBQUksYUFBYSxJQUFJLElBQUksRUFBRTt3QkFDekIsRUFBRSxDQUFDLElBQUksS0FBSyxDQUFDLGlDQUFpQyxDQUFDLENBQUMsQ0FBQztxQkFDbEQ7eUJBQU0sSUFBSSxlQUFlLElBQUksSUFBSSxFQUFFO3dCQUNsQyxFQUFFLENBQUMsSUFBSSxLQUFLLENBQUMsd0NBQXdDLENBQUMsQ0FBQyxDQUFDO3FCQUN6RDt5QkFBTTt3QkFDTCxFQUFFLENBQUMsRUFBRSxPQUFPLEVBQUUsYUFBYSxFQUFFLFdBQVcsRUFBRSxpQkFBaUIsRUFBRSxXQUFXLEVBQUUsZUFBZSxFQUFFLENBQUMsQ0FBQztxQkFDOUY7Z0JBQ0gsQ0FBQyxDQUFDO3FCQUNELEVBQUUsQ0FBQyxPQUFPLEVBQUUsQ0FBQyxPQUFPLEVBQUUsTUFBTSxFQUFFLElBQUksRUFBRSxFQUFFO29CQUNyQyxNQUFNLE1BQU0sR0FBRyxJQUFJLEtBQUssRUFBVSxDQUFDO29CQUNuQyxJQUFJLE9BQU8sQ0FBQyxJQUFJLEtBQUssZUFBZSxFQUFFO3dCQUNwQyxPQUFPLE1BQU0sQ0FBQyxFQUFFLENBQUMsTUFBTSxFQUFFLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQzs2QkFDakUsSUFBSSxDQUFDLE9BQU8sRUFBRSxFQUFFLENBQUM7NkJBQ2pCLElBQUksQ0FBQyxLQUFLLEVBQUUsR0FBRyxFQUFFOzRCQUNoQixhQUFhLEdBQUcsTUFBTSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQzs0QkFDdEMsZ0VBQWdFOzRCQUNoRSxZQUFZLENBQUMsSUFBSSxDQUFDLENBQUM7d0JBQ3JCLENBQUMsQ0FBQzs2QkFDRCxNQUFNLEVBQUUsQ0FBQztxQkFDYjt5QkFBTSxJQUFJLE9BQU8sQ0FBQyxJQUFJLEtBQUssc0JBQXNCLEVBQUU7d0JBQ2xELE9BQU8sTUFBTSxDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDOzZCQUNqRSxJQUFJLENBQUMsT0FBTyxFQUFFLEVBQUUsQ0FBQzs2QkFDakIsSUFBSSxDQUFDLEtBQUssRUFBRSxHQUFHLEVBQUU7NEJBQ2hCLGVBQWUsR0FBRyxNQUFNLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxDQUFDOzRCQUN4QyxnRUFBZ0U7NEJBQ2hFLFlBQVksQ0FBQyxJQUFJLENBQUMsQ0FBQzt3QkFDckIsQ0FBQyxDQUFDOzZCQUNELE1BQU0sRUFBRSxDQUFDO3FCQUNiO3lCQUFNLElBQUksYUFBYSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsRUFBRTt3QkFDdEMsT0FBTyxNQUFNLENBQUMsRUFBRSxDQUFDLE1BQU0sRUFBRSxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7NkJBQ2pFLElBQUksQ0FBQyxPQUFPLEVBQUUsRUFBRSxDQUFDOzZCQUNqQixJQUFJLENBQUMsS0FBSyxFQUFFLEdBQUcsRUFBRTs0QkFDaEIsaUJBQWlCLEdBQUcsTUFBTSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQzs0QkFDMUMsZ0VBQWdFOzRCQUNoRSxZQUFZLENBQUMsSUFBSSxDQUFDLENBQUM7d0JBQ3JCLENBQUMsQ0FBQzs2QkFDRCxNQUFNLEVBQUUsQ0FBQztxQkFDYjtvQkFDRCxnRUFBZ0U7b0JBQ2hFLE9BQU8sWUFBWSxDQUFDLElBQUksQ0FBQyxDQUFDO2dCQUM1QixDQUFDLENBQUMsQ0FBQztnQkFDTCxTQUFTLENBQUMsS0FBSyxDQUFDLEdBQUcsRUFBRSxDQUFDLEdBQUcsRUFBRSxFQUFFO29CQUMzQixJQUFJLEdBQUcsSUFBSSxJQUFJLEVBQUU7d0JBQ2YsRUFBRSxDQUFDLEdBQUcsQ0FBQyxDQUFDO3FCQUNUO29CQUNELFNBQVMsQ0FBQyxHQUFHLEVBQUUsQ0FBQztnQkFDbEIsQ0FBQyxDQUFDLENBQUM7WUFDTCxDQUFDLENBQUMsQ0FBQztZQUNILE9BQU8sR0FBRyxTQUFTLENBQUMsT0FBTyxDQUFDO1lBQzVCLFdBQVcsR0FBRyxTQUFTLENBQUMsV0FBVyxDQUFDO1lBQ3BDLFdBQVcsR0FBRyxTQUFTLENBQUMsV0FBVyxDQUFDO1lBQ3BDLE9BQU8sQ0FBQyxTQUFTLHlDQUE2QixDQUFDLEVBQUUsMkJBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztTQUM5RDtRQUFDLE9BQU8sR0FBRyxFQUFFO1lBQ1osT0FBTyxDQUFDLEtBQUssQ0FBQyw0QkFBNEIsR0FBRyxFQUFFLENBQUMsQ0FBQztZQUNqRCxPQUFPLENBQUMsU0FBUyx5Q0FBNkIsQ0FBQyxFQUFFLDJCQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7WUFDN0QsT0FBTztTQUNSO1FBQ0QsTUFBTSxRQUFRLEdBQUcsRUFBRSxJQUFJLEVBQUUsT0FBTyxDQUFDLElBQUksRUFBRSxXQUFXLEVBQUUsV0FBVyxhQUFYLFdBQVcsdUJBQVgsV0FBVyxDQUFFLFFBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDO1FBRXJGLElBQUksY0FBc0IsQ0FBQztRQUMzQixJQUFJLFdBQW1CLENBQUM7UUFDeEIsSUFBSSxjQUFzQixDQUFDO1FBQzNCLElBQUk7WUFDRixNQUFNLEVBQUUsT0FBTyxFQUFFLElBQUksRUFBRSxPQUFPLEVBQUUsR0FBRyx1QkFBZ0IsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQzNGLGNBQWMsR0FBRyxPQUFPLENBQUM7WUFDekIsV0FBVyxHQUFHLElBQUksQ0FBQztZQUNuQixjQUFjLEdBQUcsT0FBTyxDQUFDO1lBQ3pCLE9BQU8sQ0FBQyxTQUFTLDJDQUE4QixDQUFDLEVBQUUsMkJBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztTQUMvRDtRQUFDLE9BQU8sRUFBRSxFQUFFO1lBQ1gsT0FBTyxDQUFDLEtBQUssQ0FBQywwREFBMEQsRUFBRSxFQUFFLENBQUMsQ0FBQztZQUM5RSxPQUFPLENBQUMsU0FBUywyQ0FBOEIsQ0FBQyxFQUFFLDJCQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7WUFDOUQsT0FBTztTQUNSO1FBRUQsb0ZBQW9GO1FBQ3BGLE1BQU0sRUFBRSxJQUFJLEVBQUUsZUFBZSxFQUFFLE9BQU8sRUFBRSxrQkFBa0IsRUFBRSxPQUFPLEVBQUUsa0JBQWtCLEVBQUUsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLFdBQVcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztRQUN0SSxJQUFJLGVBQWUsS0FBSyxXQUFXLElBQUksa0JBQWtCLEtBQUssY0FBYyxJQUFJLGtCQUFrQixLQUFLLGNBQWMsRUFBRTtZQUNySCxPQUFPLENBQUMsR0FBRyxDQUFDLG1FQUFtRSxlQUFlLElBQUksa0JBQWtCLE9BQU8sa0JBQWtCLFFBQVEsV0FBVyxJQUFJLGNBQWMsT0FBTyxjQUFjLEdBQUcsQ0FBQyxDQUFDO1lBQzVNLE9BQU8sQ0FBQyxTQUFTLHNFQUE0QyxDQUFDLEVBQUUsMkJBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztZQUM1RSxTQUFTO1NBQ1Y7UUFDRCxPQUFPLENBQUMsU0FBUyxzRUFBNEMsQ0FBQyxFQUFFLDJCQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7UUFFNUUseUNBQXlDO1FBQ3pDLE9BQU8sQ0FBQyxTQUFTLDhDQUFnQyxXQUFXLElBQUksSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSwyQkFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBRTFGLE1BQU0sV0FBVyxHQUFHLEdBQUcsU0FBUyxDQUFDLGtCQUFrQixHQUFHLFdBQVcsS0FBSyxjQUFjLEdBQUcsU0FBUyxDQUFDLG1CQUFtQixFQUFFLENBQUM7UUFDdkgsT0FBTyxDQUFDLEdBQUcsQ0FBQyx1QkFBdUIsV0FBVyxFQUFFLENBQUMsQ0FBQztRQUNsRCxNQUFNLFVBQVUsR0FBRyxHQUFHLFNBQVMsQ0FBQyxrQkFBa0IsR0FBRyxXQUFXLEtBQUssY0FBYyxHQUFHLFNBQVMsQ0FBQyxrQkFBa0IsRUFBRSxDQUFDO1FBQ3JILE9BQU8sQ0FBQyxHQUFHLENBQUMsdUJBQXVCLFVBQVUsRUFBRSxDQUFDLENBQUM7UUFDakQsTUFBTSxXQUFXLEdBQUcsR0FBRyxTQUFTLENBQUMsa0JBQWtCLEdBQUcsV0FBVyxLQUFLLGNBQWMsR0FBRyxTQUFTLENBQUMsbUJBQW1CLEVBQUUsQ0FBQztRQUN2SCxPQUFPLENBQUMsR0FBRyxDQUFDLHdCQUF3QixXQUFXLEVBQUUsQ0FBQyxDQUFDO1FBRW5ELDJFQUEyRTtRQUMzRSxtREFBbUQ7UUFDbkQsT0FBTyxDQUFDLEdBQUcsQ0FBQyxHQUFHLFdBQVcsSUFBSSxjQUFjLHlDQUF5QyxDQUFDLENBQUM7UUFDdkYsTUFBTSxDQUFDLEdBQUcsRUFBRSxjQUFjLENBQUMsR0FBRyxNQUFNLE9BQU8sQ0FBQyxHQUFHLENBQUM7WUFDOUMsR0FBRyxDQUFDLEVBQUUsRUFBRSxDQUFDLFNBQVMsQ0FBQztnQkFDakIsTUFBTSxFQUFFLFdBQVc7Z0JBQ25CLEdBQUcsRUFBRSxVQUFVO2dCQUNmLElBQUksRUFBRSxPQUFPLENBQUMsSUFBSTtnQkFDbEIsWUFBWSxFQUFFLFFBQVE7Z0JBQ3RCLFdBQVcsRUFBRSwwQkFBMEI7Z0JBQ3ZDLFFBQVEsRUFBRTtvQkFDUixrQkFBa0IsRUFBRSxPQUFPLENBQUMsWUFBWTtvQkFDeEMsbUJBQW1CLEVBQUUsT0FBTyxDQUFDLGFBQWE7b0JBQzFDLGVBQWUsRUFBRSxPQUFPLENBQUMsWUFBWTtpQkFDdEM7YUFDRixDQUFDLENBQUMsT0FBTyxFQUFFO1lBQ1osR0FBRyxDQUFDLEVBQUUsRUFBRSxDQUFDLFNBQVMsQ0FBQztnQkFDakIsTUFBTSxFQUFFLFdBQVc7Z0JBQ25CLEdBQUcsRUFBRSxXQUFXO2dCQUNoQixJQUFJLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxRQUFRLENBQUM7Z0JBQzlCLFlBQVksRUFBRSxRQUFRO2dCQUN0QixXQUFXLEVBQUUsa0JBQWtCO2dCQUMvQixRQUFRLEVBQUU7b0JBQ1Isa0JBQWtCLEVBQUUsT0FBTyxDQUFDLFlBQVk7b0JBQ3hDLG1CQUFtQixFQUFFLE9BQU8sQ0FBQyxhQUFhO29CQUMxQyxlQUFlLEVBQUUsT0FBTyxDQUFDLFlBQVk7aUJBQ3RDO2FBQ0YsQ0FBQyxDQUFDLE9BQU8sRUFBRTtTQUNiLENBQUMsQ0FBQztRQUVILGtDQUFrQztRQUNsQyxPQUFPLENBQUMsR0FBRyxDQUFDLEdBQUcsV0FBVyxJQUFJLGNBQWMsNEJBQTRCLENBQUMsQ0FBQztRQUMxRSxNQUFNLFFBQVEsR0FBRyxNQUFNLEdBQUcsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxTQUFTLENBQUM7WUFDeEMsTUFBTSxFQUFFLFdBQVc7WUFDbkIsR0FBRyxFQUFFLFdBQVc7WUFDaEIsSUFBSSxFQUFFLE9BQU87WUFDYixZQUFZLEVBQUUsUUFBUTtZQUN0QixXQUFXLEVBQUUsa0JBQWtCO1lBQy9CLFFBQVEsRUFBRTtnQkFDUixrQkFBa0IsRUFBRSxPQUFPLENBQUMsWUFBWTtnQkFDeEMsbUJBQW1CLEVBQUUsT0FBTyxDQUFDLGFBQWE7Z0JBQzFDLGVBQWUsRUFBRSxPQUFPLENBQUMsWUFBWTthQUN0QztTQUNGLENBQUMsQ0FBQyxPQUFPLEVBQUUsQ0FBQztRQUViLE1BQU0sT0FBTyxHQUFzQjtZQUNqQyxNQUFNLEVBQUUsV0FBVztZQUNuQixRQUFRLEVBQUU7Z0JBQ1IsR0FBRyxFQUFFLFdBQVc7Z0JBQ2hCLFNBQVMsRUFBRSxRQUFRLENBQUMsU0FBUzthQUM5QjtZQUNELE9BQU8sRUFBRTtnQkFDUCxHQUFHLEVBQUUsVUFBVTtnQkFDZixTQUFTLEVBQUUsR0FBRyxDQUFDLFNBQVM7YUFDekI7WUFDRCxRQUFRLEVBQUU7Z0JBQ1IsR0FBRyxFQUFFLFdBQVc7Z0JBQ2hCLFNBQVMsRUFBRSxjQUFjLENBQUMsU0FBUzthQUNwQztTQUNGLENBQUM7UUFDRixPQUFPLENBQUMsR0FBRyxDQUFDLG9CQUFvQixJQUFJLENBQUMsU0FBUyxDQUFDLE9BQU8sRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDO1FBRXBFLE1BQU0sR0FBRyxHQUFHLE1BQU0sR0FBRyxDQUFDLGFBQWEsRUFBRSxDQUFDLGNBQWMsQ0FBQztZQUNuRCxLQUFLLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUM7WUFDOUIsSUFBSSxFQUFFLHlCQUF5QixDQUFDLFdBQVcsRUFBRSxJQUFJLGNBQWMsRUFBRSxFQUFFLE9BQU8sQ0FBQyxZQUFZLENBQUM7WUFDeEYsZUFBZSxFQUFFLGlCQUFpQjtTQUNuQyxDQUFDLENBQUMsT0FBTyxFQUFFLENBQUM7UUFDYixPQUFPLENBQUMsR0FBRyxDQUFDLG1DQUFtQyxHQUFHLENBQUMsWUFBWSxFQUFFLENBQUMsQ0FBQztRQUNuRSxNQUFNLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxZQUFZLENBQUMsQ0FBQztLQUMvQjtJQUVELE9BQU8sTUFBTSxDQUFDO0FBQ2hCLENBQUMsQ0FBQyxDQUFDO0FBRUgsU0FBUyxNQUFNLENBQUMsSUFBWTtJQUMxQixNQUFNLE1BQU0sR0FBRyxJQUFJLEtBQUssRUFBVSxDQUFDO0lBQ25DLE9BQU8sSUFBSSxPQUFPLENBQVMsQ0FBQyxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUUsQ0FDcEMsbUJBQVksRUFBRTtTQUNYLElBQUksQ0FBQyxPQUFPLEVBQUUsRUFBRSxDQUFDO1NBQ2pCLEVBQUUsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO1NBQ3RELElBQUksQ0FBQyxLQUFLLEVBQUUsR0FBRyxFQUFFLENBQUMsRUFBRSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQztTQUM1QyxHQUFHLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQztBQUNsQixDQUFDO0FBRUQ7Ozs7Ozs7R0FPRztBQUNILFNBQVMsYUFBYSxDQUFDLFFBQWdCO0lBQ3JDLE1BQU0sR0FBRyxHQUFHLGNBQU8sQ0FBQyxRQUFRLENBQUMsQ0FBQztJQUM5QixNQUFNLGtCQUFrQixHQUFHLElBQUksR0FBRyxDQUFDLENBQUMsRUFBRSxFQUFFLEtBQUssRUFBRSxNQUFNLENBQUMsQ0FBQyxDQUFDO0lBQ3hELE9BQU8sa0JBQWtCLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxXQUFXLEVBQUUsQ0FBQztXQUMzQyxlQUFRLENBQUMsUUFBUSxFQUFFLEdBQUcsQ0FBQyxDQUFDLFdBQVcsRUFBRSxLQUFLLFNBQVMsQ0FBQztBQUMzRCxDQUFDO0FBR0Q7Ozs7OztHQU1HO0FBQ0gsU0FBUyx5QkFBeUIsQ0FBQyxLQUFhLEVBQUUsR0FBRyxJQUF1QjtJQUMxRSxNQUFNLEtBQUssR0FBRyxDQUFDLEtBQUssRUFBRSxHQUFHLElBQUksQ0FBQyxDQUFDO0lBQy9CLE1BQU0sSUFBSSxHQUFHLEtBQUs7U0FDZixHQUFHLENBQUMsQ0FBQyxJQUFJLEVBQUUsRUFBRSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsZ0JBQWdCLEVBQUUsR0FBRyxDQUFDLENBQUM7U0FDbEQsSUFBSSxDQUFDLEdBQUcsQ0FBQztTQUNULE9BQU8sQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDO1NBQ2xCLE9BQU8sQ0FBQyxRQUFRLEVBQUUsR0FBRyxDQUFDLENBQUM7SUFDMUIsSUFBSSxJQUFJLENBQUMsTUFBTSxJQUFJLEVBQUUsRUFBRTtRQUNyQixPQUFPLElBQUksQ0FBQztLQUNiO0lBQ0QsTUFBTSxNQUFNLEdBQUcsbUJBQVUsQ0FBQyxRQUFRLENBQUM7UUFDbkMscUVBQXFFO1NBQ2xFLE1BQU0sQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1NBQ3ZCLE1BQU0sQ0FBQyxLQUFLLENBQUM7U0FDYixTQUFTLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDO0lBQ25CLE9BQU8sR0FBRyxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUMsRUFBRSxFQUFFLEdBQUcsTUFBTSxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsSUFBSSxNQUFNLEVBQUUsQ0FBQztBQUNsRSxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgY3JlYXRlSGFzaCB9IGZyb20gJ2NyeXB0byc7XG5pbXBvcnQgeyBiYXNlbmFtZSwgZXh0bmFtZSB9IGZyb20gJ3BhdGgnO1xuaW1wb3J0IHsgVVJMIH0gZnJvbSAndXJsJztcbmltcG9ydCB7IGNyZWF0ZUd1bnppcCB9IGZyb20gJ3psaWInO1xuXG5pbXBvcnQgeyB2YWxpZGF0ZUFzc2VtYmx5IH0gZnJvbSAnQGpzaWkvc3BlYyc7XG5pbXBvcnQgeyBtZXRyaWNTY29wZSwgQ29uZmlndXJhdGlvbiwgVW5pdCB9IGZyb20gJ2F3cy1lbWJlZGRlZC1tZXRyaWNzJztcbmltcG9ydCBFbnZpcm9ubWVudHMgZnJvbSAnYXdzLWVtYmVkZGVkLW1ldHJpY3MvbGliL2Vudmlyb25tZW50L0Vudmlyb25tZW50cyc7XG5pbXBvcnQgdHlwZSB7IENvbnRleHQsIFNRU0V2ZW50IH0gZnJvbSAnYXdzLWxhbWJkYSc7XG5pbXBvcnQgeyBleHRyYWN0IH0gZnJvbSAndGFyLXN0cmVhbSc7XG5pbXBvcnQgdHlwZSB7IFN0YXRlTWFjaGluZUlucHV0IH0gZnJvbSAnLi4vcGF5bG9hZC1zY2hlbWEnO1xuaW1wb3J0ICogYXMgYXdzIGZyb20gJy4uL3NoYXJlZC9hd3MubGFtYmRhLXNoYXJlZCc7XG5pbXBvcnQgKiBhcyBjb25zdGFudHMgZnJvbSAnLi4vc2hhcmVkL2NvbnN0YW50cyc7XG5pbXBvcnQgeyByZXF1aXJlRW52IH0gZnJvbSAnLi4vc2hhcmVkL2Vudi5sYW1iZGEtc2hhcmVkJztcbmltcG9ydCB7IEluZ2VzdGlvbklucHV0IH0gZnJvbSAnLi4vc2hhcmVkL2luZ2VzdGlvbi1pbnB1dC5sYW1iZGEtc2hhcmVkJztcbmltcG9ydCB7IGludGVncml0eSB9IGZyb20gJy4uL3NoYXJlZC9pbnRlZ3JpdHkubGFtYmRhLXNoYXJlZCc7XG5pbXBvcnQgeyBNZXRyaWNOYW1lLCBNRVRSSUNTX05BTUVTUEFDRSB9IGZyb20gJy4vY29uc3RhbnRzJztcblxuQ29uZmlndXJhdGlvbi5lbnZpcm9ubWVudE92ZXJyaWRlID0gRW52aXJvbm1lbnRzLkxhbWJkYTtcbkNvbmZpZ3VyYXRpb24ubmFtZXNwYWNlID0gTUVUUklDU19OQU1FU1BBQ0U7XG5cbmV4cG9ydCBjb25zdCBoYW5kbGVyID0gbWV0cmljU2NvcGUoKG1ldHJpY3MpID0+IGFzeW5jIChldmVudDogU1FTRXZlbnQsIGNvbnRleHQ6IENvbnRleHQpID0+IHtcbiAgY29uc29sZS5sb2coYEV2ZW50OiAke0pTT04uc3RyaW5naWZ5KGV2ZW50LCBudWxsLCAyKX1gKTtcblxuICAvLyBDbGVhciBvdXQgdGhlIGRlZmF1bHQgZGltZW5zaW9ucywgd2Ugd29uJ3QgbmVlZCB0aGVtLlxuICBtZXRyaWNzLnNldERpbWVuc2lvbnMoKTtcblxuICBjb25zdCBCVUNLRVRfTkFNRSA9IHJlcXVpcmVFbnYoJ0JVQ0tFVF9OQU1FJyk7XG4gIGNvbnN0IFNUQVRFX01BQ0hJTkVfQVJOID0gcmVxdWlyZUVudignU1RBVEVfTUFDSElORV9BUk4nKTtcblxuICBjb25zdCByZXN1bHQgPSBuZXcgQXJyYXk8c3RyaW5nPigpO1xuXG4gIGZvciAoY29uc3QgcmVjb3JkIG9mIGV2ZW50LlJlY29yZHMgPz8gW10pIHtcbiAgICBjb25zdCBwYXlsb2FkID0gSlNPTi5wYXJzZShyZWNvcmQuYm9keSkgYXMgSW5nZXN0aW9uSW5wdXQ7XG5cbiAgICBjb25zdCB0YXJiYWxsVXJpID0gbmV3IFVSTChwYXlsb2FkLnRhcmJhbGxVcmkpO1xuICAgIGlmICh0YXJiYWxsVXJpLnByb3RvY29sICE9PSAnczM6Jykge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBVbnN1cHBvcnRlZCBwcm90b2NvbCBpbiBVUkk6ICR7dGFyYmFsbFVyaX1gKTtcbiAgICB9XG4gICAgY29uc3QgdGFyYmFsbCA9IGF3YWl0IGF3cy5zMygpLmdldE9iamVjdCh7XG4gICAgICAvLyBOb3RlOiB3ZSBkcm9wIGFueXRoaW5nIGFmdGVyIHRoZSBmaXJzdCBgLmAgaW4gdGhlIGhvc3QsIGFzIHdlIG9ubHkgY2FyZSBhYm91dCB0aGUgYnVja2V0IG5hbWUuXG4gICAgICBCdWNrZXQ6IHRhcmJhbGxVcmkuaG9zdC5zcGxpdCgnLicpWzBdLFxuICAgICAgLy8gTm90ZTogdGhlIHBhdGhuYW1lIHBhcnQgaXMgYWJzb2x1dGUsIHNvIHdlIHN0cmlwIHRoZSBsZWFkaW5nIGAvYC5cbiAgICAgIEtleTogdGFyYmFsbFVyaS5wYXRobmFtZS5yZXBsYWNlKC9eXFwvLywgJycpLFxuICAgICAgVmVyc2lvbklkOiB0YXJiYWxsVXJpLnNlYXJjaFBhcmFtcy5nZXQoJ3ZlcnNpb25JZCcpID8/IHVuZGVmaW5lZCxcbiAgICB9KS5wcm9taXNlKCk7XG5cbiAgICBjb25zdCBpbnRlZ3JpdHlDaGVjayA9IGludGVncml0eShwYXlsb2FkLCBCdWZmZXIuZnJvbSh0YXJiYWxsLkJvZHkhKSk7XG4gICAgaWYgKHBheWxvYWQuaW50ZWdyaXR5ICE9PSBpbnRlZ3JpdHlDaGVjaykge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBJbnRlZ3JpdHkgY2hlY2sgZmFpbGVkOiAke3BheWxvYWQuaW50ZWdyaXR5fSAhPT0gJHtpbnRlZ3JpdHlDaGVja31gKTtcbiAgICB9XG5cbiAgICBjb25zdCB0YXIgPSBhd2FpdCBndW56aXAoQnVmZmVyLmZyb20odGFyYmFsbC5Cb2R5ISkpO1xuICAgIGxldCBkb3RKc2lpOiBCdWZmZXI7XG4gICAgbGV0IGxpY2Vuc2VUZXh0OiBCdWZmZXIgfCB1bmRlZmluZWQ7XG4gICAgbGV0IHBhY2thZ2VKc29uOiBCdWZmZXI7XG4gICAgdHJ5IHtcbiAgICAgIGNvbnN0IGV4dHJhY3RlZCA9IGF3YWl0IG5ldyBQcm9taXNlPHsgZG90SnNpaTogQnVmZmVyOyBsaWNlbnNlVGV4dD86IEJ1ZmZlcjsgcGFja2FnZUpzb246IEJ1ZmZlciB9Pigob2ssIGtvKSA9PiB7XG4gICAgICAgIGxldCBkb3RKc2lpQnVmZmVyOiBCdWZmZXIgfCB1bmRlZmluZWQ7XG4gICAgICAgIGxldCBsaWNlbnNlVGV4dEJ1ZmZlcjogQnVmZmVyIHwgdW5kZWZpbmVkO1xuICAgICAgICBsZXQgcGFja2FnZUpzb25EYXRhOiBCdWZmZXIgfCB1bmRlZmluZWQ7XG4gICAgICAgIGNvbnN0IGV4dHJhY3RvciA9IGV4dHJhY3QoeyBmaWxlbmFtZUVuY29kaW5nOiAndXRmLTgnIH0pXG4gICAgICAgICAgLm9uY2UoJ2Vycm9yJywgKHJlYXNvbikgPT4ge1xuICAgICAgICAgICAga28ocmVhc29uKTtcbiAgICAgICAgICB9KVxuICAgICAgICAgIC5vbmNlKCdmaW5pc2gnLCAoKSA9PiB7XG4gICAgICAgICAgICBpZiAoZG90SnNpaUJ1ZmZlciA9PSBudWxsKSB7XG4gICAgICAgICAgICAgIGtvKG5ldyBFcnJvcignTm8gLmpzaWkgZmlsZSBmb3VuZCBpbiB0YXJiYWxsIScpKTtcbiAgICAgICAgICAgIH0gZWxzZSBpZiAocGFja2FnZUpzb25EYXRhID09IG51bGwpIHtcbiAgICAgICAgICAgICAga28obmV3IEVycm9yKCdObyBwYWNrYWdlLmpzb24gZmlsZSBmb3VuZCBpbiB0YXJiYWxsIScpKTtcbiAgICAgICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgICAgIG9rKHsgZG90SnNpaTogZG90SnNpaUJ1ZmZlciwgbGljZW5zZVRleHQ6IGxpY2Vuc2VUZXh0QnVmZmVyLCBwYWNrYWdlSnNvbjogcGFja2FnZUpzb25EYXRhIH0pO1xuICAgICAgICAgICAgfVxuICAgICAgICAgIH0pXG4gICAgICAgICAgLm9uKCdlbnRyeScsIChoZWFkZXJzLCBzdHJlYW0sIG5leHQpID0+IHtcbiAgICAgICAgICAgIGNvbnN0IGNodW5rcyA9IG5ldyBBcnJheTxCdWZmZXI+KCk7XG4gICAgICAgICAgICBpZiAoaGVhZGVycy5uYW1lID09PSAncGFja2FnZS8uanNpaScpIHtcbiAgICAgICAgICAgICAgcmV0dXJuIHN0cmVhbS5vbignZGF0YScsIChjaHVuaykgPT4gY2h1bmtzLnB1c2goQnVmZmVyLmZyb20oY2h1bmspKSlcbiAgICAgICAgICAgICAgICAub25jZSgnZXJyb3InLCBrbylcbiAgICAgICAgICAgICAgICAub25jZSgnZW5kJywgKCkgPT4ge1xuICAgICAgICAgICAgICAgICAgZG90SnNpaUJ1ZmZlciA9IEJ1ZmZlci5jb25jYXQoY2h1bmtzKTtcbiAgICAgICAgICAgICAgICAgIC8vIFNraXAgb24gbmV4dCBydW5Mb29wIGl0ZXJhdGlvbiBzbyB3ZSBhdm9pZCBmaWxsaW5nIHRoZSBzdGFjay5cbiAgICAgICAgICAgICAgICAgIHNldEltbWVkaWF0ZShuZXh0KTtcbiAgICAgICAgICAgICAgICB9KVxuICAgICAgICAgICAgICAgIC5yZXN1bWUoKTtcbiAgICAgICAgICAgIH0gZWxzZSBpZiAoaGVhZGVycy5uYW1lID09PSAncGFja2FnZS9wYWNrYWdlLmpzb24nKSB7XG4gICAgICAgICAgICAgIHJldHVybiBzdHJlYW0ub24oJ2RhdGEnLCAoY2h1bmspID0+IGNodW5rcy5wdXNoKEJ1ZmZlci5mcm9tKGNodW5rKSkpXG4gICAgICAgICAgICAgICAgLm9uY2UoJ2Vycm9yJywga28pXG4gICAgICAgICAgICAgICAgLm9uY2UoJ2VuZCcsICgpID0+IHtcbiAgICAgICAgICAgICAgICAgIHBhY2thZ2VKc29uRGF0YSA9IEJ1ZmZlci5jb25jYXQoY2h1bmtzKTtcbiAgICAgICAgICAgICAgICAgIC8vIFNraXAgb24gbmV4dCBydW5Mb29wIGl0ZXJhdGlvbiBzbyB3ZSBhdm9pZCBmaWxsaW5nIHRoZSBzdGFjay5cbiAgICAgICAgICAgICAgICAgIHNldEltbWVkaWF0ZShuZXh0KTtcbiAgICAgICAgICAgICAgICB9KVxuICAgICAgICAgICAgICAgIC5yZXN1bWUoKTtcbiAgICAgICAgICAgIH0gZWxzZSBpZiAoaXNMaWNlbnNlRmlsZShoZWFkZXJzLm5hbWUpKSB7XG4gICAgICAgICAgICAgIHJldHVybiBzdHJlYW0ub24oJ2RhdGEnLCAoY2h1bmspID0+IGNodW5rcy5wdXNoKEJ1ZmZlci5mcm9tKGNodW5rKSkpXG4gICAgICAgICAgICAgICAgLm9uY2UoJ2Vycm9yJywga28pXG4gICAgICAgICAgICAgICAgLm9uY2UoJ2VuZCcsICgpID0+IHtcbiAgICAgICAgICAgICAgICAgIGxpY2Vuc2VUZXh0QnVmZmVyID0gQnVmZmVyLmNvbmNhdChjaHVua3MpO1xuICAgICAgICAgICAgICAgICAgLy8gU2tpcCBvbiBuZXh0IHJ1bkxvb3AgaXRlcmF0aW9uIHNvIHdlIGF2b2lkIGZpbGxpbmcgdGhlIHN0YWNrLlxuICAgICAgICAgICAgICAgICAgc2V0SW1tZWRpYXRlKG5leHQpO1xuICAgICAgICAgICAgICAgIH0pXG4gICAgICAgICAgICAgICAgLnJlc3VtZSgpO1xuICAgICAgICAgICAgfVxuICAgICAgICAgICAgLy8gU2tpcCBvbiBuZXh0IHJ1bkxvb3AgaXRlcmF0aW9uIHNvIHdlIGF2b2lkIGZpbGxpbmcgdGhlIHN0YWNrLlxuICAgICAgICAgICAgcmV0dXJuIHNldEltbWVkaWF0ZShuZXh0KTtcbiAgICAgICAgICB9KTtcbiAgICAgICAgZXh0cmFjdG9yLndyaXRlKHRhciwgKGVycikgPT4ge1xuICAgICAgICAgIGlmIChlcnIgIT0gbnVsbCkge1xuICAgICAgICAgICAga28oZXJyKTtcbiAgICAgICAgICB9XG4gICAgICAgICAgZXh0cmFjdG9yLmVuZCgpO1xuICAgICAgICB9KTtcbiAgICAgIH0pO1xuICAgICAgZG90SnNpaSA9IGV4dHJhY3RlZC5kb3RKc2lpO1xuICAgICAgbGljZW5zZVRleHQgPSBleHRyYWN0ZWQubGljZW5zZVRleHQ7XG4gICAgICBwYWNrYWdlSnNvbiA9IGV4dHJhY3RlZC5wYWNrYWdlSnNvbjtcbiAgICAgIG1ldHJpY3MucHV0TWV0cmljKE1ldHJpY05hbWUuSU5WQUxJRF9UQVJCQUxMLCAwLCBVbml0LkNvdW50KTtcbiAgICB9IGNhdGNoIChlcnIpIHtcbiAgICAgIGNvbnNvbGUuZXJyb3IoYEludmFsaWQgdGFyYmFsbCBjb250ZW50OiAke2Vycn1gKTtcbiAgICAgIG1ldHJpY3MucHV0TWV0cmljKE1ldHJpY05hbWUuSU5WQUxJRF9UQVJCQUxMLCAxLCBVbml0LkNvdW50KTtcbiAgICAgIHJldHVybjtcbiAgICB9XG4gICAgY29uc3QgbWV0YWRhdGEgPSB7IGRhdGU6IHBheWxvYWQudGltZSwgbGljZW5zZVRleHQ6IGxpY2Vuc2VUZXh0Py50b1N0cmluZygndXRmLTgnKSB9O1xuXG4gICAgbGV0IHBhY2thZ2VMaWNlbnNlOiBzdHJpbmc7XG4gICAgbGV0IHBhY2thZ2VOYW1lOiBzdHJpbmc7XG4gICAgbGV0IHBhY2thZ2VWZXJzaW9uOiBzdHJpbmc7XG4gICAgdHJ5IHtcbiAgICAgIGNvbnN0IHsgbGljZW5zZSwgbmFtZSwgdmVyc2lvbiB9ID0gdmFsaWRhdGVBc3NlbWJseShKU09OLnBhcnNlKGRvdEpzaWkudG9TdHJpbmcoJ3V0Zi04JykpKTtcbiAgICAgIHBhY2thZ2VMaWNlbnNlID0gbGljZW5zZTtcbiAgICAgIHBhY2thZ2VOYW1lID0gbmFtZTtcbiAgICAgIHBhY2thZ2VWZXJzaW9uID0gdmVyc2lvbjtcbiAgICAgIG1ldHJpY3MucHV0TWV0cmljKE1ldHJpY05hbWUuSU5WQUxJRF9BU1NFTUJMWSwgMCwgVW5pdC5Db3VudCk7XG4gICAgfSBjYXRjaCAoZXgpIHtcbiAgICAgIGNvbnNvbGUuZXJyb3IoYFBhY2thZ2UgZG9lcyBub3QgY29udGFpbiBhIHZhbGlkIGFzc2VtYmx5IC0tIGlnbm9yaW5nOiAke2V4fWApO1xuICAgICAgbWV0cmljcy5wdXRNZXRyaWMoTWV0cmljTmFtZS5JTlZBTElEX0FTU0VNQkxZLCAxLCBVbml0LkNvdW50KTtcbiAgICAgIHJldHVybjtcbiAgICB9XG5cbiAgICAvLyBFbnN1cmUgdGhlIGAuanNpaWAgbmFtZSwgdmVyc2lvbiAmIGxpY2Vuc2UgY29ycmVzcG9uZHMgdG8gdGhvc2UgaW4gYHBhY2thZ2UuanNvbmBcbiAgICBjb25zdCB7IG5hbWU6IHBhY2thZ2VKc29uTmFtZSwgdmVyc2lvbjogcGFja2FnZUpzb25WZXJzaW9uLCBsaWNlbnNlOiBwYWNrYWdlSnNvbkxpY2Vuc2UgfSA9IEpTT04ucGFyc2UocGFja2FnZUpzb24udG9TdHJpbmcoJ3V0Zi04JykpO1xuICAgIGlmIChwYWNrYWdlSnNvbk5hbWUgIT09IHBhY2thZ2VOYW1lIHx8IHBhY2thZ2VKc29uVmVyc2lvbiAhPT0gcGFja2FnZVZlcnNpb24gfHwgcGFja2FnZUpzb25MaWNlbnNlICE9PSBwYWNrYWdlTGljZW5zZSkge1xuICAgICAgY29uc29sZS5sb2coYElnbm9yaW5nIHBhY2thZ2Ugd2l0aCBtaXNtYXRjaGVkIG5hbWUsIHZlcnNpb24sIGFuZC9vciBsaWNlbnNlICgke3BhY2thZ2VKc29uTmFtZX1AJHtwYWNrYWdlSnNvblZlcnNpb259IGlzICR7cGFja2FnZUpzb25MaWNlbnNlfSAhPT0gJHtwYWNrYWdlTmFtZX1AJHtwYWNrYWdlVmVyc2lvbn0gaXMgJHtwYWNrYWdlTGljZW5zZX0pYCk7XG4gICAgICBtZXRyaWNzLnB1dE1ldHJpYyhNZXRyaWNOYW1lLk1JU01BVENIRURfSURFTlRJVFlfUkVKRUNUSU9OUywgMSwgVW5pdC5Db3VudCk7XG4gICAgICBjb250aW51ZTtcbiAgICB9XG4gICAgbWV0cmljcy5wdXRNZXRyaWMoTWV0cmljTmFtZS5NSVNNQVRDSEVEX0lERU5USVRZX1JFSkVDVElPTlMsIDAsIFVuaXQuQ291bnQpO1xuXG4gICAgLy8gRGlkIHdlIGlkZW50aWZ5IGEgbGljZW5zZSBmaWxlIG9yIG5vdD9cbiAgICBtZXRyaWNzLnB1dE1ldHJpYyhNZXRyaWNOYW1lLkZPVU5EX0xJQ0VOU0VfRklMRSwgbGljZW5zZVRleHQgIT0gbnVsbCA/IDEgOiAwLCBVbml0LkNvdW50KTtcblxuICAgIGNvbnN0IGFzc2VtYmx5S2V5ID0gYCR7Y29uc3RhbnRzLlNUT1JBR0VfS0VZX1BSRUZJWH0ke3BhY2thZ2VOYW1lfS92JHtwYWNrYWdlVmVyc2lvbn0ke2NvbnN0YW50cy5BU1NFTUJMWV9LRVlfU1VGRklYfWA7XG4gICAgY29uc29sZS5sb2coYFdyaXRpbmcgYXNzZW1ibHkgYXQgJHthc3NlbWJseUtleX1gKTtcbiAgICBjb25zdCBwYWNrYWdlS2V5ID0gYCR7Y29uc3RhbnRzLlNUT1JBR0VfS0VZX1BSRUZJWH0ke3BhY2thZ2VOYW1lfS92JHtwYWNrYWdlVmVyc2lvbn0ke2NvbnN0YW50cy5QQUNLQUdFX0tFWV9TVUZGSVh9YDtcbiAgICBjb25zb2xlLmxvZyhgV3JpdGluZyBwYWNrYWdlIGF0ICAke3BhY2thZ2VLZXl9YCk7XG4gICAgY29uc3QgbWV0YWRhdGFLZXkgPSBgJHtjb25zdGFudHMuU1RPUkFHRV9LRVlfUFJFRklYfSR7cGFja2FnZU5hbWV9L3Yke3BhY2thZ2VWZXJzaW9ufSR7Y29uc3RhbnRzLk1FVEFEQVRBX0tFWV9TVUZGSVh9YDtcbiAgICBjb25zb2xlLmxvZyhgV3JpdGluZyBtZXRhZGF0YSBhdCAgJHttZXRhZGF0YUtleX1gKTtcblxuICAgIC8vIHdlIHVwbG9hZCB0aGUgbWV0YWRhdGEgZmlsZSBmaXJzdCBiZWNhdXNlIHRoZSBjYXRhbG9nIGJ1aWxkZXIgZGVwZW5kcyBvblxuICAgIC8vIGl0IGFuZCBpcyB0cmlnZ2VyZWQgYnkgdGhlIGFzc2VtYmx5IGZpbGUgdXBsb2FkLlxuICAgIGNvbnNvbGUubG9nKGAke3BhY2thZ2VOYW1lfUAke3BhY2thZ2VWZXJzaW9ufSB8IFVwbG9hZGluZyBwYWNrYWdlIGFuZCBtZXRhZGF0YSBmaWxlc2ApO1xuICAgIGNvbnN0IFtwa2csIHN0b3JlZE1ldGFkYXRhXSA9IGF3YWl0IFByb21pc2UuYWxsKFtcbiAgICAgIGF3cy5zMygpLnB1dE9iamVjdCh7XG4gICAgICAgIEJ1Y2tldDogQlVDS0VUX05BTUUsXG4gICAgICAgIEtleTogcGFja2FnZUtleSxcbiAgICAgICAgQm9keTogdGFyYmFsbC5Cb2R5LFxuICAgICAgICBDYWNoZUNvbnRyb2w6ICdwdWJsaWMnLFxuICAgICAgICBDb250ZW50VHlwZTogJ2FwcGxpY2F0aW9uL29jdGV0LXN0cmVhbScsXG4gICAgICAgIE1ldGFkYXRhOiB7XG4gICAgICAgICAgJ0xhbWJkYS1Mb2ctR3JvdXAnOiBjb250ZXh0LmxvZ0dyb3VwTmFtZSxcbiAgICAgICAgICAnTGFtYmRhLUxvZy1TdHJlYW0nOiBjb250ZXh0LmxvZ1N0cmVhbU5hbWUsXG4gICAgICAgICAgJ0xhbWJkYS1SdW4tSWQnOiBjb250ZXh0LmF3c1JlcXVlc3RJZCxcbiAgICAgICAgfSxcbiAgICAgIH0pLnByb21pc2UoKSxcbiAgICAgIGF3cy5zMygpLnB1dE9iamVjdCh7XG4gICAgICAgIEJ1Y2tldDogQlVDS0VUX05BTUUsXG4gICAgICAgIEtleTogbWV0YWRhdGFLZXksXG4gICAgICAgIEJvZHk6IEpTT04uc3RyaW5naWZ5KG1ldGFkYXRhKSxcbiAgICAgICAgQ2FjaGVDb250cm9sOiAncHVibGljJyxcbiAgICAgICAgQ29udGVudFR5cGU6ICdhcHBsaWNhdGlvbi9qc29uJyxcbiAgICAgICAgTWV0YWRhdGE6IHtcbiAgICAgICAgICAnTGFtYmRhLUxvZy1Hcm91cCc6IGNvbnRleHQubG9nR3JvdXBOYW1lLFxuICAgICAgICAgICdMYW1iZGEtTG9nLVN0cmVhbSc6IGNvbnRleHQubG9nU3RyZWFtTmFtZSxcbiAgICAgICAgICAnTGFtYmRhLVJ1bi1JZCc6IGNvbnRleHQuYXdzUmVxdWVzdElkLFxuICAgICAgICB9LFxuICAgICAgfSkucHJvbWlzZSgpLFxuICAgIF0pO1xuXG4gICAgLy8gbm93IHdlIGNhbiB1cGxvYWQgdGhlIGFzc2VtYmx5LlxuICAgIGNvbnNvbGUubG9nKGAke3BhY2thZ2VOYW1lfUAke3BhY2thZ2VWZXJzaW9ufSB8IFVwbG9hZGluZyBhc3NlbWJseSBmaWxlYCk7XG4gICAgY29uc3QgYXNzZW1ibHkgPSBhd2FpdCBhd3MuczMoKS5wdXRPYmplY3Qoe1xuICAgICAgQnVja2V0OiBCVUNLRVRfTkFNRSxcbiAgICAgIEtleTogYXNzZW1ibHlLZXksXG4gICAgICBCb2R5OiBkb3RKc2lpLFxuICAgICAgQ2FjaGVDb250cm9sOiAncHVibGljJyxcbiAgICAgIENvbnRlbnRUeXBlOiAnYXBwbGljYXRpb24vanNvbicsXG4gICAgICBNZXRhZGF0YToge1xuICAgICAgICAnTGFtYmRhLUxvZy1Hcm91cCc6IGNvbnRleHQubG9nR3JvdXBOYW1lLFxuICAgICAgICAnTGFtYmRhLUxvZy1TdHJlYW0nOiBjb250ZXh0LmxvZ1N0cmVhbU5hbWUsXG4gICAgICAgICdMYW1iZGEtUnVuLUlkJzogY29udGV4dC5hd3NSZXF1ZXN0SWQsXG4gICAgICB9LFxuICAgIH0pLnByb21pc2UoKTtcblxuICAgIGNvbnN0IGNyZWF0ZWQ6IFN0YXRlTWFjaGluZUlucHV0ID0ge1xuICAgICAgYnVja2V0OiBCVUNLRVRfTkFNRSxcbiAgICAgIGFzc2VtYmx5OiB7XG4gICAgICAgIGtleTogYXNzZW1ibHlLZXksXG4gICAgICAgIHZlcnNpb25JZDogYXNzZW1ibHkuVmVyc2lvbklkLFxuICAgICAgfSxcbiAgICAgIHBhY2thZ2U6IHtcbiAgICAgICAga2V5OiBwYWNrYWdlS2V5LFxuICAgICAgICB2ZXJzaW9uSWQ6IHBrZy5WZXJzaW9uSWQsXG4gICAgICB9LFxuICAgICAgbWV0YWRhdGE6IHtcbiAgICAgICAga2V5OiBtZXRhZGF0YUtleSxcbiAgICAgICAgdmVyc2lvbklkOiBzdG9yZWRNZXRhZGF0YS5WZXJzaW9uSWQsXG4gICAgICB9LFxuICAgIH07XG4gICAgY29uc29sZS5sb2coYENyZWF0ZWQgb2JqZWN0czogJHtKU09OLnN0cmluZ2lmeShjcmVhdGVkLCBudWxsLCAyKX1gKTtcblxuICAgIGNvbnN0IHNmbiA9IGF3YWl0IGF3cy5zdGVwRnVuY3Rpb25zKCkuc3RhcnRFeGVjdXRpb24oe1xuICAgICAgaW5wdXQ6IEpTT04uc3RyaW5naWZ5KGNyZWF0ZWQpLFxuICAgICAgbmFtZTogc2ZuRXhlY3V0aW9uTmFtZUZyb21QYXJ0cyhwYWNrYWdlTmFtZSwgYHYke3BhY2thZ2VWZXJzaW9ufWAsIGNvbnRleHQuYXdzUmVxdWVzdElkKSxcbiAgICAgIHN0YXRlTWFjaGluZUFybjogU1RBVEVfTUFDSElORV9BUk4sXG4gICAgfSkucHJvbWlzZSgpO1xuICAgIGNvbnNvbGUubG9nKGBTdGFydGVkIFN0YXRlTWFjaGluZSBleGVjdXRpb246ICR7c2ZuLmV4ZWN1dGlvbkFybn1gKTtcbiAgICByZXN1bHQucHVzaChzZm4uZXhlY3V0aW9uQXJuKTtcbiAgfVxuXG4gIHJldHVybiByZXN1bHQ7XG59KTtcblxuZnVuY3Rpb24gZ3VuemlwKGRhdGE6IEJ1ZmZlcik6IFByb21pc2U8QnVmZmVyPiB7XG4gIGNvbnN0IGNodW5rcyA9IG5ldyBBcnJheTxCdWZmZXI+KCk7XG4gIHJldHVybiBuZXcgUHJvbWlzZTxCdWZmZXI+KChvaywga28pID0+XG4gICAgY3JlYXRlR3VuemlwKClcbiAgICAgIC5vbmNlKCdlcnJvcicsIGtvKVxuICAgICAgLm9uKCdkYXRhJywgKGNodW5rKSA9PiBjaHVua3MucHVzaChCdWZmZXIuZnJvbShjaHVuaykpKVxuICAgICAgLm9uY2UoJ2VuZCcsICgpID0+IG9rKEJ1ZmZlci5jb25jYXQoY2h1bmtzKSkpXG4gICAgICAuZW5kKGRhdGEpKTtcbn1cblxuLyoqXG4gKiBDaGVja3Mgd2hldGhlciB0aGUgcHJvdmlkZWQgZmlsZSBuYW1lIGNvcnJlc3BvbmRzIHRvIGEgbGljZW5zZSBmaWxlIG9yIG5vdC5cbiAqXG4gKiBAcGFyYW0gZmlsZU5hbWUgdGhlIGZpbGUgbmFtZSB0byBiZSBjaGVja2VkLlxuICpcbiAqIEByZXR1cm5zIGB0cnVlYCBJSUYgdGhlIGZpbGUgaXMgbmFtZWQgTElDRU5TRSBhbmQgaGFzIHRoZSAuTUQgb3IgLlRYVFxuICogICAgICAgICAgZXh0ZW5zaW9uLCBvciBubyBleHRlbnNpb24gYXQgYWxsLiBUaGUgdGVzdCBpcyBjYXNlLWluc2Vuc2l0aXZlLlxuICovXG5mdW5jdGlvbiBpc0xpY2Vuc2VGaWxlKGZpbGVOYW1lOiBzdHJpbmcpOiBib29sZWFuIHtcbiAgY29uc3QgZXh0ID0gZXh0bmFtZShmaWxlTmFtZSk7XG4gIGNvbnN0IHBvc3NpYmxlRXh0ZW5zaW9ucyA9IG5ldyBTZXQoWycnLCAnLm1kJywgJy50eHQnXSk7XG4gIHJldHVybiBwb3NzaWJsZUV4dGVuc2lvbnMuaGFzKGV4dC50b0xvd2VyQ2FzZSgpKVxuICAgICYmIGJhc2VuYW1lKGZpbGVOYW1lLCBleHQpLnRvVXBwZXJDYXNlKCkgPT09ICdMSUNFTlNFJztcbn1cblxuXG4vKipcbiAqIENyZWF0ZXMgYSBTdGVwRnVuY3Rpb24gZXhlY3V0aW9uIHJlcXVlc3QgbmFtZSBiYXNlZCBvbiB0aGUgcHJvdmlkZWQgcGFydHMuXG4gKiBUaGUgcmVzdWx0IGlzIGd1YXJhbnRlZWQgdG8gYmUgODAgY2hhcmFjdGVycyBvciBsZXNzIGFuZCB0byBjb250YWluIG9ubHlcbiAqIGNoYXJhY3RlcnMgdGhhdCBhcmUgdmFsaWQgZm9yIGEgU3RlcEZ1bmN0aW9uIGV4ZWN1dGlvbiByZXF1ZXN0IG5hbWUgZm9yIHdoaWNoXG4gKiBDbG91ZFdhdGNoIExvZ2dpbmcgY2FuIGJlIGVuYWJsZWQuIFRoZSByZXN1bHRpbmcgbmFtZSBpcyB2ZXJ5IGxpa2VseSB0b1xuICogYmUgdW5pcXVlIGZvciBhIGdpdmVuIGlucHV0LlxuICovXG5mdW5jdGlvbiBzZm5FeGVjdXRpb25OYW1lRnJvbVBhcnRzKGZpcnN0OiBzdHJpbmcsIC4uLnJlc3Q6IHJlYWRvbmx5IHN0cmluZ1tdKTogc3RyaW5nIHtcbiAgY29uc3QgcGFydHMgPSBbZmlyc3QsIC4uLnJlc3RdO1xuICBjb25zdCBuYW1lID0gcGFydHNcbiAgICAubWFwKChwYXJ0KSA9PiBwYXJ0LnJlcGxhY2UoL1teYS16MC05Xy1dKy9pZywgJ18nKSlcbiAgICAuam9pbignXycpXG4gICAgLnJlcGxhY2UoL15fL2csICcnKVxuICAgIC5yZXBsYWNlKC9fezIsfS9nLCAnXycpO1xuICBpZiAobmFtZS5sZW5ndGggPD0gODApIHtcbiAgICByZXR1cm4gbmFtZTtcbiAgfVxuICBjb25zdCBzdWZmaXggPSBjcmVhdGVIYXNoKCdzaGEyNTYnKVxuICAvLyBUaGUgaGFzaCBpcyBjb21wdXRlZCBiYXNlZCBvbiBpbnB1dCBhcmd1bWVudHMsIHRvIG1heGltaXplIHVuaWNpdHlcbiAgICAudXBkYXRlKHBhcnRzLmpvaW4oJ18nKSlcbiAgICAuZGlnZXN0KCdoZXgnKVxuICAgIC5zdWJzdHJpbmcoMCwgNik7XG4gIHJldHVybiBgJHtuYW1lLnN1YnN0cmluZygwLCA4MCAtIHN1ZmZpeC5sZW5ndGggLSAxKX1fJHtzdWZmaXh9YDtcbn1cbiJdfQ==