"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ingestion = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_lambda_event_sources_1 = require("@aws-cdk/aws-lambda-event-sources");
const aws_sqs_1 = require("@aws-cdk/aws-sqs");
const core_1 = require("@aws-cdk/core");
const constants_1 = require("./constants");
const ingestion_1 = require("./ingestion");
/**
 * The ingestion function receives messages from discovery integrations and
 * processes their payloads into the provided S3 Bucket.
 *
 * This function is also an `IGrantable`, so that it can be granted permissions
 * to read from the source S3 buckets.
 */
class Ingestion extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.queueRetentionPeriod = core_1.Duration.days(14);
        this.deadLetterQueue = new aws_sqs_1.Queue(this, 'DLQ', {
            encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            retentionPeriod: this.queueRetentionPeriod,
            visibilityTimeout: core_1.Duration.minutes(15),
        });
        this.queue = new aws_sqs_1.Queue(this, 'Queue', {
            deadLetterQueue: {
                maxReceiveCount: 5,
                queue: this.deadLetterQueue,
            },
            encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            retentionPeriod: this.queueRetentionPeriod,
            visibilityTimeout: core_1.Duration.minutes(15),
        });
        const handler = new ingestion_1.Ingestion(this, 'Default', {
            description: 'Ingests new package versions into the Construct Hub',
            environment: {
                BUCKET_NAME: props.bucket.bucketName,
                STATE_MACHINE_ARN: props.orchestration.stateMachine.stateMachineArn,
            },
            memorySize: 10240,
            timeout: core_1.Duration.minutes(15),
        });
        this.function = handler;
        props.bucket.grantWrite(this.function);
        props.orchestration.stateMachine.grantStartExecution(this.function);
        this.function.addEventSource(new aws_lambda_event_sources_1.SqsEventSource(this.queue, { batchSize: 1 }));
        // This event source is disabled, and can be used to re-process dead-letter-queue messages
        this.function.addEventSource(new aws_lambda_event_sources_1.SqsEventSource(this.deadLetterQueue, { batchSize: 1, enabled: false }));
        this.grantPrincipal = this.function.grantPrincipal;
        props.monitoring.watchful.watchLambdaFunction('Ingestion Function', handler);
    }
    metricFoundLicenseFile(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "FoundLicenseFile" /* FOUND_LICENSE_FILE */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    metricIneligibleLicense(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "IneligibleLicense" /* INELIGIBLE_LICENSE */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    metricInvalidAssembly(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "InvalidAssembly" /* INVALID_ASSEMBLY */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    metricInvalidTarball(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "InvalidTarball" /* INVALID_TARBALL */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    /**
     * This metrics is the total count of packages that were rejected due to
     * mismatched identity (name, version, license) between the `package.json`
     * file and te `.jsii` attribute.
     */
    metricMismatchedIdentityRejections(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "MismatchedIdentityRejections" /* MISMATCHED_IDENTITY_REJECTIONS */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
}
exports.Ingestion = Ingestion;
//# sourceMappingURL=data:application/json;base64,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