"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const console = require("console");
const https = require("https");
const url_1 = require("url");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const Environments_1 = require("aws-embedded-metrics/lib/environment/Environments");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const Nano = require("nano");
const aws = require("../shared/aws.lambda-shared");
const constants_1 = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const integrity_lambda_shared_1 = require("../shared/integrity.lambda-shared");
const constants_lambda_shared_1 = require("./constants.lambda-shared");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const normalizeNPMMetadata = require('normalize-registry-metadata');
const TIMEOUT_MILLISECONDS = 10000;
const CONSTRUCT_KEYWORDS = new Set(['cdk', 'aws-cdk', 'cdk8s', 'cdktf']);
const MARKER_FILE_NAME = 'couchdb-last-transaction-id';
const NPM_REPLICA_REGISTRY_URL = 'https://replicate.npmjs.com/';
// Configure embedded metrics format
aws_embedded_metrics_1.Configuration.environmentOverride = Environments_1.default.Lambda;
aws_embedded_metrics_1.Configuration.namespace = constants_lambda_shared_1.METRICS_NAMESPACE;
/**
 * This function triggers on a fixed schedule and reads a stream of changes from npmjs couchdb _changes endpoint.
 * Upon invocation the function starts reading from a sequence stored in an s3 object - the `marker`.
 * If the marker fails to load (or do not exist), the stream will start from `now` - the latest change.
 * For each change:
 *  - the package version tarball will be copied from the npm registry to a stating bucket.
 *  - a message will be sent to an sqs queue
 * npm registry API docs: https://github.com/npm/registry/blob/master/docs/REGISTRY-API.md
 * @param context a Lambda execution context
 */
async function handler(event, context) {
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    const stagingBucket = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const queueUrl = env_lambda_shared_1.requireEnv('QUEUE_URL');
    const initialMarker = await loadLastTransactionMarker(1800000 /* @aws-cdk/cdk initial release was at 1_846_709 */);
    const config = {
        includeDocs: true,
        // pause the changes reader after each request
        wait: true,
        since: initialMarker.toFixed(),
        // `changesReader.get` stops once a response with zero changes is received, however it waits too long
        //  since we want to terminate the Lambda function we define a timeout shorter than the default
        timeout: TIMEOUT_MILLISECONDS,
        // Only items with a name
        selector: {
            name: { $gt: null },
        },
        batchSize: 30,
    };
    const nano = Nano(NPM_REPLICA_REGISTRY_URL);
    const db = nano.db.use('registry');
    // We need to make an explicit Promise here, because otherwise Lambda won't
    // know when it's done...
    return new Promise((ok, ko) => {
        let updatedMarker = initialMarker;
        db.changesReader.get(config)
            .on('batch', aws_embedded_metrics_1.metricScope((metrics) => async (batch) => {
            var _a;
            // Clear automatically set dimensions - we don't need them (see https://github.com/awslabs/aws-embedded-metrics-node/issues/73)
            metrics.setDimensions();
            metrics.setProperty('StartSeq', updatedMarker);
            const startTime = Date.now();
            // Emit npm.js replication lag
            for (const { doc } of batch) {
                if ((_a = doc === null || doc === void 0 ? void 0 : doc.time) === null || _a === void 0 ? void 0 : _a.modified) {
                    metrics.putMetric("NpmJsChangeAge" /* NPMJS_CHANGE_AGE */, startTime - new Date(doc.time.modified).getTime(), aws_embedded_metrics_1.Unit.Milliseconds);
                }
            }
            try {
                console.log(`Received a batch of ${batch.length} element(s)`);
                metrics.putMetric("ChangeCount" /* CHANGE_COUNT */, batch.length, aws_embedded_metrics_1.Unit.Count);
                const lastSeq = Math.max(...batch.map((change) => change.seq));
                metrics.setProperty('EndSeq', updatedMarker);
                // Obtain the modified package version from the update event, and filter
                // out packages that are not of interest to us (not construct libraries).
                const versionInfos = getRelevantVersionInfos(batch, metrics);
                console.log(`Identified ${versionInfos.length} relevant package version update(s)`);
                metrics.putMetric("RelevantPackageVersions" /* RELEVANT_PACKAGE_VERSIONS */, versionInfos.length, aws_embedded_metrics_1.Unit.Count);
                // Process all remaining updates
                await Promise.all(versionInfos.map(async (infos) => {
                    const before = Date.now();
                    await processUpdatedVersion(infos, metrics);
                    metrics.putMetric("StagingTime" /* STAGING_TIME */, Date.now() - before, aws_embedded_metrics_1.Unit.Milliseconds);
                }));
                // Update the transaction marker in S3.
                await saveLastTransactionMarker(lastSeq);
                updatedMarker = lastSeq;
                // If we have enough time left before timeout, proceed with the next batch, otherwise we're done here.
                // Since the distribution of the time it takes to process each package/batch is non uniform, this is a best
                // effort, and we expect the function to timeout in some invocations, we rely on the downstream idempotency to handle this.
                if (context.getRemainingTimeInMillis() >= 30000 /* 30 seconds */) {
                    console.log('There is still time, requesting the next batch...');
                    // Note: the `resume` function is missing from the `nano` type definitions, but is there...
                    db.changesReader.resume();
                }
                else {
                    console.log('We are almost out of time, so stopping here.');
                    db.changesReader.stop();
                    metrics.putMetric("RemainingTime" /* REMAINING_TIME */, context.getRemainingTimeInMillis(), aws_embedded_metrics_1.Unit.Milliseconds);
                    ok({ initialMarker, updatedMarker });
                }
            }
            catch (err) {
                // An exception bubbled out, which means this Lambda execution has failed.
                console.error(`Unexpected error: ${err}`);
                db.changesReader.stop();
                ko(err);
            }
            finally {
                metrics.putMetric("BatchProcessingTime" /* BATCH_PROCESSING_TIME */, Date.now() - startTime, aws_embedded_metrics_1.Unit.Milliseconds);
            }
        }))
            .once('end', () => {
            console.log('No more updates to process, exiting.');
            ok({ initialMarker, updatedMarker });
        });
    });
    //#region Last transaction marker
    /**
     * Loads the last transaction marker from S3.
     *
     * @param defaultValue the value to return in case the marker does not exist
     *
     * @returns the value of the last transaction marker.
     */
    async function loadLastTransactionMarker(defaultValue) {
        try {
            const response = await aws.s3().getObject({
                Bucket: stagingBucket,
                Key: MARKER_FILE_NAME,
            }).promise();
            const marker = Number.parseInt(response.Body.toString('utf-8'), 10);
            console.log(`Read last transaction marker: ${marker}`);
            return marker;
        }
        catch (error) {
            if (error.code !== 'NoSuchKey') {
                throw error;
            }
            console.log(`Marker object (s3://${stagingBucket}/${MARKER_FILE_NAME}) does not exist, starting from the default (${defaultValue})`);
            return defaultValue;
        }
    }
    /**
     * Updates the last transaction marker in S3.
     *
     * @param sequence the last transaction marker value
     */
    async function saveLastTransactionMarker(sequence) {
        console.log(`Updating last transaction marker to ${sequence}`);
        return putObject(MARKER_FILE_NAME, sequence.toFixed(), { ContentType: 'text/plain; charset=UTF-8' });
    }
    //#endregion
    //#region Business Logic
    async function processUpdatedVersion({ infos, modified, seq }, metrics) {
        try {
            // Download the tarball
            const tarball = await httpGet(infos.dist.tarball);
            // Store the tarball into the staging bucket
            // - infos.dist.tarball => https://registry.npmjs.org/<@scope>/<name>/-/<name>-<version>.tgz
            // - stagingKey         =>                     staged/<@scope>/<name>/-/<name>-<version>.tgz
            const stagingKey = `${"staged/" /* STAGED_KEY_PREFIX */}${new url_1.URL(infos.dist.tarball).pathname}`.replace(/\/{2,}/g, '/');
            await putObject(stagingKey, tarball, {
                ContentType: 'application/octet-stream',
                Metadata: {
                    'Modified-At': modified.toISOString(),
                    'Origin-Integrity': infos.dist.shasum,
                    'Origin-URI': infos.dist.tarball,
                    'Sequence': seq.toFixed(),
                },
            });
            // Prepare SQS message for ingestion
            const messageBase = {
                tarballUri: `s3://${stagingBucket}/${stagingKey}`,
                metadata: {
                    dist: infos.dist.tarball,
                    seq: seq.toFixed(),
                },
                time: modified.toUTCString(),
            };
            const message = {
                ...messageBase,
                integrity: integrity_lambda_shared_1.integrity(messageBase, tarball),
            };
            // Send the SQS message out
            await aws.sqs().sendMessage({
                MessageBody: JSON.stringify(message, null, 2),
                QueueUrl: queueUrl,
            }).promise();
            metrics.putMetric("StagingFailureCount" /* STAGING_FAILURE_COUNT */, 0, aws_embedded_metrics_1.Unit.Count);
        }
        catch (err) {
            // Something failed, store the payload in the problem prefix, and move on.
            console.error(`[${seq}] Failed processing, logging error to S3 and resuming work. ${infos.name}@${infos.version}: ${err}`);
            metrics.putMetric("StagingFailureCount" /* STAGING_FAILURE_COUNT */, 1, aws_embedded_metrics_1.Unit.Count);
            await putObject(`${"failed/" /* FAILED_KEY_PREFIX */}${seq}`, JSON.stringify({ ...infos, _construct_hub_failure_reason: err }, null, 2), {
                ContentType: 'application/json',
                Metadata: {
                    'Modified-At': modified.toISOString(),
                },
            });
        }
    }
    //#endregion
    //#region Asynchronous Primitives
    /**
     * Makes an HTTP GET request, and returns the resulting payload.
     *
     * @param url the URL to get.
     *
     * @returns a Buffer containing the received data.
     */
    function httpGet(url) {
        return new Promise((ok, ko) => {
            https.get(url, (response) => {
                if (response.statusCode !== 200) {
                    throw new Error(`Unsuccessful GET: ${response.statusCode} - ${response.statusMessage}`);
                }
                let body = Buffer.alloc(0);
                response.on('data', (chunk) => body = Buffer.concat([body, Buffer.from(chunk)]));
                response.once('close', () => ok(body));
                response.once('error', ko);
            });
        });
    }
    /**
     * Puts an object in the staging bucket, with standardized object metadata.
     *
     * @param key  the key for the object to be put.
     * @param body the body of the object to be put.
     * @param opts any other options to use when sending the S3 request.
     *
     * @returns the result of the S3 request.
     */
    function putObject(key, body, opts = {}) {
        return aws.s3().putObject({
            Bucket: stagingBucket,
            Key: key,
            Body: body,
            Metadata: {
                'Lambda-Log-Group': context.logGroupName,
                'Lambda-Log-Stream': context.logStreamName,
                'Lambda-Run-Id': context.awsRequestId,
                ...opts.Metadata,
            },
            ...opts,
        }).promise();
    }
    //#endregion
}
exports.handler = handler;
/**
 * Obtains the `VersionInfo` corresponding to the modified version(s) in the
 * provided `Change` objects, ensures they are relevant (construct libraries),
 * and returns those only.
 *
 * @param changes the changes to be processed.
 *
 * @returns a list of `VersionInfo` objects
 */
function getRelevantVersionInfos(changes, metrics) {
    var _a;
    const result = new Array();
    for (const change of changes) {
        // Filter out all elements that don't have a "name" in the document, as
        // these are schemas, which are not relevant to our business here.
        if (change.doc.name === undefined) {
            console.error(`[${change.seq}] Changed document contains no 'name': ${change.id}`);
            metrics.putMetric("UnprocessableEntity" /* UNPROCESSABLE_ENTITY */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        // The normalize function change the object in place, if the doc object is invalid it will return undefined
        if (normalizeNPMMetadata(change.doc) === undefined) {
            console.error(`[${change.seq}] Changed document invalid, npm normalize returned undefined: ${change.id}`);
            metrics.putMetric("UnprocessableEntity" /* UNPROCESSABLE_ENTITY */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        // Sometimes, there are no versions in the document. We skip those.
        if (change.doc.versions == null) {
            console.error(`[${change.seq}] Changed document contains no 'versions': ${change.id}`);
            metrics.putMetric("UnprocessableEntity" /* UNPROCESSABLE_ENTITY */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        // Sometimes, there is no 'time' entry in the document. We skip those.
        if (change.doc.time == null) {
            console.error(`[${change.seq}] Changed document contains no 'time': ${change.id}`);
            metrics.putMetric("UnprocessableEntity" /* UNPROCESSABLE_ENTITY */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        // Get the last modification date from the change
        const sortedUpdates = Object.entries(change.doc.time)
            // Ignore the "created" and "modified" keys here
            .filter(([key]) => key !== 'created' && key !== 'modified')
            // Parse all the dates to ensure they are comparable
            .map(([version, isoDate]) => [version, new Date(isoDate)])
            // Sort by date, descending
            .sort(([, l], [, r]) => r.getTime() - l.getTime());
        metrics.putMetric("PackageVersionCount" /* PACKAGE_VERSION_COUNT */, sortedUpdates.length, aws_embedded_metrics_1.Unit.Count);
        let latestModified;
        for (const [version, modified] of sortedUpdates) {
            if (latestModified == null || latestModified.getTime() === modified.getTime()) {
                const infos = change.doc.versions[version];
                if (infos == null) {
                    // Could be the version in question was un-published.
                    console.log(`[${change.seq}] Could not find info for "${change.doc.name}@${version}". Was it un-published?`);
                }
                else if (isRelevantPackageVersion(infos)) {
                    metrics.putMetric("PackageVersionAge" /* PACKAGE_VERSION_AGE */, Date.now() - modified.getTime(), aws_embedded_metrics_1.Unit.Milliseconds);
                    const isEligible = usesEligibleLicenses(infos);
                    metrics.putMetric("IneligibleLicense" /* INELIGIBLE_LICENSE */, isEligible ? 0 : 1, aws_embedded_metrics_1.Unit.Count);
                    if (usesEligibleLicenses(infos)) {
                        result.push({ infos, modified, seq: change.seq });
                    }
                    else {
                        console.log(`[${change.seq}] Package "${change.doc.name}@${version}" does not use allow-listed license: ${(_a = infos.license) !== null && _a !== void 0 ? _a : 'UNLICENSED'}`);
                    }
                }
                else {
                    console.log(`[${change.seq}] Ignoring "${change.doc.name}@${version}" as it is not a construct library.`);
                }
                latestModified = modified;
            }
        }
    }
    return result;
    function isRelevantPackageVersion(infos) {
        var _a;
        if (infos.jsii == null) {
            return false;
        }
        return infos.name === 'construct'
            || infos.name === 'aws-cdk-lib'
            || infos.name.startsWith('@aws-cdk')
            || ((_a = infos.keywords) === null || _a === void 0 ? void 0 : _a.some((kw) => CONSTRUCT_KEYWORDS.has(kw)));
    }
    function usesEligibleLicenses({ license }) {
        var _a;
        return constants_1.ELIGIBLE_LICENSES.has((_a = license === null || license === void 0 ? void 0 : license.toUpperCase()) !== null && _a !== void 0 ? _a : 'UNLICENSED');
    }
}
//# sourceMappingURL=data:application/json;base64,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