"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const language_1 = require("../../../backend/shared/language");
const transliterator_1 = require("../../../backend/transliterator");
const repository_1 = require("../../../codeartifact/repository");
const monitoring_1 = require("../../../monitoring");
test('basic use', () => {
    // GIVEN
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'TestStack');
    const bucket = new aws_s3_1.Bucket(stack, 'Bucket');
    const monitoring = new monitoring_1.Monitoring(stack, 'Monitoring', {
        alarmActions: { highSeverity: 'high-sev', normalSeverity: 'normal-sev' },
        dashboardName: 'monitoring',
    });
    // WHEN
    new transliterator_1.Transliterator(stack, 'Transliterator', {
        bucket,
        language: language_1.DocumentationLanguage.PYTHON,
        monitoring,
    });
    // THEN
    expect(app.synth().getStackByName(stack.stackName).template).toMatchSnapshot({
        Outputs: expect.anything(),
        Parameters: expect.anything(),
        Resources: ignoreResources(stack, bucket, monitoring),
    });
});
test('CodeArtifact repository', () => {
    // GIVEN
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'TestStack');
    const bucket = new aws_s3_1.Bucket(stack, 'Bucket');
    const monitoring = new monitoring_1.Monitoring(stack, 'Monitoring', {
        alarmActions: { highSeverity: 'high-sev', normalSeverity: 'normal-sev' },
        dashboardName: 'monitoring',
    });
    const codeArtifact = new repository_1.Repository(stack, 'CodeArtifact');
    // WHEN
    new transliterator_1.Transliterator(stack, 'Transliterator', {
        bucket,
        codeArtifact,
        language: language_1.DocumentationLanguage.PYTHON,
        monitoring,
    });
    // THEN
    expect(stack).toHaveResourceLike('AWS::Lambda::Function', {
        Environment: {
            Variables: stack.resolve({
                CODE_ARTIFACT_DOMAIN_NAME: codeArtifact.repositoryDomainName,
                CODE_ARTIFACT_DOMAIN_OWNER: codeArtifact.repositoryDomainOwner,
                CODE_ARTIFACT_REPOSITORY_ENDPOINT: codeArtifact.repositoryNpmEndpoint,
            }),
        },
    });
    expect(app.synth().getStackByName(stack.stackName).template).toMatchSnapshot({
        Outputs: expect.anything(),
        Parameters: expect.anything(),
        Resources: ignoreResources(stack, bucket, monitoring, codeArtifact),
    });
});
test('VPC Endpoints', () => {
    // GIVEN
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'TestStack');
    const bucket = new aws_s3_1.Bucket(stack, 'Bucket');
    const monitoring = new monitoring_1.Monitoring(stack, 'Monitoring', {
        alarmActions: { highSeverity: 'high-sev', normalSeverity: 'normal-sev' },
        dashboardName: 'monitoring',
    });
    const vpc = new aws_ec2_1.Vpc(stack, 'VPC', { subnetConfiguration: [{ name: 'Isolated', subnetType: aws_ec2_1.SubnetType.ISOLATED }] });
    const codeArtifactApi = vpc.addInterfaceEndpoint('CodeArtifact.API', {
        service: new aws_ec2_1.InterfaceVpcEndpointAwsService('codeartifact.api'),
    });
    const codeArtifact = vpc.addInterfaceEndpoint('CodeArtifact.Repo', {
        service: new aws_ec2_1.InterfaceVpcEndpointAwsService('codeartifact.repositories'),
    });
    const s3 = vpc.addGatewayEndpoint('S3', {
        service: aws_ec2_1.GatewayVpcEndpointAwsService.S3,
    });
    // WHEN
    new transliterator_1.Transliterator(stack, 'Transliterator', {
        bucket,
        language: language_1.DocumentationLanguage.PYTHON,
        monitoring,
        vpc,
        vpcEndpoints: { codeArtifactApi, codeArtifact, s3 },
    });
    // THEN
    expect(stack).toHaveResourceLike('AWS::Lambda::Function', {
        Environment: {
            Variables: stack.resolve({
                CODE_ARTIFACT_API_ENDPOINT: core_1.Fn.select(1, core_1.Fn.split(':', core_1.Fn.select(0, codeArtifactApi.vpcEndpointDnsEntries))),
            }),
        },
    });
    expect(app.synth().getStackByName(stack.stackName).template).toMatchSnapshot({
        Outputs: expect.anything(),
        Parameters: expect.anything(),
        Resources: ignoreResources(stack, bucket, monitoring, vpc),
    });
});
test('VPC Endpoints and CodeArtifact repository', () => {
    // GIVEN
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'TestStack');
    const bucket = new aws_s3_1.Bucket(stack, 'Bucket');
    const repository = new repository_1.Repository(stack, 'CodeArtifact');
    const monitoring = new monitoring_1.Monitoring(stack, 'Monitoring', {
        alarmActions: { highSeverity: 'high-sev', normalSeverity: 'normal-sev' },
        dashboardName: 'monitoring',
    });
    const vpc = new aws_ec2_1.Vpc(stack, 'VPC', { subnetConfiguration: [{ name: 'Isolated', subnetType: aws_ec2_1.SubnetType.ISOLATED }] });
    const codeArtifactApi = vpc.addInterfaceEndpoint('CodeArtifact.API', {
        service: new aws_ec2_1.InterfaceVpcEndpointAwsService('codeartifact.api'),
    });
    const codeArtifact = vpc.addInterfaceEndpoint('CodeArtifact.Repo', {
        service: new aws_ec2_1.InterfaceVpcEndpointAwsService('codeartifact.repositories'),
    });
    const s3 = vpc.addGatewayEndpoint('S3', {
        service: aws_ec2_1.GatewayVpcEndpointAwsService.S3,
    });
    // WHEN
    new transliterator_1.Transliterator(stack, 'Transliterator', {
        bucket,
        codeArtifact: repository,
        language: language_1.DocumentationLanguage.PYTHON,
        monitoring,
        vpc,
        vpcEndpoints: { codeArtifactApi, codeArtifact, s3 },
    });
    // THEN
    expect(stack).toHaveResourceLike('AWS::Lambda::Function', {
        Environment: {
            Variables: stack.resolve({
                CODE_ARTIFACT_DOMAIN_NAME: repository.repositoryDomainName,
                CODE_ARTIFACT_DOMAIN_OWNER: repository.repositoryDomainOwner,
                CODE_ARTIFACT_REPOSITORY_ENDPOINT: repository.repositoryNpmEndpoint,
                CODE_ARTIFACT_API_ENDPOINT: core_1.Fn.select(1, core_1.Fn.split(':', core_1.Fn.select(0, codeArtifactApi.vpcEndpointDnsEntries))),
            }),
        },
    });
    expect(app.synth().getStackByName(stack.stackName).template).toMatchSnapshot({
        Outputs: expect.anything(),
        Parameters: expect.anything(),
        Resources: ignoreResources(stack, bucket, repository, monitoring, vpc),
    });
});
/**
 * Creates a property matcher that expects "anything" for all CfnResource
 * logicalIds found within the provided constructs.
 */
function ignoreResources(stack, ...constructs) {
    const result = {};
    for (const resource of resourcesIn(...constructs)) {
        result[stack.resolve(resource.logicalId)] = expect.anything();
    }
    return result;
    function* resourcesIn(...scopes) {
        for (const scope of scopes) {
            if (core_1.CfnResource.isCfnResource(scope)) {
                yield scope;
            }
            yield* resourcesIn(...scope.node.children);
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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