# lightcon

## Installation
### Pip
```pip install lightcon```
<!--
## Changelog
### 1.1
#### Added
fast_daq module
examples/fast_daq_wrapper_example.py
#### Fixed
#### Removed

### 1.0
Initial release
<!--
#### Added
#### Fixed
#### Removed
-->

## Modules

### `lightcon.beam_alignment`
Beam Alignment App REST client

### `lightcon.common`
Tools and converters

### `lightcon.laser_clients`
REST API clients for PHAROS and CARBIDE lasers

Example:
```python
from lightcon.laser_clients import Carbide, Pharos
import time

my_laser = Pharos('192.168.8.113')
# my_laser = Carbide('192.168.8.113')

pp_ratio = my_laser.get_pp()
print("Pulse picker ratio: {:d}".format(pp_ratio))

target_pp_ratio = pp_ratio + 1
print("Setting pulse picker to PP={:d}...".format(
    target_pp_ratio), end='', flush=True)

my_laser.set_pp(target_pp_ratio, blocking=True)
print("OK", flush=True)

print("Setting pulse picker to PP={:d}...".format(pp_ratio), end='', flush=True)
my_laser.set_pp(pp_ratio, blocking=True)
print("OK", flush=True)

print("Enabling laser output...")
my_laser.enable_output()

print("Waiting for 3s...")
time.sleep(3)

print("Disabling laser output...")
my_laser.close_output()
```

Output:
```python console
Pharos initialized at http://192.168.8.113:20020/v1/
Pulse picker ratio: 1
Setting pulse picker to PP=2...OK
Setting pulse picker to PP=1...OK
Enabling laser output...
Waiting for 3s...
Disabling laser output...
```


### `lightcon.harpia`
HARPIA Service App REST client

### `lightcon.fast_daq`
Interface to the fast single-channel DAQ (E13-10023-02 or newer) DLL wrapper
Example:
```python
import lightcon.fast_daq

fdw = lightcon.fast_daq.FastDaqWrapper()

if fdw.is_connected():
    # sets missing trigger/clock timeout, after which TimeoutException is raised
    fdw.set_timeout(1000)
    
    # choose channel 'PFI0' for external clocking, 'internal' for internal clocking. Use 'rising' or 'falling' for active_edge
    fdw.configure_sample_clock(channel = 'PFI0', active_edge = 'rising')
    
    # choose channel 'PFI0' for external clocking, 'internal' for internal triggering
    fdw.configure_start_trigger(channel = 'internal')
    
    # sets external trigger delay for sampling to 100 ns
    fdw.set_external_trigger_delay(1000)
    
    # acquires n = 1000 samples as one-dimensional array
    data = fdw.get_daq_data(10)
    
    fdw.close()
```

### `lightcon.harpia_daq`
Interface to the universal six-channel DAQ (PE04-005-04 or newer) DLL wrapper
Example:
```python
import lightcon.harpia_daq

# provide in ascending order to keep plot labels right
enabled_channels = ['AI0', 'AI1', 'AI3']
hdw = lightcon.harpia_daq.HarpiaDaqWrapper()
    
if hdw.is_connected:
    # sets missing trigger/clock timeout, after which TimeoutException is raised
    hdw.set_timeout(100)
    
    # enable analog input channels
    hdw.enable_channels(enabled_channels)
    
    # choose channel 'PFI0' - 'PFI5' for external clocking, 'internal' for internal clocking. Use 'rising' or 'falling' for active_edge
    hdw.configure_sample_clock(channel = 'internal', active_edge = 'falling')
    
    # choose channel 'PFI0' - 'PFI5' for external clocking, 'internal' for internal triggering
    hdw.configure_start_trigger(channel = 'PFI0')
    
    # acquires n=1000 samples and arranges to (m,n) two-dimensonal array, where m is number of enabled channels (in ascending order)
    data = hdw.get_daq_data(1000)    
```

### `lightcon.timing_controller`
Timing controller API

Example:
```python
import lightcon.timing_controller

tc = TimingController()
tc.connect()

if tc.connected:
    print ('Trigger source', 'EXTERNAL' if tc.get_trigger_source() == 1 else 'INTERNAL {:} Hz'.format(tc.get_frequency()))
    print ('\n'.join(['Channel {:}, delay {:} ns, {:}'.format(i, tc.get_delay(i), 'ENABLED' if tc.get_enabled(i)==1 else 'DISABLED') for i in [1,2,3,4]]))
```

Output:
```python console
Connected to timer_stopwatch on COM12 (E131001401, firmware version: 1.0.1)
Trigger source INTERNAL 1750 Hz
Channel 1, delay 0 ns, DISABLED
Channel 2, delay 0 ns, DISABLED
Channel 3, delay 0 ns, DISABLED
Channel 4, delay 0 ns, DISABLED
```

### `lightcon.eth_motor_board`
EthMotorBoard API

Example:
```python
from lightcon.eth_motor_board import EthMotorBoard

emb = EthMotorBoard()

print ('Hardware version:', emb.send('HARDWARE_VERSION'))
```

Output:
```python console
Successfullly connected to EthMotorBoard, name: Beam alignment, firmware version: 5.5.1
Hardware version: E04-351-03
```

CAN extension example:
```python
from lightcon.eth_motor_board import EthMotorBoard
from lightcon.common import BytesArrayToFloat

emb = EthMotorBoard()

can_output = emb.GetRegister(0x180, 0x0201, 2)
print ('Response status', can_output[0])
print ('Response data', can_output[1])
print ('Quadrant detector SUM voltage', BytesArrayToFloat(can_output[1]), 'V')
```

Output:
```python
Successfullly connected to EthMotorBoard, name: Beam alignment, firmware version: 5.5.1
Response status 0
Response data [54, 121, 86, 64]
Quadrant detector SUM voltage 3.3511481285095215 V
```


### `lightcon.wintopas`
WinTopas REST client

### `lightcon.camera_app_client`
CameraApp REST client

### `lightcon.style`
Light Conversion style for 1D graphs


Example:
```python
import lightcon.style

lightcon.style.apply_style()

...plot whatever here...

lightcon.style.add_watermarks()
```



