"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const cf = require("@aws-cdk/aws-cloudfront");
const s3 = require("@aws-cdk/aws-s3");
const aws_s3_deployment_1 = require("@aws-cdk/aws-s3-deployment");
const cdk = require("@aws-cdk/core");
const extensions = require("../extensions");
test('minimal usage', () => {
    // GIVEN
    const app = new cdk.App();
    const stack = new cdk.Stack(app, 'demo-stack');
    // WHEN
    // create the cloudfront distribution with extension(s)
    const customErrorPage = new extensions.CustomErrorPage(stack, 'CustomErrorPage');
    // create s3 bucket
    const bucket = new s3.Bucket(customErrorPage, 'demoBucket', {
        autoDeleteObjects: true,
        removalPolicy: cdk.RemovalPolicy.DESTROY,
        websiteIndexDocument: 'index.html',
        websiteErrorDocument: 'error.html',
    });
    // Put demo Object to Bucket.
    new aws_s3_deployment_1.BucketDeployment(customErrorPage, 'Deployment', {
        sources: [aws_s3_deployment_1.Source.asset(path.join(__dirname, './'))],
        destinationBucket: bucket,
        retainOnDelete: false,
    });
    // cloudFront OriginAccessIdentity for bucket
    const originAccessIdentity = new cf.OriginAccessIdentity(customErrorPage, 'OriginAccessIdentity', {
        comment: `CloudFront OriginAccessIdentity for ${bucket.bucketName}`,
    });
    // cloudfront distribution
    const distribution = new cf.CloudFrontWebDistribution(stack, 'distribution', {
        originConfigs: [
            {
                s3OriginSource: {
                    originAccessIdentity,
                    s3BucketSource: bucket,
                },
                behaviors: [{
                        isDefaultBehavior: true,
                        defaultTtl: cdk.Duration.seconds(10),
                        lambdaFunctionAssociations: [customErrorPage],
                    }],
            },
        ],
    });
    new cdk.CfnOutput(stack, 'distributionDomainName', {
        value: distribution.distributionDomainName,
    });
    // THEN
    expect(assert_1.SynthUtils.synthesize(stack).template).toMatchSnapshot();
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            DefaultCacheBehavior: {
                AllowedMethods: [
                    'GET',
                    'HEAD',
                ],
                CachedMethods: [
                    'GET',
                    'HEAD',
                ],
                Compress: true,
                DefaultTTL: 10,
                ForwardedValues: {
                    Cookies: {
                        Forward: 'none',
                    },
                    QueryString: false,
                },
                LambdaFunctionAssociations: [
                    {
                        EventType: 'origin-response',
                        LambdaFunctionARN: {
                            'Fn::GetAtt': [
                                'CustomErrorPageNestedStackCustomErrorPageNestedStackResource5229F8E2',
                                'Outputs.demostackCustomErrorPageCustomFuncCurrentVersion4B2B308BRef',
                            ],
                        },
                    },
                ],
                TargetOriginId: 'origin1',
                ViewerProtocolPolicy: 'redirect-to-https',
            },
            DefaultRootObject: 'index.html',
        },
    });
});
//# sourceMappingURL=data:application/json;base64,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