"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const AWS = require("aws-sdk");
const AWSMock = require("aws-sdk-mock");
const constants = require("../../../backend/shared/constants");
const transliterator_lambda_1 = require("../../../backend/transliterator/transliterator.lambda");
jest.mock('child_process');
jest.mock('jsii-docgen');
jest.mock('jsii-rosetta/lib/commands/transliterate');
jest.mock('../../../backend/shared/code-artifact.lambda-shared');
beforeEach((done) => {
    AWSMock.setSDKInstance(AWS);
    process.env.TARGET_LANGUAGE = 'typescript';
    done();
});
afterEach((done) => {
    AWSMock.restore();
    delete process.env.TARGET_LANGUAGE;
    transliterator_lambda_1.reset();
    done();
});
describe('VPC Endpoints', () => {
    const previousEnv = process.env;
    const endpoint = 'codeartifact.d.bermuda-triangle-1.amazonaws.com';
    const apiEndpoint = 'codeartifact.api.bermuda-triangle-1.amazonaws.com';
    const domain = 'domain-name';
    const domainOwner = '123456789012';
    beforeAll(() => {
        process.env = {
            ...previousEnv,
            CODE_ARTIFACT_REPOSITORY_ENDPOINT: endpoint,
            CODE_ARTIFACT_DOMAIN_NAME: domain,
            CODE_ARTIFACT_DOMAIN_OWNER: domainOwner,
            CODE_ARTIFACT_API_ENDPOINT: apiEndpoint,
        };
    });
    afterAll(() => {
        process.env = { ...previousEnv };
    });
    test('happy path', async () => {
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        const forPackage = require('jsii-docgen').Documentation.forPackage;
        forPackage.mockImplementation(async (target) => {
            return new MockDocumentation(target);
        });
        // GIVEN
        const packageScope = 'scope';
        const packageName = 'package-name';
        const packageVersion = '1.2.3-dev.4';
        const s3Event = {
            Records: [{
                    awsRegion: 'bemuda-triangle-1',
                    s3: {
                        bucket: {
                            name: 'dummy-bucket',
                        },
                        object: {
                            key: `${constants.STORAGE_KEY_PREFIX}%40${packageScope}/${packageName}/v${packageVersion}${constants.ASSEMBLY_KEY_SUFFIX}`,
                            versionId: 'VersionId',
                        },
                    },
                }],
        };
        const event = {
            Records: [{
                    Sns: {
                        Message: JSON.stringify(s3Event),
                    },
                }],
        };
        const assembly = {
            targets: { python: {} },
        };
        // mock the assembly request
        mockFetchAssembly(assembly);
        // mock the file uploads
        mockPutDocs('/docs-typescript.md');
        const created = await transliterator_lambda_1.handler(event, {});
        expect(created.length).toEqual(1);
        expect(created[0].key).toEqual(`data/@${packageScope}/${packageName}/v${packageVersion}/docs-typescript.md`);
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        expect(require('../../../backend/shared/code-artifact.lambda-shared').logInWithCodeArtifact).toHaveBeenCalledWith({
            endpoint,
            domain,
            domainOwner,
            apiEndpoint,
        });
    });
});
test('uploads a file per language (scoped package)', async () => {
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const forPackage = require('jsii-docgen').Documentation.forPackage;
    forPackage.mockImplementation(async (target) => {
        return new MockDocumentation(target);
    });
    // GIVEN
    const packageScope = 'scope';
    const packageName = 'package-name';
    const packageVersion = '1.2.3-dev.4';
    const s3Event = {
        Records: [{
                awsRegion: 'bemuda-triangle-1',
                s3: {
                    bucket: {
                        name: 'dummy-bucket',
                    },
                    object: {
                        key: `${constants.STORAGE_KEY_PREFIX}%40${packageScope}/${packageName}/v${packageVersion}${constants.ASSEMBLY_KEY_SUFFIX}`,
                        versionId: 'VersionId',
                    },
                },
            }],
    };
    const event = {
        Records: [{
                Sns: {
                    Message: JSON.stringify(s3Event),
                },
            }],
    };
    const assembly = {
        targets: { python: {} },
    };
    // mock the assembly request
    mockFetchAssembly(assembly);
    // mock the file uploads
    mockPutDocs('/docs-typescript.md');
    const created = await transliterator_lambda_1.handler(event, {});
    expect(created.length).toEqual(1);
    expect(created[0].key).toEqual(`data/@${packageScope}/${packageName}/v${packageVersion}/docs-typescript.md`);
});
test('uploads a file per submodule (unscoped package)', async () => {
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const forPackage = require('jsii-docgen').Documentation.forPackage;
    forPackage.mockImplementation(async (target) => {
        return new MockDocumentation(target);
    });
    // GIVEN
    const packageName = 'package-name';
    const packageVersion = '1.2.3-dev.4';
    const s3Event = {
        Records: [{
                awsRegion: 'bemuda-triangle-1',
                s3: {
                    bucket: {
                        name: 'dummy-bucket',
                    },
                    object: {
                        key: `${constants.STORAGE_KEY_PREFIX}${packageName}/v${packageVersion}${constants.ASSEMBLY_KEY_SUFFIX}`,
                        versionId: 'VersionId',
                    },
                },
            }],
    };
    const event = {
        Records: [{
                Sns: {
                    Message: JSON.stringify(s3Event),
                },
            }],
    };
    const assembly = {
        targets: { python: {} },
        submodules: { '@scope/package-name.sub1': {}, '@scope/package-name.sub2': {} },
    };
    // mock the assembly request
    mockFetchAssembly(assembly);
    // mock the file uploads
    mockPutDocs('/docs-typescript.md', '/docs-sub1-typescript.md', '/docs-sub2-typescript.md');
    const created = await transliterator_lambda_1.handler(event, {});
    expect(created.map(({ key }) => key)).toEqual([
        `data/${packageName}/v${packageVersion}/docs-typescript.md`,
        `data/${packageName}/v${packageVersion}/docs-sub1-typescript.md`,
        `data/${packageName}/v${packageVersion}/docs-sub2-typescript.md`,
    ]);
});
class MockDocumentation {
    constructor(target) {
        this.target = target;
    }
    render() {
        return {
            render: () => `docs for ${this.target}`,
        };
    }
}
function mockFetchAssembly(response) {
    AWSMock.mock('S3', 'getObject', (request, callback) => {
        if (request.Key.endsWith(constants.ASSEMBLY_KEY_SUFFIX)) {
            callback(null, {
                Body: JSON.stringify(response),
            });
        }
        else {
            throw new Error(`Unexpected GET request: ${request.Key}`);
        }
    });
}
function mockPutDocs(...suffixes) {
    AWSMock.mock('S3', 'putObject', (request, callback) => {
        if (suffixes.filter(s => request.Key.endsWith(s)).length > 0) {
            callback(null, { VersionId: `versionId-${request.Key}` });
        }
        else {
            throw new Error(`Unexpected PUT request: ${request.Key}`);
        }
    });
}
//# sourceMappingURL=data:application/json;base64,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