"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const transliterator_1 = require("../../../backend/transliterator");
const repository_1 = require("../../../codeartifact/repository");
const monitoring_1 = require("../../../monitoring");
test('basic use', () => {
    // GIVEN
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'TestStack');
    const bucket = new aws_s3_1.Bucket(stack, 'Bucket');
    const monitoring = new monitoring_1.Monitoring(stack, 'Monitoring', {
        alarmActions: { highSeverity: 'high-sev', normalSeverity: 'normal-sev' },
        dashboardName: 'monitoring',
    });
    // WHEN
    new transliterator_1.Transliterator(stack, 'Transliterator', {
        bucket,
        monitoring,
    });
    // THEN
    expect(app.synth().getStackByName(stack.stackName).template).toMatchSnapshot({
        Outputs: expect.anything(),
        Parameters: expect.anything(),
        Resources: ignoreResources(stack, bucket, monitoring),
    });
});
test('CodeArtifact repository', () => {
    // GIVEN
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'TestStack');
    const bucket = new aws_s3_1.Bucket(stack, 'Bucket');
    const monitoring = new monitoring_1.Monitoring(stack, 'Monitoring', {
        alarmActions: { highSeverity: 'high-sev', normalSeverity: 'normal-sev' },
        dashboardName: 'monitoring',
    });
    const codeArtifact = new repository_1.Repository(stack, 'CodeArtifact');
    // WHEN
    new transliterator_1.Transliterator(stack, 'Transliterator', {
        bucket,
        codeArtifact,
        monitoring,
    });
    // THEN
    expect(stack).toHaveResourceLike('AWS::Lambda::Function', {
        Environment: {
            Variables: stack.resolve({
                CODE_ARTIFACT_DOMAIN_NAME: codeArtifact.repositoryDomainName,
                CODE_ARTIFACT_DOMAIN_OWNER: codeArtifact.repositoryDomainOwner,
                CODE_ARTIFACT_REPOSITORY_ENDPOINT: codeArtifact.repositoryNpmEndpoint,
            }),
        },
    });
    expect(app.synth().getStackByName(stack.stackName).template).toMatchSnapshot({
        Outputs: expect.anything(),
        Parameters: expect.anything(),
        Resources: ignoreResources(stack, bucket, monitoring, codeArtifact),
    });
});
test('VPC Endpoints', () => {
    // GIVEN
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'TestStack');
    const bucket = new aws_s3_1.Bucket(stack, 'Bucket');
    const monitoring = new monitoring_1.Monitoring(stack, 'Monitoring', {
        alarmActions: { highSeverity: 'high-sev', normalSeverity: 'normal-sev' },
        dashboardName: 'monitoring',
    });
    const vpc = new aws_ec2_1.Vpc(stack, 'VPC', { subnetConfiguration: [{ name: 'Isolated', subnetType: aws_ec2_1.SubnetType.ISOLATED }] });
    const codeArtifactApi = vpc.addInterfaceEndpoint('CodeArtifact.API', {
        service: new aws_ec2_1.InterfaceVpcEndpointAwsService('codeartifact.api'),
    });
    const codeArtifact = vpc.addInterfaceEndpoint('CodeArtifact.Repo', {
        service: new aws_ec2_1.InterfaceVpcEndpointAwsService('codeartifact.repositories'),
    });
    const s3 = vpc.addGatewayEndpoint('S3', {
        service: aws_ec2_1.GatewayVpcEndpointAwsService.S3,
    });
    // WHEN
    new transliterator_1.Transliterator(stack, 'Transliterator', {
        bucket,
        monitoring,
        vpc,
        vpcEndpoints: { codeArtifactApi, codeArtifact, s3 },
    });
    // THEN
    expect(stack).toHaveResourceLike('AWS::Lambda::Function', {
        Environment: {
            Variables: stack.resolve({
                CODE_ARTIFACT_API_ENDPOINT: core_1.Fn.select(1, core_1.Fn.split(':', core_1.Fn.select(0, codeArtifactApi.vpcEndpointDnsEntries))),
            }),
        },
    });
    expect(app.synth().getStackByName(stack.stackName).template).toMatchSnapshot({
        Outputs: expect.anything(),
        Parameters: expect.anything(),
        Resources: ignoreResources(stack, bucket, monitoring, vpc),
    });
});
test('VPC Endpoints and CodeArtifact repository', () => {
    // GIVEN
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'TestStack');
    const bucket = new aws_s3_1.Bucket(stack, 'Bucket');
    const repository = new repository_1.Repository(stack, 'CodeArtifact');
    const monitoring = new monitoring_1.Monitoring(stack, 'Monitoring', {
        alarmActions: { highSeverity: 'high-sev', normalSeverity: 'normal-sev' },
        dashboardName: 'monitoring',
    });
    const vpc = new aws_ec2_1.Vpc(stack, 'VPC', { subnetConfiguration: [{ name: 'Isolated', subnetType: aws_ec2_1.SubnetType.ISOLATED }] });
    const codeArtifactApi = vpc.addInterfaceEndpoint('CodeArtifact.API', {
        service: new aws_ec2_1.InterfaceVpcEndpointAwsService('codeartifact.api'),
    });
    const codeArtifact = vpc.addInterfaceEndpoint('CodeArtifact.Repo', {
        service: new aws_ec2_1.InterfaceVpcEndpointAwsService('codeartifact.repositories'),
    });
    const s3 = vpc.addGatewayEndpoint('S3', {
        service: aws_ec2_1.GatewayVpcEndpointAwsService.S3,
    });
    // WHEN
    new transliterator_1.Transliterator(stack, 'Transliterator', {
        bucket,
        codeArtifact: repository,
        monitoring,
        vpc,
        vpcEndpoints: { codeArtifactApi, codeArtifact, s3 },
    });
    // THEN
    expect(stack).toHaveResourceLike('AWS::Lambda::Function', {
        Environment: {
            Variables: stack.resolve({
                CODE_ARTIFACT_DOMAIN_NAME: repository.repositoryDomainName,
                CODE_ARTIFACT_DOMAIN_OWNER: repository.repositoryDomainOwner,
                CODE_ARTIFACT_REPOSITORY_ENDPOINT: repository.repositoryNpmEndpoint,
                CODE_ARTIFACT_API_ENDPOINT: core_1.Fn.select(1, core_1.Fn.split(':', core_1.Fn.select(0, codeArtifactApi.vpcEndpointDnsEntries))),
            }),
        },
    });
    expect(app.synth().getStackByName(stack.stackName).template).toMatchSnapshot({
        Outputs: expect.anything(),
        Parameters: expect.anything(),
        Resources: ignoreResources(stack, bucket, repository, monitoring, vpc),
    });
});
/**
 * Creates a property matcher that expects "anything" for all CfnResource
 * logicalIds found within the provided constructs.
 */
function ignoreResources(stack, ...constructs) {
    const result = {};
    for (const resource of resourcesIn(...constructs)) {
        result[stack.resolve(resource.logicalId)] = expect.anything();
    }
    return result;
    function* resourcesIn(...scopes) {
        for (const scope of scopes) {
            if (core_1.CfnResource.isCfnResource(scope)) {
                yield scope;
            }
            yield* resourcesIn(...scope.node.children);
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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