"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const crypto_1 = require("crypto");
const stream_1 = require("stream");
const zip = require("zlib");
const AWS = require("aws-sdk");
const AWSMock = require("aws-sdk-mock");
const tar = require("tar-stream");
const catalog_builder_lambda_1 = require("../../../backend/catalog-builder/catalog-builder.lambda");
const aws = require("../../../backend/shared/aws.lambda-shared");
const constants = require("../../../backend/shared/constants");
let mockBucketName;
beforeEach((done) => {
    process.env.BUCKET_NAME = mockBucketName = crypto_1.randomBytes(16).toString('base64');
    AWSMock.setSDKInstance(AWS);
    done();
});
afterEach((done) => {
    AWSMock.restore();
    aws.reset();
    process.env.BUCKET_NAME = mockBucketName = undefined;
    done();
});
test('no indexed packages', () => {
    // GIVEN
    AWSMock.mock('S3', 'getObject', (req, cb) => {
        try {
            expect(req.Bucket).toBe(mockBucketName);
            expect(req.Key).toBe(constants.CATALOG_KEY);
        }
        catch (e) {
            return cb(e);
        }
        return cb(new NoSuchKeyError());
    });
    AWSMock.mock('S3', 'listObjectsV2', (req, cb) => {
        try {
            expect(req.Bucket).toBe(mockBucketName);
            expect(req.Prefix).toBe(constants.STORAGE_KEY_PREFIX);
            expect(req.ContinuationToken).toBeUndefined();
        }
        catch (e) {
            return cb(e);
        }
        return cb(null, {});
    });
    const mockPutObjectResult = {};
    AWSMock.mock('S3', 'putObject', (req, cb) => {
        var _a, _b;
        try {
            expect(req.Bucket).toBe(mockBucketName);
            expect(req.Key).toBe(constants.CATALOG_KEY);
            expect(req.ContentType).toBe('text/json');
            const body = JSON.parse((_b = (_a = req.Body) === null || _a === void 0 ? void 0 : _a.toString('utf-8')) !== null && _b !== void 0 ? _b : 'null');
            expect(body.packages).toEqual([]);
            expect(Date.parse(body.updatedAt)).toBeDefined();
        }
        catch (e) {
            return cb(e);
        }
        return cb(null, mockPutObjectResult);
    });
    // WHEN
    const result = catalog_builder_lambda_1.handler({ Records: [] }, { /* context */});
    // THEN
    return expect(result).resolves.toBe(mockPutObjectResult);
});
test('initial build', () => {
    // GIVEN
    const npmMetadata = { date: 'Thu, 17 Jun 2021 01:52:04 GMT' };
    AWSMock.mock('S3', 'getObject', (req, cb) => {
        try {
            expect(req.Bucket).toBe(mockBucketName);
        }
        catch (e) {
            return cb(e);
        }
        if (req.Key.endsWith(constants.METADATA_KEY_SUFFIX)) {
            return cb(null, { Body: JSON.stringify(npmMetadata) });
        }
        const matches = new RegExp(`^${constants.STORAGE_KEY_PREFIX}((?:@[^/]+/)?[^/]+)/v([^/]+)/.*$`).exec(req.Key);
        if (matches != null) {
            mockNpmPackage(matches[1], matches[2]).then((pack) => cb(null, { Body: pack }), cb);
        }
        else {
            return cb(new NoSuchKeyError());
        }
    });
    // this is the suffix that triggers the catalog builder.
    const docsSuffix = constants.DOCS_KEY_SUFFIX_TYPESCRIPT;
    const mockFirstPage = [
        { Key: `${constants.STORAGE_KEY_PREFIX}@scope/package/v1.2.3${constants.ASSEMBLY_KEY_SUFFIX}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}@scope/package/v1.2.3${constants.PACKAGE_KEY_SUFFIX}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}@scope/package/v1.2.3${docsSuffix}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}name/v1.2.3${constants.ASSEMBLY_KEY_SUFFIX}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}name/v1.2.3${constants.PACKAGE_KEY_SUFFIX}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}name/v1.2.3${docsSuffix}` },
    ];
    const mockSecondPage = [
        { Key: `${constants.STORAGE_KEY_PREFIX}@scope/package/v1.0.0${constants.ASSEMBLY_KEY_SUFFIX}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}@scope/package/v1.0.0${constants.PACKAGE_KEY_SUFFIX}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}@scope/package/v1.0.0${docsSuffix}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}name/v2.0.0-pre${constants.ASSEMBLY_KEY_SUFFIX}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}name/v2.0.0-pre${constants.PACKAGE_KEY_SUFFIX}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}name/v2.0.0-pre${docsSuffix}` },
    ];
    AWSMock.mock('S3', 'listObjectsV2', (req, cb) => {
        try {
            expect(req.Bucket).toBe(mockBucketName);
            expect(req.Prefix).toBe(constants.STORAGE_KEY_PREFIX);
        }
        catch (e) {
            return cb(e);
        }
        if (req.ContinuationToken == null) {
            return cb(null, { Contents: mockFirstPage, NextContinuationToken: 'next' });
        }
        try {
            expect(req.ContinuationToken).toBe('next');
        }
        catch (e) {
            return cb(e);
        }
        return cb(null, { Contents: mockSecondPage });
    });
    const mockPutObjectResult = {};
    AWSMock.mock('S3', 'putObject', (req, cb) => {
        var _a, _b;
        try {
            expect(req.Bucket).toBe(mockBucketName);
            expect(req.Key).toBe(constants.CATALOG_KEY);
            expect(req.ContentType).toBe('text/json');
            expect(req.Metadata).toHaveProperty('Package-Count', '3');
            const body = JSON.parse((_b = (_a = req.Body) === null || _a === void 0 ? void 0 : _a.toString('utf-8')) !== null && _b !== void 0 ? _b : 'null');
            expect(body.packages).toEqual([
                {
                    description: 'Package @scope/package, version 1.2.3',
                    languages: { foo: 'bar' },
                    major: 1,
                    metadata: npmMetadata,
                    name: '@scope/package',
                    version: '1.2.3',
                },
                {
                    description: 'Package name, version 1.2.3',
                    languages: { foo: 'bar' },
                    major: 1,
                    metadata: npmMetadata,
                    name: 'name',
                    version: '1.2.3',
                },
                {
                    description: 'Package name, version 2.0.0-pre',
                    languages: { foo: 'bar' },
                    major: 2,
                    metadata: npmMetadata,
                    name: 'name',
                    version: '2.0.0-pre',
                },
            ]);
            expect(Date.parse(body.updatedAt)).toBeDefined();
        }
        catch (e) {
            return cb(e);
        }
        return cb(null, mockPutObjectResult);
    });
    // WHEN
    const result = catalog_builder_lambda_1.handler({ Records: [] }, { /* context */});
    // THEN
    return expect(result).resolves.toBe(mockPutObjectResult);
});
test('incremental build', () => {
    const npmMetadata = { date: 'Thu, 17 Jun 2021 01:52:04 GMT' };
    // GIVEN
    AWSMock.mock('S3', 'getObject', (req, cb) => {
        try {
            expect(req.Bucket).toBe(mockBucketName);
        }
        catch (e) {
            return cb(e);
        }
        if (req.Key.endsWith(constants.METADATA_KEY_SUFFIX)) {
            return cb(null, { Body: JSON.stringify(npmMetadata) });
        }
        const matches = new RegExp(`^${constants.STORAGE_KEY_PREFIX}((?:@[^/]+/)?[^/]+)/v([^/]+)/.*$`).exec(req.Key);
        if (matches != null) {
            mockNpmPackage(matches[1], matches[2]).then((pack) => cb(null, { Body: pack }), cb);
        }
        else if (req.Key === constants.CATALOG_KEY) {
            return cb(null, {
                Body: JSON.stringify({
                    packages: [
                        {
                            description: 'Package @scope/package, version 2.3.4',
                            languages: { foo: 'bar' },
                            major: 2,
                            metadata: npmMetadata,
                            name: '@scope/package',
                            version: '2.3.4',
                        },
                        {
                            description: 'Package name, version 1.0.0',
                            languages: { foo: 'bar' },
                            major: 1,
                            metadata: npmMetadata,
                            name: 'name',
                            version: '1.0.0',
                        },
                        {
                            description: 'Package name, version 2.0.0-pre.10',
                            languages: { foo: 'bar' },
                            major: 2,
                            metadata: npmMetadata,
                            name: 'name',
                            version: '2.0.0-pre.10',
                        },
                    ],
                    updatedAt: new Date().toISOString(),
                }, null, 2),
            });
        }
        else {
            return cb(new NoSuchKeyError());
        }
    });
    function createRecord(key) {
        return { s3: { object: { key } } };
    }
    const event = {
        Records: [
            createRecord(`${constants.STORAGE_KEY_PREFIX}@scope/package/v1.2.3${constants.PACKAGE_KEY_SUFFIX}`),
            createRecord(`${constants.STORAGE_KEY_PREFIX}name/v1.2.3${constants.PACKAGE_KEY_SUFFIX}`),
            createRecord(`${constants.STORAGE_KEY_PREFIX}@scope/package/v2.0.5${constants.PACKAGE_KEY_SUFFIX}`),
            createRecord(`${constants.STORAGE_KEY_PREFIX}name/v2.0.0-pre.1${constants.PACKAGE_KEY_SUFFIX}`),
        ],
    };
    const mockPutObjectResult = {};
    AWSMock.mock('S3', 'putObject', (req, cb) => {
        var _a, _b;
        try {
            expect(req.Bucket).toBe(mockBucketName);
            expect(req.Key).toBe(constants.CATALOG_KEY);
            expect(req.ContentType).toBe('text/json');
            expect(req.Metadata).toHaveProperty('Package-Count', '4');
            const body = JSON.parse((_b = (_a = req.Body) === null || _a === void 0 ? void 0 : _a.toString('utf-8')) !== null && _b !== void 0 ? _b : 'null');
            expect(body.packages).toEqual([
                {
                    description: 'Package @scope/package, version 2.3.4',
                    languages: { foo: 'bar' },
                    major: 2,
                    metadata: npmMetadata,
                    name: '@scope/package',
                    version: '2.3.4',
                },
                {
                    description: 'Package @scope/package, version 1.2.3',
                    languages: { foo: 'bar' },
                    major: 1,
                    metadata: npmMetadata,
                    name: '@scope/package',
                    version: '1.2.3',
                },
                {
                    description: 'Package name, version 1.2.3',
                    languages: { foo: 'bar' },
                    major: 1,
                    metadata: npmMetadata,
                    name: 'name',
                    version: '1.2.3',
                },
                {
                    description: 'Package name, version 2.0.0-pre.10',
                    languages: { foo: 'bar' },
                    major: 2,
                    metadata: npmMetadata,
                    name: 'name',
                    version: '2.0.0-pre.10',
                },
            ]);
            expect(Date.parse(body.updatedAt)).toBeDefined();
        }
        catch (e) {
            return cb(e);
        }
        return cb(null, mockPutObjectResult);
    });
    // WHEN
    const result = catalog_builder_lambda_1.handler(event, { /* context */});
    // THEN
    return expect(result).resolves.toBe(mockPutObjectResult);
});
class NoSuchKeyError extends Error {
    constructor() {
        super(...arguments);
        this.code = 'NoSuchKey';
        this.time = new Date();
    }
}
function mockNpmPackage(name, version) {
    const packageJson = {
        name,
        version,
        description: `Package ${name}, version ${version}`,
        jsii: {
            targets: { foo: 'bar' },
        },
    };
    const tarball = tar.pack();
    tarball.entry({ name: 'package/ignore-me.txt' }, 'Ignore Me!');
    tarball.entry({ name: 'package/package.json' }, JSON.stringify(packageJson, null, 2));
    tarball.finalize();
    const gzip = zip.createGzip();
    tarball.pipe(gzip);
    const passthrough = new stream_1.PassThrough();
    gzip.pipe(passthrough);
    return new Promise((ok) => {
        const chunks = new Array();
        passthrough.on('data', (chunk) => chunks.push(Buffer.from(chunk)));
        passthrough.once('end', () => {
            ok(Buffer.concat(chunks));
        });
    });
}
//# sourceMappingURL=data:application/json;base64,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