# import scimap as sm
from .. import tools as tl
from .. import preprocessing as pp
from .. import plotting as pl
from .. import helpers as hl

import sys
import argparse
import pathlib


def main(argv=sys.argv):

    parser = argparse.ArgumentParser()

    parser.add_argument(
        'csv',
        help='single-cell quantification table (CSV file) generated by mcmicro',
    )
    parser.add_argument(
        '-o', '--output', default='.',
        help='output directory'
    )
    clustering_methods = ['all', 'spatial', 'kmeans', 'phenograph', 'leiden', 'pass']
    parser.add_argument(
        '--clustering-method', default=['all'], choices=clustering_methods, nargs='+', 
        help='choice of clustering algorithms, "pass": do not run any clustering methods; "all": run all clustering methods; default: "all"'
    )

    args = parser.parse_args(argv[1:])

    _output_dir = pathlib.Path(args.output)
    mcmicro_csv_path = args.csv
    methods = set(args.clustering_method)

    assert '.csv' in mcmicro_csv_path, 'input file must be a csv file'
    
    if 'all' in methods:
        methods = clustering_methods[1:-1]

    if 'pass' in methods:
        pp.mcmicro_to_scimap(
            image_path = mcmicro_csv_path, 
            output_dir = str(_output_dir)
        )
        return

    for method in methods:

        output_dir = _output_dir / method
        
        pp.mcmicro_to_scimap(
            image_path = mcmicro_csv_path, 
            output_dir = str(output_dir)
        )

        adata_path = pathlib.Path(output_dir) / f'{pathlib.Path(mcmicro_csv_path).stem}.h5ad'

        if method == 'spatial':
            # Spatial clustering
            tl.spatial_expression(
                adata = str(adata_path), 
                output_dir = output_dir
            )
            tl.spatial_cluster(
                adata = str(adata_path), 
                df_name = "spatial_expression",
                output_dir = output_dir
            )
        
        else:
            # Expression clustering
            tl.cluster(
                adata = str(adata_path), 
                method = method,
                output_dir = output_dir
            )

            # Expression clustering plots
            pl.cluster_plots(
                adata = str(adata_path), 
                group_by  = method,
                output_dir = output_dir
            ) 
import textwrap

def merge(argv=sys.argv):
    
    parser = argparse.ArgumentParser()

    parser.add_argument(
        'directory',
        help='recursively search for .h5ad files to merge',
    )
    parser.add_argument(
        '-o', '--output', default='.',
        help='output directory'
    )
    parser.add_argument(
        '-d', '--delete-merged', default=False, action='store_true',
        help='delete found input files after merging; default: False'
    )

    args = parser.parse_args(argv[1:])

    input_dir = pathlib.Path(args.directory)
    output_dir = pathlib.Path(args.output)
    delete_after = args.delete_merged

    input_files = sorted(input_dir.rglob('*.h5ad'))

    print_text = '''
        Merging:

        {}

        Writing:

        {}
    '''
    print_text = textwrap.dedent(print_text).format(
        "\n".join([str(p) for p in input_files]),
        str(output_dir / 'combined.h5ad')
    )
    print(textwrap.indent(print_text, '    '))
    # Merge data
    hl.merge_adata_obs(
        adata = [
            str(p)
            for p in input_files
        ],
        output_dir = output_dir
    ) 

    if delete_after == True:
        filtered_input_files = sorted(filter(
            lambda x: x.name != 'combined.h5ad',
            input_files
        ))
        print(
            textwrap.indent(
                'Deleting:\n\n' + "\n".join([str(f) for f in filtered_input_files]),
                '    '
            )
        )
        for f in filtered_input_files:
            f.unlink()
    
    return 0