"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SimpleSynthAction = void 0;
const crypto = require("crypto");
const path = require("path");
const codebuild = require("../../../aws-codebuild"); // Automatically re-written from '@aws-cdk/aws-codebuild'
const codepipeline = require("../../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const codepipeline_actions = require("../../../aws-codepipeline-actions"); // Automatically re-written from '@aws-cdk/aws-codepipeline-actions'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const construct_internals_1 = require("../private/construct-internals");
const _util_1 = require("./_util");
/**
 * (experimental) A standard synth with a generated buildspec.
 *
 * @experimental
 */
class SimpleSynthAction {
    /**
     * @experimental
     */
    constructor(props) {
        var _a, _b, _c;
        this.props = props;
        // A number of actionProperties get read before bind() is even called (so before we
        // have made the Project and can construct the actual CodeBuildAction)
        //
        // - actionName
        // - resource
        // - region
        // - category
        // - role
        // - owner
        this._actionProperties = {
            actionName: (_a = props.actionName) !== null && _a !== void 0 ? _a : 'Synth',
            category: codepipeline.ActionCategory.BUILD,
            provider: 'CodeBuild',
            artifactBounds: { minInputs: 0, maxInputs: 5, minOutputs: 0, maxOutputs: 5 },
            inputs: [props.sourceArtifact],
            outputs: [props.cloudAssemblyArtifact, ...((_b = props.additionalArtifacts) !== null && _b !== void 0 ? _b : []).map(a => a.artifact)],
        };
        if (this.props.installCommand && this.props.installCommands) {
            throw new Error('Pass either \'installCommand\' or \'installCommands\', but not both');
        }
        if (this.props.buildCommand && this.props.buildCommands) {
            throw new Error('Pass either \'buildCommand\' or \'buildCommands\', but not both');
        }
        const addls = (_c = props.additionalArtifacts) !== null && _c !== void 0 ? _c : [];
        if (Object.keys(addls).length > 0) {
            if (!props.cloudAssemblyArtifact.artifactName) {
                throw new Error('You must give all output artifacts, including the \'cloudAssemblyArtifact\', names when using \'additionalArtifacts\'');
            }
            for (const addl of addls) {
                if (!addl.artifact.artifactName) {
                    throw new Error('You must give all output artifacts passed to SimpleSynthAction names when using \'additionalArtifacts\'');
                }
            }
        }
    }
    /**
     * (experimental) Create a standard NPM synth action.
     *
     * Uses `npm ci` to install dependencies and `npx cdk synth` to synthesize.
     *
     * If you need a build step, add `buildCommand: 'npm run build'`.
     *
     * @experimental
     */
    static standardNpmSynth(options) {
        var _a, _b;
        return new SimpleSynthAction({
            ...options,
            installCommand: (_a = options.installCommand) !== null && _a !== void 0 ? _a : 'npm ci',
            synthCommand: (_b = options.synthCommand) !== null && _b !== void 0 ? _b : 'npx cdk synth',
            vpc: options.vpc,
            subnetSelection: options.subnetSelection,
        });
    }
    /**
     * (experimental) Create a standard Yarn synth action.
     *
     * Uses `yarn install --frozen-lockfile` to install dependencies and `npx cdk synth` to synthesize.
     *
     * If you need a build step, add `buildCommand: 'yarn build'`.
     *
     * @experimental
     */
    static standardYarnSynth(options) {
        var _a, _b;
        return new SimpleSynthAction({
            ...options,
            installCommand: (_a = options.installCommand) !== null && _a !== void 0 ? _a : 'yarn install --frozen-lockfile',
            synthCommand: (_b = options.synthCommand) !== null && _b !== void 0 ? _b : 'npx cdk synth',
            vpc: options.vpc,
            subnetSelection: options.subnetSelection,
        });
    }
    /**
     * (experimental) Exists to implement IAction.
     *
     * @experimental
     */
    get actionProperties() {
        return this._actionProperties;
    }
    /**
     * (experimental) Project generated to run the synth command.
     *
     * @experimental
     */
    get project() {
        if (!this._project) {
            throw new Error('Project becomes available after SimpleSynthAction has been bound to a stage');
        }
        return this._project;
    }
    /**
     * (experimental) Exists to implement IAction.
     *
     * @experimental
     */
    bind(scope, stage, options) {
        var _a, _b, _c, _d;
        const buildCommands = (_a = this.props.buildCommands) !== null && _a !== void 0 ? _a : [this.props.buildCommand];
        const installCommands = (_b = this.props.installCommands) !== null && _b !== void 0 ? _b : [this.props.installCommand];
        const testCommands = (_c = this.props.testCommands) !== null && _c !== void 0 ? _c : [];
        const synthCommand = this.props.synthCommand;
        const buildSpec = codebuild.BuildSpec.fromObject({
            version: '0.2',
            phases: {
                pre_build: {
                    commands: _util_1.filterEmpty([
                        this.props.subdirectory ? `cd ${this.props.subdirectory}` : '',
                        ...installCommands,
                    ]),
                },
                build: {
                    commands: _util_1.filterEmpty([
                        ...buildCommands,
                        ...testCommands,
                        synthCommand,
                    ]),
                },
            },
            artifacts: renderArtifacts(this),
        });
        const environment = { buildImage: codebuild.LinuxBuildImage.STANDARD_4_0, ...this.props.environment };
        const environmentVariables = {
            ..._util_1.copyEnvironmentVariables(...this.props.copyEnvironmentVariables || []),
            ...this.props.environmentVariables,
        };
        // A hash over the values that make the CodeBuild Project unique (and necessary
        // to restart the pipeline if one of them changes). projectName is not necessary to include
        // here because the pipeline will definitely restart if projectName changes.
        // (Resolve tokens)
        const projectConfigHash = hash(core_1.Stack.of(scope).resolve({
            environment: serializeBuildEnvironment(environment),
            buildSpecString: buildSpec.toBuildSpec(),
            environmentVariables,
        }));
        const project = new codebuild.PipelineProject(scope, 'CdkBuildProject', {
            projectName: this.props.projectName,
            environment,
            vpc: this.props.vpc,
            subnetSelection: this.props.subnetSelection,
            buildSpec,
            environmentVariables,
        });
        if (this.props.rolePolicyStatements !== undefined) {
            this.props.rolePolicyStatements.forEach(policyStatement => {
                project.addToRolePolicy(policyStatement);
            });
        }
        this._project = project;
        this._action = new codepipeline_actions.CodeBuildAction({
            actionName: this.actionProperties.actionName,
            input: this.props.sourceArtifact,
            outputs: [this.props.cloudAssemblyArtifact, ...((_d = this.props.additionalArtifacts) !== null && _d !== void 0 ? _d : []).map(a => a.artifact)],
            // Inclusion of the hash here will lead to the pipeline structure for any changes
            // made the config of the underlying CodeBuild Project.
            // Hence, the pipeline will be restarted. This is necessary if the users
            // adds (for example) build or test commands to the buildspec.
            environmentVariables: {
                _PROJECT_CONFIG_HASH: { value: projectConfigHash },
            },
            project,
        });
        this._actionProperties = this._action.actionProperties;
        return this._action.bind(scope, stage, options);
        function renderArtifacts(self) {
            var _a;
            // save the generated files in the output artifact
            // This part of the buildspec has to look completely different depending on whether we're
            // using secondary artifacts or not.
            const cloudAsmArtifactSpec = {
                'base-directory': path.join((_a = self.props.subdirectory) !== null && _a !== void 0 ? _a : '.', construct_internals_1.cloudAssemblyBuildSpecDir(scope)),
                'files': '**/*',
            };
            if (self.props.additionalArtifacts) {
                const secondary = {};
                if (!self.props.cloudAssemblyArtifact.artifactName) {
                    throw new Error('When using additional output artifacts, you must also name the CloudAssembly artifact');
                }
                secondary[self.props.cloudAssemblyArtifact.artifactName] = cloudAsmArtifactSpec;
                self.props.additionalArtifacts.forEach((art) => {
                    var _a;
                    if (!art.artifact.artifactName) {
                        throw new Error('You must give the output artifact a name');
                    }
                    secondary[art.artifact.artifactName] = {
                        'base-directory': path.join((_a = self.props.subdirectory) !== null && _a !== void 0 ? _a : '.', art.directory),
                        'files': '**/*',
                    };
                });
                return { 'secondary-artifacts': secondary };
            }
            return cloudAsmArtifactSpec;
        }
    }
    /**
     * (experimental) The CodeBuild Project's principal.
     *
     * @experimental
     */
    get grantPrincipal() {
        return this.project.grantPrincipal;
    }
    /**
     * (experimental) Exists to implement IAction.
     *
     * @experimental
     */
    onStateChange(name, target, options) {
        if (!this._action) {
            throw new Error('Need bind() first');
        }
        return this._action.onStateChange(name, target, options);
    }
}
exports.SimpleSynthAction = SimpleSynthAction;
function hash(obj) {
    const d = crypto.createHash('sha256');
    d.update(JSON.stringify(obj));
    return d.digest('hex');
}
/**
 * Serialize a build environment to data (get rid of constructs & objects), so we can JSON.stringify it
 */
function serializeBuildEnvironment(env) {
    var _a, _b, _c, _d, _e;
    return {
        privileged: env.privileged,
        environmentVariables: env.environmentVariables,
        type: (_a = env.buildImage) === null || _a === void 0 ? void 0 : _a.type,
        imageId: (_b = env.buildImage) === null || _b === void 0 ? void 0 : _b.imageId,
        computeType: env.computeType,
        imagePullPrincipalType: (_c = env.buildImage) === null || _c === void 0 ? void 0 : _c.imagePullPrincipalType,
        secretsManagerArn: (_e = (_d = env.buildImage) === null || _d === void 0 ? void 0 : _d.secretsManagerCredentials) === null || _e === void 0 ? void 0 : _e.secretArn,
    };
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic2ltcGxlLXN5bnRoLWFjdGlvbi5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInNpbXBsZS1zeW50aC1hY3Rpb24udHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsaUNBQWlDO0FBQ2pDLDZCQUE2QjtBQUM3QixvREFBb0QsQ0FBQyx5REFBeUQ7QUFDOUcsMERBQTBELENBQUMsNERBQTREO0FBQ3ZILDBFQUEwRSxDQUFDLG9FQUFvRTtBQUkvSSx3Q0FBaUQsQ0FBQyxnREFBZ0Q7QUFDbEcsd0VBQTJFO0FBQzNFLG1DQUFnRTs7Ozs7O0FBZ0toRSxNQUFhLGlCQUFpQjs7OztJQW9DMUIsWUFBNkIsS0FBNkI7O1FBQTdCLFVBQUssR0FBTCxLQUFLLENBQXdCO1FBQ3RELG1GQUFtRjtRQUNuRixzRUFBc0U7UUFDdEUsRUFBRTtRQUNGLGVBQWU7UUFDZixhQUFhO1FBQ2IsV0FBVztRQUNYLGFBQWE7UUFDYixTQUFTO1FBQ1QsVUFBVTtRQUNWLElBQUksQ0FBQyxpQkFBaUIsR0FBRztZQUNyQixVQUFVLFFBQUUsS0FBSyxDQUFDLFVBQVUsbUNBQUksT0FBTztZQUN2QyxRQUFRLEVBQUUsWUFBWSxDQUFDLGNBQWMsQ0FBQyxLQUFLO1lBQzNDLFFBQVEsRUFBRSxXQUFXO1lBQ3JCLGNBQWMsRUFBRSxFQUFFLFNBQVMsRUFBRSxDQUFDLEVBQUUsU0FBUyxFQUFFLENBQUMsRUFBRSxVQUFVLEVBQUUsQ0FBQyxFQUFFLFVBQVUsRUFBRSxDQUFDLEVBQUU7WUFDNUUsTUFBTSxFQUFFLENBQUMsS0FBSyxDQUFDLGNBQWMsQ0FBQztZQUM5QixPQUFPLEVBQUUsQ0FBQyxLQUFLLENBQUMscUJBQXFCLEVBQUUsR0FBRyxPQUFDLEtBQUssQ0FBQyxtQkFBbUIsbUNBQUksRUFBRSxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1NBQ3BHLENBQUM7UUFDRixJQUFJLElBQUksQ0FBQyxLQUFLLENBQUMsY0FBYyxJQUFJLElBQUksQ0FBQyxLQUFLLENBQUMsZUFBZSxFQUFFO1lBQ3pELE1BQU0sSUFBSSxLQUFLLENBQUMscUVBQXFFLENBQUMsQ0FBQztTQUMxRjtRQUNELElBQUksSUFBSSxDQUFDLEtBQUssQ0FBQyxZQUFZLElBQUksSUFBSSxDQUFDLEtBQUssQ0FBQyxhQUFhLEVBQUU7WUFDckQsTUFBTSxJQUFJLEtBQUssQ0FBQyxpRUFBaUUsQ0FBQyxDQUFDO1NBQ3RGO1FBQ0QsTUFBTSxLQUFLLFNBQUcsS0FBSyxDQUFDLG1CQUFtQixtQ0FBSSxFQUFFLENBQUM7UUFDOUMsSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7WUFDL0IsSUFBSSxDQUFDLEtBQUssQ0FBQyxxQkFBcUIsQ0FBQyxZQUFZLEVBQUU7Z0JBQzNDLE1BQU0sSUFBSSxLQUFLLENBQUMsdUhBQXVILENBQUMsQ0FBQzthQUM1STtZQUNELEtBQUssTUFBTSxJQUFJLElBQUksS0FBSyxFQUFFO2dCQUN0QixJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxZQUFZLEVBQUU7b0JBQzdCLE1BQU0sSUFBSSxLQUFLLENBQUMseUdBQXlHLENBQUMsQ0FBQztpQkFDOUg7YUFDSjtTQUNKO0lBQ0wsQ0FBQzs7Ozs7Ozs7OztJQS9ETSxNQUFNLENBQUMsZ0JBQWdCLENBQUMsT0FBZ0M7O1FBQzNELE9BQU8sSUFBSSxpQkFBaUIsQ0FBQztZQUN6QixHQUFHLE9BQU87WUFDVixjQUFjLFFBQUUsT0FBTyxDQUFDLGNBQWMsbUNBQUksUUFBUTtZQUNsRCxZQUFZLFFBQUUsT0FBTyxDQUFDLFlBQVksbUNBQUksZUFBZTtZQUNyRCxHQUFHLEVBQUUsT0FBTyxDQUFDLEdBQUc7WUFDaEIsZUFBZSxFQUFFLE9BQU8sQ0FBQyxlQUFlO1NBQzNDLENBQUMsQ0FBQztJQUNQLENBQUM7Ozs7Ozs7Ozs7SUFRTSxNQUFNLENBQUMsaUJBQWlCLENBQUMsT0FBaUM7O1FBQzdELE9BQU8sSUFBSSxpQkFBaUIsQ0FBQztZQUN6QixHQUFHLE9BQU87WUFDVixjQUFjLFFBQUUsT0FBTyxDQUFDLGNBQWMsbUNBQUksZ0NBQWdDO1lBQzFFLFlBQVksUUFBRSxPQUFPLENBQUMsWUFBWSxtQ0FBSSxlQUFlO1lBQ3JELEdBQUcsRUFBRSxPQUFPLENBQUMsR0FBRztZQUNoQixlQUFlLEVBQUUsT0FBTyxDQUFDLGVBQWU7U0FDM0MsQ0FBQyxDQUFDO0lBQ1AsQ0FBQzs7Ozs7O0lBMkNELElBQVcsZ0JBQWdCO1FBQ3ZCLE9BQU8sSUFBSSxDQUFDLGlCQUFpQixDQUFDO0lBQ2xDLENBQUM7Ozs7OztJQUlELElBQVcsT0FBTztRQUNkLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxFQUFFO1lBQ2hCLE1BQU0sSUFBSSxLQUFLLENBQUMsNkVBQTZFLENBQUMsQ0FBQztTQUNsRztRQUNELE9BQU8sSUFBSSxDQUFDLFFBQVEsQ0FBQztJQUN6QixDQUFDOzs7Ozs7SUFJTSxJQUFJLENBQUMsS0FBZ0IsRUFBRSxLQUEwQixFQUFFLE9BQXVDOztRQUM3RixNQUFNLGFBQWEsU0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLGFBQWEsbUNBQUksQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLFlBQVksQ0FBQyxDQUFDO1FBQzVFLE1BQU0sZUFBZSxTQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsZUFBZSxtQ0FBSSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsY0FBYyxDQUFDLENBQUM7UUFDbEYsTUFBTSxZQUFZLFNBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxZQUFZLG1DQUFJLEVBQUUsQ0FBQztRQUNuRCxNQUFNLFlBQVksR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLFlBQVksQ0FBQztRQUM3QyxNQUFNLFNBQVMsR0FBRyxTQUFTLENBQUMsU0FBUyxDQUFDLFVBQVUsQ0FBQztZQUM3QyxPQUFPLEVBQUUsS0FBSztZQUNkLE1BQU0sRUFBRTtnQkFDSixTQUFTLEVBQUU7b0JBQ1AsUUFBUSxFQUFFLG1CQUFXLENBQUM7d0JBQ2xCLElBQUksQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxNQUFNLElBQUksQ0FBQyxLQUFLLENBQUMsWUFBWSxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUU7d0JBQzlELEdBQUcsZUFBZTtxQkFDckIsQ0FBQztpQkFDTDtnQkFDRCxLQUFLLEVBQUU7b0JBQ0gsUUFBUSxFQUFFLG1CQUFXLENBQUM7d0JBQ2xCLEdBQUcsYUFBYTt3QkFDaEIsR0FBRyxZQUFZO3dCQUNmLFlBQVk7cUJBQ2YsQ0FBQztpQkFDTDthQUNKO1lBQ0QsU0FBUyxFQUFFLGVBQWUsQ0FBQyxJQUFJLENBQUM7U0FDbkMsQ0FBQyxDQUFDO1FBQ0gsTUFBTSxXQUFXLEdBQUcsRUFBRSxVQUFVLEVBQUUsU0FBUyxDQUFDLGVBQWUsQ0FBQyxZQUFZLEVBQUUsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLFdBQVcsRUFBRSxDQUFDO1FBQ3RHLE1BQU0sb0JBQW9CLEdBQUc7WUFDekIsR0FBRyxnQ0FBd0IsQ0FBQyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsd0JBQXdCLElBQUksRUFBRSxDQUFDO1lBQ3pFLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxvQkFBb0I7U0FDckMsQ0FBQztRQUNGLCtFQUErRTtRQUMvRSwyRkFBMkY7UUFDM0YsNEVBQTRFO1FBQzVFLG1CQUFtQjtRQUNuQixNQUFNLGlCQUFpQixHQUFHLElBQUksQ0FBQyxZQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDLE9BQU8sQ0FBQztZQUNuRCxXQUFXLEVBQUUseUJBQXlCLENBQUMsV0FBVyxDQUFDO1lBQ25ELGVBQWUsRUFBRSxTQUFTLENBQUMsV0FBVyxFQUFFO1lBQ3hDLG9CQUFvQjtTQUN2QixDQUFDLENBQUMsQ0FBQztRQUNKLE1BQU0sT0FBTyxHQUFHLElBQUksU0FBUyxDQUFDLGVBQWUsQ0FBQyxLQUFLLEVBQUUsaUJBQWlCLEVBQUU7WUFDcEUsV0FBVyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsV0FBVztZQUNuQyxXQUFXO1lBQ1gsR0FBRyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsR0FBRztZQUNuQixlQUFlLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxlQUFlO1lBQzNDLFNBQVM7WUFDVCxvQkFBb0I7U0FDdkIsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxJQUFJLENBQUMsS0FBSyxDQUFDLG9CQUFvQixLQUFLLFNBQVMsRUFBRTtZQUMvQyxJQUFJLENBQUMsS0FBSyxDQUFDLG9CQUFvQixDQUFDLE9BQU8sQ0FBQyxlQUFlLENBQUMsRUFBRTtnQkFDdEQsT0FBTyxDQUFDLGVBQWUsQ0FBQyxlQUFlLENBQUMsQ0FBQztZQUM3QyxDQUFDLENBQUMsQ0FBQztTQUNOO1FBQ0QsSUFBSSxDQUFDLFFBQVEsR0FBRyxPQUFPLENBQUM7UUFDeEIsSUFBSSxDQUFDLE9BQU8sR0FBRyxJQUFJLG9CQUFvQixDQUFDLGVBQWUsQ0FBQztZQUNwRCxVQUFVLEVBQUUsSUFBSSxDQUFDLGdCQUFnQixDQUFDLFVBQVU7WUFDNUMsS0FBSyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsY0FBYztZQUNoQyxPQUFPLEVBQUUsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLHFCQUFxQixFQUFFLEdBQUcsT0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLG1CQUFtQixtQ0FBSSxFQUFFLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsUUFBUSxDQUFDLENBQUM7WUFDM0csaUZBQWlGO1lBQ2pGLHVEQUF1RDtZQUN2RCx3RUFBd0U7WUFDeEUsOERBQThEO1lBQzlELG9CQUFvQixFQUFFO2dCQUNsQixvQkFBb0IsRUFBRSxFQUFFLEtBQUssRUFBRSxpQkFBaUIsRUFBRTthQUNyRDtZQUNELE9BQU87U0FDVixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsaUJBQWlCLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxnQkFBZ0IsQ0FBQztRQUN2RCxPQUFPLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxLQUFLLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFDaEQsU0FBUyxlQUFlLENBQUMsSUFBdUI7O1lBQzVDLGtEQUFrRDtZQUNsRCx5RkFBeUY7WUFDekYsb0NBQW9DO1lBQ3BDLE1BQU0sb0JBQW9CLEdBQUc7Z0JBQ3pCLGdCQUFnQixFQUFFLElBQUksQ0FBQyxJQUFJLE9BQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxZQUFZLG1DQUFJLEdBQUcsRUFBRSwrQ0FBeUIsQ0FBQyxLQUFLLENBQUMsQ0FBQztnQkFDN0YsT0FBTyxFQUFFLE1BQU07YUFDbEIsQ0FBQztZQUNGLElBQUksSUFBSSxDQUFDLEtBQUssQ0FBQyxtQkFBbUIsRUFBRTtnQkFDaEMsTUFBTSxTQUFTLEdBQXdCLEVBQUUsQ0FBQztnQkFDMUMsSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMscUJBQXFCLENBQUMsWUFBWSxFQUFFO29CQUNoRCxNQUFNLElBQUksS0FBSyxDQUFDLHVGQUF1RixDQUFDLENBQUM7aUJBQzVHO2dCQUNELFNBQVMsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLHFCQUFxQixDQUFDLFlBQVksQ0FBQyxHQUFHLG9CQUFvQixDQUFDO2dCQUNoRixJQUFJLENBQUMsS0FBSyxDQUFDLG1CQUFtQixDQUFDLE9BQU8sQ0FBQyxDQUFDLEdBQUcsRUFBRSxFQUFFOztvQkFDM0MsSUFBSSxDQUFDLEdBQUcsQ0FBQyxRQUFRLENBQUMsWUFBWSxFQUFFO3dCQUM1QixNQUFNLElBQUksS0FBSyxDQUFDLDBDQUEwQyxDQUFDLENBQUM7cUJBQy9EO29CQUNELFNBQVMsQ0FBQyxHQUFHLENBQUMsUUFBUSxDQUFDLFlBQVksQ0FBQyxHQUFHO3dCQUNuQyxnQkFBZ0IsRUFBRSxJQUFJLENBQUMsSUFBSSxPQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsWUFBWSxtQ0FBSSxHQUFHLEVBQUUsR0FBRyxDQUFDLFNBQVMsQ0FBQzt3QkFDMUUsT0FBTyxFQUFFLE1BQU07cUJBQ2xCLENBQUM7Z0JBQ04sQ0FBQyxDQUFDLENBQUM7Z0JBQ0gsT0FBTyxFQUFFLHFCQUFxQixFQUFFLFNBQVMsRUFBRSxDQUFDO2FBQy9DO1lBQ0QsT0FBTyxvQkFBb0IsQ0FBQztRQUNoQyxDQUFDO0lBQ0wsQ0FBQzs7Ozs7O0lBSUQsSUFBVyxjQUFjO1FBQ3JCLE9BQU8sSUFBSSxDQUFDLE9BQU8sQ0FBQyxjQUFjLENBQUM7SUFDdkMsQ0FBQzs7Ozs7O0lBSU0sYUFBYSxDQUFDLElBQVksRUFBRSxNQUEyQixFQUFFLE9BQTBCO1FBQ3RGLElBQUksQ0FBQyxJQUFJLENBQUMsT0FBTyxFQUFFO1lBQ2YsTUFBTSxJQUFJLEtBQUssQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO1NBQ3hDO1FBQ0QsT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFDLGFBQWEsQ0FBQyxJQUFJLEVBQUUsTUFBTSxFQUFFLE9BQU8sQ0FBQyxDQUFDO0lBQzdELENBQUM7Q0FDSjtBQXhNRCw4Q0F3TUM7QUF5REQsU0FBUyxJQUFJLENBQUksR0FBTTtJQUNuQixNQUFNLENBQUMsR0FBRyxNQUFNLENBQUMsVUFBVSxDQUFDLFFBQVEsQ0FBQyxDQUFDO0lBQ3RDLENBQUMsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDO0lBQzlCLE9BQU8sQ0FBQyxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQztBQUMzQixDQUFDO0FBQ0Q7O0dBRUc7QUFDSCxTQUFTLHlCQUF5QixDQUFDLEdBQStCOztJQUM5RCxPQUFPO1FBQ0gsVUFBVSxFQUFFLEdBQUcsQ0FBQyxVQUFVO1FBQzFCLG9CQUFvQixFQUFFLEdBQUcsQ0FBQyxvQkFBb0I7UUFDOUMsSUFBSSxRQUFFLEdBQUcsQ0FBQyxVQUFVLDBDQUFFLElBQUk7UUFDMUIsT0FBTyxRQUFFLEdBQUcsQ0FBQyxVQUFVLDBDQUFFLE9BQU87UUFDaEMsV0FBVyxFQUFFLEdBQUcsQ0FBQyxXQUFXO1FBQzVCLHNCQUFzQixRQUFFLEdBQUcsQ0FBQyxVQUFVLDBDQUFFLHNCQUFzQjtRQUM5RCxpQkFBaUIsY0FBRSxHQUFHLENBQUMsVUFBVSwwQ0FBRSx5QkFBeUIsMENBQUUsU0FBUztLQUMxRSxDQUFDO0FBQ04sQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGNyeXB0byBmcm9tICdjcnlwdG8nO1xuaW1wb3J0ICogYXMgcGF0aCBmcm9tICdwYXRoJztcbmltcG9ydCAqIGFzIGNvZGVidWlsZCBmcm9tIFwiLi4vLi4vLi4vYXdzLWNvZGVidWlsZFwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWNvZGVidWlsZCdcbmltcG9ydCAqIGFzIGNvZGVwaXBlbGluZSBmcm9tIFwiLi4vLi4vLi4vYXdzLWNvZGVwaXBlbGluZVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWNvZGVwaXBlbGluZSdcbmltcG9ydCAqIGFzIGNvZGVwaXBlbGluZV9hY3Rpb25zIGZyb20gXCIuLi8uLi8uLi9hd3MtY29kZXBpcGVsaW5lLWFjdGlvbnNcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1jb2RlcGlwZWxpbmUtYWN0aW9ucydcbmltcG9ydCAqIGFzIGVjMiBmcm9tIFwiLi4vLi4vLi4vYXdzLWVjMlwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWVjMidcbmltcG9ydCAqIGFzIGV2ZW50cyBmcm9tIFwiLi4vLi4vLi4vYXdzLWV2ZW50c1wiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWV2ZW50cydcbmltcG9ydCAqIGFzIGlhbSBmcm9tIFwiLi4vLi4vLi4vYXdzLWlhbVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSdcbmltcG9ydCB7IENvbnN0cnVjdCwgU3RhY2sgfSBmcm9tIFwiLi4vLi4vLi4vY29yZVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvY29yZSdcbmltcG9ydCB7IGNsb3VkQXNzZW1ibHlCdWlsZFNwZWNEaXIgfSBmcm9tICcuLi9wcml2YXRlL2NvbnN0cnVjdC1pbnRlcm5hbHMnO1xuaW1wb3J0IHsgY29weUVudmlyb25tZW50VmFyaWFibGVzLCBmaWx0ZXJFbXB0eSB9IGZyb20gJy4vX3V0aWwnO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgU2ltcGxlU3ludGhPcHRpb25zIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgc291cmNlQXJ0aWZhY3Q6IGNvZGVwaXBlbGluZS5BcnRpZmFjdDtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgY2xvdWRBc3NlbWJseUFydGlmYWN0OiBjb2RlcGlwZWxpbmUuQXJ0aWZhY3Q7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgZW52aXJvbm1lbnRWYXJpYWJsZXM/OiBSZWNvcmQ8c3RyaW5nLCBjb2RlYnVpbGQuQnVpbGRFbnZpcm9ubWVudFZhcmlhYmxlPjtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IGNvcHlFbnZpcm9ubWVudFZhcmlhYmxlcz86IHN0cmluZ1tdO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgYWN0aW9uTmFtZT86IHN0cmluZztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgcHJvamVjdE5hbWU/OiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IGVudmlyb25tZW50PzogY29kZWJ1aWxkLkJ1aWxkRW52aXJvbm1lbnQ7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBzdWJkaXJlY3Rvcnk/OiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBhZGRpdGlvbmFsQXJ0aWZhY3RzPzogQWRkaXRpb25hbEFydGlmYWN0W107XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSByb2xlUG9saWN5U3RhdGVtZW50cz86IGlhbS5Qb2xpY3lTdGF0ZW1lbnRbXTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IHZwYz86IGVjMi5JVnBjO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IHN1Ym5ldFNlbGVjdGlvbj86IGVjMi5TdWJuZXRTZWxlY3Rpb247XG59XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgU2ltcGxlU3ludGhBY3Rpb25Qcm9wcyBleHRlbmRzIFNpbXBsZVN5bnRoT3B0aW9ucyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgc3ludGhDb21tYW5kOiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgaW5zdGFsbENvbW1hbmQ/OiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBidWlsZENvbW1hbmQ/OiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBpbnN0YWxsQ29tbWFuZHM/OiBzdHJpbmdbXTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgYnVpbGRDb21tYW5kcz86IHN0cmluZ1tdO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgdGVzdENvbW1hbmRzPzogc3RyaW5nW107XG59XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBBZGRpdGlvbmFsQXJ0aWZhY3Qge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBkaXJlY3Rvcnk6IHN0cmluZztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgYXJ0aWZhY3Q6IGNvZGVwaXBlbGluZS5BcnRpZmFjdDtcbn1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGNsYXNzIFNpbXBsZVN5bnRoQWN0aW9uIGltcGxlbWVudHMgY29kZXBpcGVsaW5lLklBY3Rpb24sIGlhbS5JR3JhbnRhYmxlIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyBzdGFuZGFyZE5wbVN5bnRoKG9wdGlvbnM6IFN0YW5kYXJkTnBtU3ludGhPcHRpb25zKSB7XG4gICAgICAgIHJldHVybiBuZXcgU2ltcGxlU3ludGhBY3Rpb24oe1xuICAgICAgICAgICAgLi4ub3B0aW9ucyxcbiAgICAgICAgICAgIGluc3RhbGxDb21tYW5kOiBvcHRpb25zLmluc3RhbGxDb21tYW5kID8/ICducG0gY2knLFxuICAgICAgICAgICAgc3ludGhDb21tYW5kOiBvcHRpb25zLnN5bnRoQ29tbWFuZCA/PyAnbnB4IGNkayBzeW50aCcsXG4gICAgICAgICAgICB2cGM6IG9wdGlvbnMudnBjLFxuICAgICAgICAgICAgc3VibmV0U2VsZWN0aW9uOiBvcHRpb25zLnN1Ym5ldFNlbGVjdGlvbixcbiAgICAgICAgfSk7XG4gICAgfVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyBzdGFuZGFyZFlhcm5TeW50aChvcHRpb25zOiBTdGFuZGFyZFlhcm5TeW50aE9wdGlvbnMpIHtcbiAgICAgICAgcmV0dXJuIG5ldyBTaW1wbGVTeW50aEFjdGlvbih7XG4gICAgICAgICAgICAuLi5vcHRpb25zLFxuICAgICAgICAgICAgaW5zdGFsbENvbW1hbmQ6IG9wdGlvbnMuaW5zdGFsbENvbW1hbmQgPz8gJ3lhcm4gaW5zdGFsbCAtLWZyb3plbi1sb2NrZmlsZScsXG4gICAgICAgICAgICBzeW50aENvbW1hbmQ6IG9wdGlvbnMuc3ludGhDb21tYW5kID8/ICducHggY2RrIHN5bnRoJyxcbiAgICAgICAgICAgIHZwYzogb3B0aW9ucy52cGMsXG4gICAgICAgICAgICBzdWJuZXRTZWxlY3Rpb246IG9wdGlvbnMuc3VibmV0U2VsZWN0aW9uLFxuICAgICAgICB9KTtcbiAgICB9XG4gICAgcHJpdmF0ZSBfYWN0aW9uPzogY29kZXBpcGVsaW5lX2FjdGlvbnMuQ29kZUJ1aWxkQWN0aW9uO1xuICAgIHByaXZhdGUgX2FjdGlvblByb3BlcnRpZXM6IGNvZGVwaXBlbGluZS5BY3Rpb25Qcm9wZXJ0aWVzO1xuICAgIHByaXZhdGUgX3Byb2plY3Q/OiBjb2RlYnVpbGQuSVByb2plY3Q7XG4gICAgY29uc3RydWN0b3IocHJpdmF0ZSByZWFkb25seSBwcm9wczogU2ltcGxlU3ludGhBY3Rpb25Qcm9wcykge1xuICAgICAgICAvLyBBIG51bWJlciBvZiBhY3Rpb25Qcm9wZXJ0aWVzIGdldCByZWFkIGJlZm9yZSBiaW5kKCkgaXMgZXZlbiBjYWxsZWQgKHNvIGJlZm9yZSB3ZVxuICAgICAgICAvLyBoYXZlIG1hZGUgdGhlIFByb2plY3QgYW5kIGNhbiBjb25zdHJ1Y3QgdGhlIGFjdHVhbCBDb2RlQnVpbGRBY3Rpb24pXG4gICAgICAgIC8vXG4gICAgICAgIC8vIC0gYWN0aW9uTmFtZVxuICAgICAgICAvLyAtIHJlc291cmNlXG4gICAgICAgIC8vIC0gcmVnaW9uXG4gICAgICAgIC8vIC0gY2F0ZWdvcnlcbiAgICAgICAgLy8gLSByb2xlXG4gICAgICAgIC8vIC0gb3duZXJcbiAgICAgICAgdGhpcy5fYWN0aW9uUHJvcGVydGllcyA9IHtcbiAgICAgICAgICAgIGFjdGlvbk5hbWU6IHByb3BzLmFjdGlvbk5hbWUgPz8gJ1N5bnRoJyxcbiAgICAgICAgICAgIGNhdGVnb3J5OiBjb2RlcGlwZWxpbmUuQWN0aW9uQ2F0ZWdvcnkuQlVJTEQsXG4gICAgICAgICAgICBwcm92aWRlcjogJ0NvZGVCdWlsZCcsXG4gICAgICAgICAgICBhcnRpZmFjdEJvdW5kczogeyBtaW5JbnB1dHM6IDAsIG1heElucHV0czogNSwgbWluT3V0cHV0czogMCwgbWF4T3V0cHV0czogNSB9LFxuICAgICAgICAgICAgaW5wdXRzOiBbcHJvcHMuc291cmNlQXJ0aWZhY3RdLFxuICAgICAgICAgICAgb3V0cHV0czogW3Byb3BzLmNsb3VkQXNzZW1ibHlBcnRpZmFjdCwgLi4uKHByb3BzLmFkZGl0aW9uYWxBcnRpZmFjdHMgPz8gW10pLm1hcChhID0+IGEuYXJ0aWZhY3QpXSxcbiAgICAgICAgfTtcbiAgICAgICAgaWYgKHRoaXMucHJvcHMuaW5zdGFsbENvbW1hbmQgJiYgdGhpcy5wcm9wcy5pbnN0YWxsQ29tbWFuZHMpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignUGFzcyBlaXRoZXIgXFwnaW5zdGFsbENvbW1hbmRcXCcgb3IgXFwnaW5zdGFsbENvbW1hbmRzXFwnLCBidXQgbm90IGJvdGgnKTtcbiAgICAgICAgfVxuICAgICAgICBpZiAodGhpcy5wcm9wcy5idWlsZENvbW1hbmQgJiYgdGhpcy5wcm9wcy5idWlsZENvbW1hbmRzKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1Bhc3MgZWl0aGVyIFxcJ2J1aWxkQ29tbWFuZFxcJyBvciBcXCdidWlsZENvbW1hbmRzXFwnLCBidXQgbm90IGJvdGgnKTtcbiAgICAgICAgfVxuICAgICAgICBjb25zdCBhZGRscyA9IHByb3BzLmFkZGl0aW9uYWxBcnRpZmFjdHMgPz8gW107XG4gICAgICAgIGlmIChPYmplY3Qua2V5cyhhZGRscykubGVuZ3RoID4gMCkge1xuICAgICAgICAgICAgaWYgKCFwcm9wcy5jbG91ZEFzc2VtYmx5QXJ0aWZhY3QuYXJ0aWZhY3ROYW1lKSB7XG4gICAgICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdZb3UgbXVzdCBnaXZlIGFsbCBvdXRwdXQgYXJ0aWZhY3RzLCBpbmNsdWRpbmcgdGhlIFxcJ2Nsb3VkQXNzZW1ibHlBcnRpZmFjdFxcJywgbmFtZXMgd2hlbiB1c2luZyBcXCdhZGRpdGlvbmFsQXJ0aWZhY3RzXFwnJyk7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICBmb3IgKGNvbnN0IGFkZGwgb2YgYWRkbHMpIHtcbiAgICAgICAgICAgICAgICBpZiAoIWFkZGwuYXJ0aWZhY3QuYXJ0aWZhY3ROYW1lKSB7XG4gICAgICAgICAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignWW91IG11c3QgZ2l2ZSBhbGwgb3V0cHV0IGFydGlmYWN0cyBwYXNzZWQgdG8gU2ltcGxlU3ludGhBY3Rpb24gbmFtZXMgd2hlbiB1c2luZyBcXCdhZGRpdGlvbmFsQXJ0aWZhY3RzXFwnJyk7XG4gICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgfVxuICAgICAgICB9XG4gICAgfVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgZ2V0IGFjdGlvblByb3BlcnRpZXMoKTogY29kZXBpcGVsaW5lLkFjdGlvblByb3BlcnRpZXMge1xuICAgICAgICByZXR1cm4gdGhpcy5fYWN0aW9uUHJvcGVydGllcztcbiAgICB9XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBnZXQgcHJvamVjdCgpOiBjb2RlYnVpbGQuSVByb2plY3Qge1xuICAgICAgICBpZiAoIXRoaXMuX3Byb2plY3QpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignUHJvamVjdCBiZWNvbWVzIGF2YWlsYWJsZSBhZnRlciBTaW1wbGVTeW50aEFjdGlvbiBoYXMgYmVlbiBib3VuZCB0byBhIHN0YWdlJyk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIHRoaXMuX3Byb2plY3Q7XG4gICAgfVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgYmluZChzY29wZTogQ29uc3RydWN0LCBzdGFnZTogY29kZXBpcGVsaW5lLklTdGFnZSwgb3B0aW9uczogY29kZXBpcGVsaW5lLkFjdGlvbkJpbmRPcHRpb25zKTogY29kZXBpcGVsaW5lLkFjdGlvbkNvbmZpZyB7XG4gICAgICAgIGNvbnN0IGJ1aWxkQ29tbWFuZHMgPSB0aGlzLnByb3BzLmJ1aWxkQ29tbWFuZHMgPz8gW3RoaXMucHJvcHMuYnVpbGRDb21tYW5kXTtcbiAgICAgICAgY29uc3QgaW5zdGFsbENvbW1hbmRzID0gdGhpcy5wcm9wcy5pbnN0YWxsQ29tbWFuZHMgPz8gW3RoaXMucHJvcHMuaW5zdGFsbENvbW1hbmRdO1xuICAgICAgICBjb25zdCB0ZXN0Q29tbWFuZHMgPSB0aGlzLnByb3BzLnRlc3RDb21tYW5kcyA/PyBbXTtcbiAgICAgICAgY29uc3Qgc3ludGhDb21tYW5kID0gdGhpcy5wcm9wcy5zeW50aENvbW1hbmQ7XG4gICAgICAgIGNvbnN0IGJ1aWxkU3BlYyA9IGNvZGVidWlsZC5CdWlsZFNwZWMuZnJvbU9iamVjdCh7XG4gICAgICAgICAgICB2ZXJzaW9uOiAnMC4yJyxcbiAgICAgICAgICAgIHBoYXNlczoge1xuICAgICAgICAgICAgICAgIHByZV9idWlsZDoge1xuICAgICAgICAgICAgICAgICAgICBjb21tYW5kczogZmlsdGVyRW1wdHkoW1xuICAgICAgICAgICAgICAgICAgICAgICAgdGhpcy5wcm9wcy5zdWJkaXJlY3RvcnkgPyBgY2QgJHt0aGlzLnByb3BzLnN1YmRpcmVjdG9yeX1gIDogJycsXG4gICAgICAgICAgICAgICAgICAgICAgICAuLi5pbnN0YWxsQ29tbWFuZHMsXG4gICAgICAgICAgICAgICAgICAgIF0pLFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgYnVpbGQ6IHtcbiAgICAgICAgICAgICAgICAgICAgY29tbWFuZHM6IGZpbHRlckVtcHR5KFtcbiAgICAgICAgICAgICAgICAgICAgICAgIC4uLmJ1aWxkQ29tbWFuZHMsXG4gICAgICAgICAgICAgICAgICAgICAgICAuLi50ZXN0Q29tbWFuZHMsXG4gICAgICAgICAgICAgICAgICAgICAgICBzeW50aENvbW1hbmQsXG4gICAgICAgICAgICAgICAgICAgIF0pLFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgYXJ0aWZhY3RzOiByZW5kZXJBcnRpZmFjdHModGhpcyksXG4gICAgICAgIH0pO1xuICAgICAgICBjb25zdCBlbnZpcm9ubWVudCA9IHsgYnVpbGRJbWFnZTogY29kZWJ1aWxkLkxpbnV4QnVpbGRJbWFnZS5TVEFOREFSRF80XzAsIC4uLnRoaXMucHJvcHMuZW52aXJvbm1lbnQgfTtcbiAgICAgICAgY29uc3QgZW52aXJvbm1lbnRWYXJpYWJsZXMgPSB7XG4gICAgICAgICAgICAuLi5jb3B5RW52aXJvbm1lbnRWYXJpYWJsZXMoLi4udGhpcy5wcm9wcy5jb3B5RW52aXJvbm1lbnRWYXJpYWJsZXMgfHwgW10pLFxuICAgICAgICAgICAgLi4udGhpcy5wcm9wcy5lbnZpcm9ubWVudFZhcmlhYmxlcyxcbiAgICAgICAgfTtcbiAgICAgICAgLy8gQSBoYXNoIG92ZXIgdGhlIHZhbHVlcyB0aGF0IG1ha2UgdGhlIENvZGVCdWlsZCBQcm9qZWN0IHVuaXF1ZSAoYW5kIG5lY2Vzc2FyeVxuICAgICAgICAvLyB0byByZXN0YXJ0IHRoZSBwaXBlbGluZSBpZiBvbmUgb2YgdGhlbSBjaGFuZ2VzKS4gcHJvamVjdE5hbWUgaXMgbm90IG5lY2Vzc2FyeSB0byBpbmNsdWRlXG4gICAgICAgIC8vIGhlcmUgYmVjYXVzZSB0aGUgcGlwZWxpbmUgd2lsbCBkZWZpbml0ZWx5IHJlc3RhcnQgaWYgcHJvamVjdE5hbWUgY2hhbmdlcy5cbiAgICAgICAgLy8gKFJlc29sdmUgdG9rZW5zKVxuICAgICAgICBjb25zdCBwcm9qZWN0Q29uZmlnSGFzaCA9IGhhc2goU3RhY2sub2Yoc2NvcGUpLnJlc29sdmUoe1xuICAgICAgICAgICAgZW52aXJvbm1lbnQ6IHNlcmlhbGl6ZUJ1aWxkRW52aXJvbm1lbnQoZW52aXJvbm1lbnQpLFxuICAgICAgICAgICAgYnVpbGRTcGVjU3RyaW5nOiBidWlsZFNwZWMudG9CdWlsZFNwZWMoKSxcbiAgICAgICAgICAgIGVudmlyb25tZW50VmFyaWFibGVzLFxuICAgICAgICB9KSk7XG4gICAgICAgIGNvbnN0IHByb2plY3QgPSBuZXcgY29kZWJ1aWxkLlBpcGVsaW5lUHJvamVjdChzY29wZSwgJ0Nka0J1aWxkUHJvamVjdCcsIHtcbiAgICAgICAgICAgIHByb2plY3ROYW1lOiB0aGlzLnByb3BzLnByb2plY3ROYW1lLFxuICAgICAgICAgICAgZW52aXJvbm1lbnQsXG4gICAgICAgICAgICB2cGM6IHRoaXMucHJvcHMudnBjLFxuICAgICAgICAgICAgc3VibmV0U2VsZWN0aW9uOiB0aGlzLnByb3BzLnN1Ym5ldFNlbGVjdGlvbixcbiAgICAgICAgICAgIGJ1aWxkU3BlYyxcbiAgICAgICAgICAgIGVudmlyb25tZW50VmFyaWFibGVzLFxuICAgICAgICB9KTtcbiAgICAgICAgaWYgKHRoaXMucHJvcHMucm9sZVBvbGljeVN0YXRlbWVudHMgIT09IHVuZGVmaW5lZCkge1xuICAgICAgICAgICAgdGhpcy5wcm9wcy5yb2xlUG9saWN5U3RhdGVtZW50cy5mb3JFYWNoKHBvbGljeVN0YXRlbWVudCA9PiB7XG4gICAgICAgICAgICAgICAgcHJvamVjdC5hZGRUb1JvbGVQb2xpY3kocG9saWN5U3RhdGVtZW50KTtcbiAgICAgICAgICAgIH0pO1xuICAgICAgICB9XG4gICAgICAgIHRoaXMuX3Byb2plY3QgPSBwcm9qZWN0O1xuICAgICAgICB0aGlzLl9hY3Rpb24gPSBuZXcgY29kZXBpcGVsaW5lX2FjdGlvbnMuQ29kZUJ1aWxkQWN0aW9uKHtcbiAgICAgICAgICAgIGFjdGlvbk5hbWU6IHRoaXMuYWN0aW9uUHJvcGVydGllcy5hY3Rpb25OYW1lLFxuICAgICAgICAgICAgaW5wdXQ6IHRoaXMucHJvcHMuc291cmNlQXJ0aWZhY3QsXG4gICAgICAgICAgICBvdXRwdXRzOiBbdGhpcy5wcm9wcy5jbG91ZEFzc2VtYmx5QXJ0aWZhY3QsIC4uLih0aGlzLnByb3BzLmFkZGl0aW9uYWxBcnRpZmFjdHMgPz8gW10pLm1hcChhID0+IGEuYXJ0aWZhY3QpXSxcbiAgICAgICAgICAgIC8vIEluY2x1c2lvbiBvZiB0aGUgaGFzaCBoZXJlIHdpbGwgbGVhZCB0byB0aGUgcGlwZWxpbmUgc3RydWN0dXJlIGZvciBhbnkgY2hhbmdlc1xuICAgICAgICAgICAgLy8gbWFkZSB0aGUgY29uZmlnIG9mIHRoZSB1bmRlcmx5aW5nIENvZGVCdWlsZCBQcm9qZWN0LlxuICAgICAgICAgICAgLy8gSGVuY2UsIHRoZSBwaXBlbGluZSB3aWxsIGJlIHJlc3RhcnRlZC4gVGhpcyBpcyBuZWNlc3NhcnkgaWYgdGhlIHVzZXJzXG4gICAgICAgICAgICAvLyBhZGRzIChmb3IgZXhhbXBsZSkgYnVpbGQgb3IgdGVzdCBjb21tYW5kcyB0byB0aGUgYnVpbGRzcGVjLlxuICAgICAgICAgICAgZW52aXJvbm1lbnRWYXJpYWJsZXM6IHtcbiAgICAgICAgICAgICAgICBfUFJPSkVDVF9DT05GSUdfSEFTSDogeyB2YWx1ZTogcHJvamVjdENvbmZpZ0hhc2ggfSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBwcm9qZWN0LFxuICAgICAgICB9KTtcbiAgICAgICAgdGhpcy5fYWN0aW9uUHJvcGVydGllcyA9IHRoaXMuX2FjdGlvbi5hY3Rpb25Qcm9wZXJ0aWVzO1xuICAgICAgICByZXR1cm4gdGhpcy5fYWN0aW9uLmJpbmQoc2NvcGUsIHN0YWdlLCBvcHRpb25zKTtcbiAgICAgICAgZnVuY3Rpb24gcmVuZGVyQXJ0aWZhY3RzKHNlbGY6IFNpbXBsZVN5bnRoQWN0aW9uKSB7XG4gICAgICAgICAgICAvLyBzYXZlIHRoZSBnZW5lcmF0ZWQgZmlsZXMgaW4gdGhlIG91dHB1dCBhcnRpZmFjdFxuICAgICAgICAgICAgLy8gVGhpcyBwYXJ0IG9mIHRoZSBidWlsZHNwZWMgaGFzIHRvIGxvb2sgY29tcGxldGVseSBkaWZmZXJlbnQgZGVwZW5kaW5nIG9uIHdoZXRoZXIgd2UncmVcbiAgICAgICAgICAgIC8vIHVzaW5nIHNlY29uZGFyeSBhcnRpZmFjdHMgb3Igbm90LlxuICAgICAgICAgICAgY29uc3QgY2xvdWRBc21BcnRpZmFjdFNwZWMgPSB7XG4gICAgICAgICAgICAgICAgJ2Jhc2UtZGlyZWN0b3J5JzogcGF0aC5qb2luKHNlbGYucHJvcHMuc3ViZGlyZWN0b3J5ID8/ICcuJywgY2xvdWRBc3NlbWJseUJ1aWxkU3BlY0RpcihzY29wZSkpLFxuICAgICAgICAgICAgICAgICdmaWxlcyc6ICcqKi8qJyxcbiAgICAgICAgICAgIH07XG4gICAgICAgICAgICBpZiAoc2VsZi5wcm9wcy5hZGRpdGlvbmFsQXJ0aWZhY3RzKSB7XG4gICAgICAgICAgICAgICAgY29uc3Qgc2Vjb25kYXJ5OiBSZWNvcmQ8c3RyaW5nLCBhbnk+ID0ge307XG4gICAgICAgICAgICAgICAgaWYgKCFzZWxmLnByb3BzLmNsb3VkQXNzZW1ibHlBcnRpZmFjdC5hcnRpZmFjdE5hbWUpIHtcbiAgICAgICAgICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdXaGVuIHVzaW5nIGFkZGl0aW9uYWwgb3V0cHV0IGFydGlmYWN0cywgeW91IG11c3QgYWxzbyBuYW1lIHRoZSBDbG91ZEFzc2VtYmx5IGFydGlmYWN0Jyk7XG4gICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICAgIHNlY29uZGFyeVtzZWxmLnByb3BzLmNsb3VkQXNzZW1ibHlBcnRpZmFjdC5hcnRpZmFjdE5hbWVdID0gY2xvdWRBc21BcnRpZmFjdFNwZWM7XG4gICAgICAgICAgICAgICAgc2VsZi5wcm9wcy5hZGRpdGlvbmFsQXJ0aWZhY3RzLmZvckVhY2goKGFydCkgPT4ge1xuICAgICAgICAgICAgICAgICAgICBpZiAoIWFydC5hcnRpZmFjdC5hcnRpZmFjdE5hbWUpIHtcbiAgICAgICAgICAgICAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignWW91IG11c3QgZ2l2ZSB0aGUgb3V0cHV0IGFydGlmYWN0IGEgbmFtZScpO1xuICAgICAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgICAgIHNlY29uZGFyeVthcnQuYXJ0aWZhY3QuYXJ0aWZhY3ROYW1lXSA9IHtcbiAgICAgICAgICAgICAgICAgICAgICAgICdiYXNlLWRpcmVjdG9yeSc6IHBhdGguam9pbihzZWxmLnByb3BzLnN1YmRpcmVjdG9yeSA/PyAnLicsIGFydC5kaXJlY3RvcnkpLFxuICAgICAgICAgICAgICAgICAgICAgICAgJ2ZpbGVzJzogJyoqLyonLFxuICAgICAgICAgICAgICAgICAgICB9O1xuICAgICAgICAgICAgICAgIH0pO1xuICAgICAgICAgICAgICAgIHJldHVybiB7ICdzZWNvbmRhcnktYXJ0aWZhY3RzJzogc2Vjb25kYXJ5IH07XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICByZXR1cm4gY2xvdWRBc21BcnRpZmFjdFNwZWM7XG4gICAgICAgIH1cbiAgICB9XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBnZXQgZ3JhbnRQcmluY2lwYWwoKTogaWFtLklQcmluY2lwYWwge1xuICAgICAgICByZXR1cm4gdGhpcy5wcm9qZWN0LmdyYW50UHJpbmNpcGFsO1xuICAgIH1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIG9uU3RhdGVDaGFuZ2UobmFtZTogc3RyaW5nLCB0YXJnZXQ/OiBldmVudHMuSVJ1bGVUYXJnZXQsIG9wdGlvbnM/OiBldmVudHMuUnVsZVByb3BzKTogZXZlbnRzLlJ1bGUge1xuICAgICAgICBpZiAoIXRoaXMuX2FjdGlvbikge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdOZWVkIGJpbmQoKSBmaXJzdCcpO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiB0aGlzLl9hY3Rpb24ub25TdGF0ZUNoYW5nZShuYW1lLCB0YXJnZXQsIG9wdGlvbnMpO1xuICAgIH1cbn1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBTdGFuZGFyZE5wbVN5bnRoT3B0aW9ucyBleHRlbmRzIFNpbXBsZVN5bnRoT3B0aW9ucyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IGluc3RhbGxDb21tYW5kPzogc3RyaW5nO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgYnVpbGRDb21tYW5kPzogc3RyaW5nO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IHN5bnRoQ29tbWFuZD86IHN0cmluZztcbn1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgU3RhbmRhcmRZYXJuU3ludGhPcHRpb25zIGV4dGVuZHMgU2ltcGxlU3ludGhPcHRpb25zIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgaW5zdGFsbENvbW1hbmQ/OiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBidWlsZENvbW1hbmQ/OiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgc3ludGhDb21tYW5kPzogc3RyaW5nO1xufVxuZnVuY3Rpb24gaGFzaDxBPihvYmo6IEEpIHtcbiAgICBjb25zdCBkID0gY3J5cHRvLmNyZWF0ZUhhc2goJ3NoYTI1NicpO1xuICAgIGQudXBkYXRlKEpTT04uc3RyaW5naWZ5KG9iaikpO1xuICAgIHJldHVybiBkLmRpZ2VzdCgnaGV4Jyk7XG59XG4vKipcbiAqIFNlcmlhbGl6ZSBhIGJ1aWxkIGVudmlyb25tZW50IHRvIGRhdGEgKGdldCByaWQgb2YgY29uc3RydWN0cyAmIG9iamVjdHMpLCBzbyB3ZSBjYW4gSlNPTi5zdHJpbmdpZnkgaXRcbiAqL1xuZnVuY3Rpb24gc2VyaWFsaXplQnVpbGRFbnZpcm9ubWVudChlbnY6IGNvZGVidWlsZC5CdWlsZEVudmlyb25tZW50KSB7XG4gICAgcmV0dXJuIHtcbiAgICAgICAgcHJpdmlsZWdlZDogZW52LnByaXZpbGVnZWQsXG4gICAgICAgIGVudmlyb25tZW50VmFyaWFibGVzOiBlbnYuZW52aXJvbm1lbnRWYXJpYWJsZXMsXG4gICAgICAgIHR5cGU6IGVudi5idWlsZEltYWdlPy50eXBlLFxuICAgICAgICBpbWFnZUlkOiBlbnYuYnVpbGRJbWFnZT8uaW1hZ2VJZCxcbiAgICAgICAgY29tcHV0ZVR5cGU6IGVudi5jb21wdXRlVHlwZSxcbiAgICAgICAgaW1hZ2VQdWxsUHJpbmNpcGFsVHlwZTogZW52LmJ1aWxkSW1hZ2U/LmltYWdlUHVsbFByaW5jaXBhbFR5cGUsXG4gICAgICAgIHNlY3JldHNNYW5hZ2VyQXJuOiBlbnYuYnVpbGRJbWFnZT8uc2VjcmV0c01hbmFnZXJDcmVkZW50aWFscz8uc2VjcmV0QXJuLFxuICAgIH07XG59XG4iXX0=