"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DestinationPattern = exports.DestinationIdentifier = exports.DockerImageManifestEntry = exports.FileManifestEntry = exports.AssetManifestReader = void 0;
// FIXME: copied from `ckd-assets`, because this tool needs to read the asset manifest aswell.
const fs = require("fs");
const path = require("path");
const cloud_assembly_schema_1 = require("../../../cloud-assembly-schema"); // Automatically re-written from '@aws-cdk/cloud-assembly-schema'
/**
 * A manifest of assets
 */
class AssetManifestReader {
    constructor(directory, manifest) {
        this.manifest = manifest;
        this.directory = directory;
    }
    /**
     * Load an asset manifest from the given file
     */
    static fromFile(fileName) {
        try {
            const obj = cloud_assembly_schema_1.Manifest.loadAssetManifest(fileName);
            return new AssetManifestReader(path.dirname(fileName), obj);
        }
        catch (e) {
            throw new Error(`Canot read asset manifest '${fileName}': ${e.message}`);
        }
    }
    /**
     * Load an asset manifest from the given file or directory
     *
     * If the argument given is a directoy, the default asset file name will be used.
     */
    static fromPath(filePath) {
        let st;
        try {
            st = fs.statSync(filePath);
        }
        catch (e) {
            throw new Error(`Cannot read asset manifest at '${filePath}': ${e.message}`);
        }
        if (st.isDirectory()) {
            return AssetManifestReader.fromFile(path.join(filePath, AssetManifestReader.DEFAULT_FILENAME));
        }
        return AssetManifestReader.fromFile(filePath);
    }
    /**
     * Select a subset of assets and destinations from this manifest.
     *
     * Only assets with at least 1 selected destination are retained.
     *
     * If selection is not given, everything is returned.
     */
    select(selection) {
        if (selection === undefined) {
            return this;
        }
        const ret = { version: this.manifest.version, dockerImages: {}, files: {} };
        for (const assetType of ASSET_TYPES) {
            for (const [assetId, asset] of Object.entries(this.manifest[assetType] || {})) {
                const filteredDestinations = filterDict(asset.destinations, (_, destId) => selection.some(sel => sel.matches(new DestinationIdentifier(assetId, destId))));
                if (Object.keys(filteredDestinations).length > 0) {
                    ret[assetType][assetId] = {
                        ...asset,
                        destinations: filteredDestinations,
                    };
                }
            }
        }
        return new AssetManifestReader(this.directory, ret);
    }
    /**
     * Describe the asset manifest as a list of strings
     */
    list() {
        return [
            ...describeAssets('file', this.manifest.files || {}),
            ...describeAssets('docker-image', this.manifest.dockerImages || {}),
        ];
        function describeAssets(type, assets) {
            const ret = new Array();
            for (const [assetId, asset] of Object.entries(assets || {})) {
                ret.push(`${assetId} ${type} ${JSON.stringify(asset.source)}`);
                const destStrings = Object.entries(asset.destinations).map(([destId, dest]) => ` ${assetId}:${destId} ${JSON.stringify(dest)}`);
                ret.push(...prefixTreeChars(destStrings, '  '));
            }
            return ret;
        }
    }
    /**
     * List of assets, splat out to destinations
     */
    get entries() {
        return [
            ...makeEntries(this.manifest.files || {}, FileManifestEntry),
            ...makeEntries(this.manifest.dockerImages || {}, DockerImageManifestEntry),
        ];
        function makeEntries(assets, ctor) {
            const ret = new Array();
            for (const [assetId, asset] of Object.entries(assets)) {
                for (const [destId, destination] of Object.entries(asset.destinations)) {
                    ret.push(new ctor(new DestinationIdentifier(assetId, destId), asset.source, destination));
                }
            }
            return ret;
        }
    }
}
exports.AssetManifestReader = AssetManifestReader;
/**
 * The default name of the asset manifest in a cdk.out directory
 */
AssetManifestReader.DEFAULT_FILENAME = 'assets.json';
const ASSET_TYPES = ['files', 'dockerImages'];
/**
 * A manifest entry for a file asset
 */
class FileManifestEntry {
    constructor(
    /** Identifier for this asset */
    id, 
    /** Source of the file asset */
    source, 
    /** Destination for the file asset */
    destination) {
        this.id = id;
        this.source = source;
        this.destination = destination;
        this.type = 'file';
        this.genericSource = source;
        this.genericDestination = destination;
    }
}
exports.FileManifestEntry = FileManifestEntry;
/**
 * A manifest entry for a docker image asset
 */
class DockerImageManifestEntry {
    constructor(
    /** Identifier for this asset */
    id, 
    /** Source of the file asset */
    source, 
    /** Destination for the file asset */
    destination) {
        this.id = id;
        this.source = source;
        this.destination = destination;
        this.type = 'docker-image';
        this.genericSource = source;
        this.genericDestination = destination;
    }
}
exports.DockerImageManifestEntry = DockerImageManifestEntry;
/**
 * Identify an asset destination in an asset manifest
 */
class DestinationIdentifier {
    constructor(assetId, destinationId) {
        this.assetId = assetId;
        this.destinationId = destinationId;
    }
    /**
     * Return a string representation for this asset identifier
     */
    toString() {
        return this.destinationId ? `${this.assetId}:${this.destinationId}` : this.assetId;
    }
}
exports.DestinationIdentifier = DestinationIdentifier;
function filterDict(xs, pred) {
    const ret = {};
    for (const [key, value] of Object.entries(xs)) {
        if (pred(value, key)) {
            ret[key] = value;
        }
    }
    return ret;
}
/**
 * A filter pattern for an destination identifier
 */
class DestinationPattern {
    constructor(assetId, destinationId) {
        this.assetId = assetId;
        this.destinationId = destinationId;
    }
    /**
     * Parse a ':'-separated string into an asset/destination identifier
     */
    static parse(s) {
        if (!s) {
            throw new Error('Empty string is not a valid destination identifier');
        }
        const parts = s.split(':').map(x => x !== '*' ? x : undefined);
        if (parts.length === 1) {
            return new DestinationPattern(parts[0]);
        }
        if (parts.length === 2) {
            return new DestinationPattern(parts[0] || undefined, parts[1] || undefined);
        }
        throw new Error(`Asset identifier must contain at most 2 ':'-separated parts, got '${s}'`);
    }
    /**
     * Whether or not this pattern matches the given identifier
     */
    matches(id) {
        return (this.assetId === undefined || this.assetId === id.assetId)
            && (this.destinationId === undefined || this.destinationId === id.destinationId);
    }
    /**
     * Return a string representation for this asset identifier
     */
    toString() {
        var _a, _b;
        return `${(_a = this.assetId) !== null && _a !== void 0 ? _a : '*'}:${(_b = this.destinationId) !== null && _b !== void 0 ? _b : '*'}`;
    }
}
exports.DestinationPattern = DestinationPattern;
/**
 * Prefix box-drawing characters to make lines look like a hanging tree
 */
function prefixTreeChars(xs, prefix = '') {
    const ret = new Array();
    for (let i = 0; i < xs.length; i++) {
        const isLast = i === xs.length - 1;
        const boxChar = isLast ? '└' : '├';
        ret.push(`${prefix}${boxChar}${xs[i]}`);
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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