"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PublishAssetsAction = exports.AssetType = void 0;
const codebuild = require("../../../aws-codebuild"); // Automatically re-written from '@aws-cdk/aws-codebuild'
const codepipeline_actions = require("../../../aws-codepipeline-actions"); // Automatically re-written from '@aws-cdk/aws-codepipeline-actions'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * (experimental) Type of the asset that is being published.
 *
 * @experimental
 */
var AssetType;
(function (AssetType) {
    AssetType["FILE"] = "file";
    AssetType["DOCKER_IMAGE"] = "docker-image";
})(AssetType = exports.AssetType || (exports.AssetType = {}));
/**
 * (experimental) Action to publish an asset in the pipeline.
 *
 * Creates a CodeBuild project which will use the CDK CLI
 * to prepare and publish the asset.
 *
 * You do not need to instantiate this action -- it will automatically
 * be added by the pipeline when you add stacks that use assets.
 *
 * @experimental
 */
class PublishAssetsAction extends core_2.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.commands = new Array();
        const installSuffix = props.cdkCliVersion ? `@${props.cdkCliVersion}` : '';
        const project = new codebuild.PipelineProject(this, 'Default', {
            projectName: this.props.projectName,
            environment: {
                buildImage: codebuild.LinuxBuildImage.STANDARD_4_0,
                privileged: (props.assetType === AssetType.DOCKER_IMAGE) ? true : undefined,
            },
            vpc: props.vpc,
            subnetSelection: props.subnetSelection,
            buildSpec: codebuild.BuildSpec.fromObject({
                version: '0.2',
                phases: {
                    install: {
                        commands: `npm install -g cdk-assets${installSuffix}`,
                    },
                    build: {
                        commands: core_1.Lazy.listValue({ produce: () => this.commands }),
                    },
                },
            }),
            role: props.role,
        });
        const rolePattern = props.assetType === AssetType.DOCKER_IMAGE
            ? 'arn:*:iam::*:role/*-image-publishing-role-*'
            : 'arn:*:iam::*:role/*-file-publishing-role-*';
        project.addToRolePolicy(new iam.PolicyStatement({
            actions: ['sts:AssumeRole'],
            resources: [rolePattern],
        }));
        this.action = new codepipeline_actions.CodeBuildAction({
            actionName: props.actionName,
            project,
            input: this.props.cloudAssemblyInput,
            role: props.role,
            // Add this purely so that the pipeline will selfupdate if the CLI version changes
            environmentVariables: props.cdkCliVersion ? {
                CDK_CLI_VERSION: { value: props.cdkCliVersion },
            } : undefined,
        });
    }
    /**
     * (experimental) Add a single publishing command.
     *
     * Manifest path should be relative to the root Cloud Assembly.
     *
     * @experimental
     */
    addPublishCommand(relativeManifestPath, assetSelector) {
        const command = `cdk-assets --path "${relativeManifestPath}" --verbose publish "${assetSelector}"`;
        if (!this.commands.includes(command)) {
            this.commands.push(command);
        }
    }
    /**
     * (experimental) Exists to implement IAction.
     *
     * @experimental
     */
    bind(scope, stage, options) {
        return this.action.bind(scope, stage, options);
    }
    /**
     * (experimental) Exists to implement IAction.
     *
     * @experimental
     */
    onStateChange(name, target, options) {
        return this.action.onStateChange(name, target, options);
    }
    /**
     * (experimental) Exists to implement IAction.
     *
     * @experimental
     */
    get actionProperties() {
        // FIXME: I have had to make this class a Construct, because:
        //
        // - It needs access to the Construct tree, because it is going to add a `PipelineProject`.
        // - I would have liked to have done that in bind(), however,
        // - `actionProperties` (this method) is called BEFORE bind() is called, and by that point I
        //   don't have the "inner" Action yet to forward the call to.
        //
        // I've therefore had to construct the inner CodeBuildAction in the constructor, which requires making this
        // Action a Construct.
        //
        // Combined with how non-intuitive it is to make the "StackDeployAction", I feel there is something
        // wrong with the Action abstraction here.
        return this.action.actionProperties;
    }
}
exports.PublishAssetsAction = PublishAssetsAction;
//# sourceMappingURL=data:application/json;base64,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