"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudAssemblyBuilder = exports.CloudAssembly = void 0;
const fs = require("fs");
const os = require("os");
const path = require("path");
const cxschema = require("../../cloud-assembly-schema"); // Automatically re-written from '@aws-cdk/cloud-assembly-schema'
const cloudformation_artifact_1 = require("./artifacts/cloudformation-artifact");
const nested_cloud_assembly_artifact_1 = require("./artifacts/nested-cloud-assembly-artifact");
const tree_cloud_artifact_1 = require("./artifacts/tree-cloud-artifact");
const cloud_artifact_1 = require("./cloud-artifact");
const toposort_1 = require("./toposort");
/**
 * The name of the root manifest file of the assembly.
 */
const MANIFEST_FILE = 'manifest.json';
/**
 * (experimental) Represents a deployable cloud application.
 *
 * @experimental
 */
class CloudAssembly {
    /**
     * (experimental) Reads a cloud assembly from the specified directory.
     *
     * @param directory The root directory of the assembly.
     * @experimental
     */
    constructor(directory) {
        this.directory = directory;
        this.manifest = cxschema.Manifest.load(path.join(directory, MANIFEST_FILE));
        this.version = this.manifest.version;
        this.artifacts = this.renderArtifacts();
        this.runtime = this.manifest.runtime || { libraries: {} };
        // force validation of deps by accessing 'depends' on all artifacts
        this.validateDeps();
    }
    /**
     * (experimental) Attempts to find an artifact with a specific identity.
     *
     * @param id The artifact ID.
     * @returns A `CloudArtifact` object or `undefined` if the artifact does not exist in this assembly.
     * @experimental
     */
    tryGetArtifact(id) {
        return this.artifacts.find(a => a.id === id);
    }
    /**
     * (experimental) Returns a CloudFormation stack artifact from this assembly.
     *
     * Will only search the current assembly.
     *
     * @param stackName the name of the CloudFormation stack.
     * @returns a `CloudFormationStackArtifact` object.
     * @experimental
     * @throws if there is more than one stack with the same stack name. You can
     * use `getStackArtifact(stack.artifactId)` instead.
     */
    getStackByName(stackName) {
        const artifacts = this.artifacts.filter(a => a instanceof cloudformation_artifact_1.CloudFormationStackArtifact && a.stackName === stackName);
        if (!artifacts || artifacts.length === 0) {
            throw new Error(`Unable to find stack with stack name "${stackName}"`);
        }
        if (artifacts.length > 1) {
            // eslint-disable-next-line max-len
            throw new Error(`There are multiple stacks with the stack name "${stackName}" (${artifacts.map(a => a.id).join(',')}). Use "getStackArtifact(id)" instead`);
        }
        return artifacts[0];
    }
    /**
     * (deprecated) Returns a CloudFormation stack artifact by name from this assembly.
     *
     * @deprecated renamed to `getStackByName` (or `getStackArtifact(id)`)
     */
    getStack(stackName) {
        return this.getStackByName(stackName);
    }
    /**
     * (experimental) Returns a CloudFormation stack artifact from this assembly.
     *
     * @param artifactId the artifact id of the stack (can be obtained through `stack.artifactId`).
     * @returns a `CloudFormationStackArtifact` object.
     * @experimental
     * @throws if there is no stack artifact with that id
     */
    getStackArtifact(artifactId) {
        const artifact = this.tryGetArtifact(artifactId);
        if (!artifact) {
            throw new Error(`Unable to find artifact with id "${artifactId}"`);
        }
        if (!(artifact instanceof cloudformation_artifact_1.CloudFormationStackArtifact)) {
            throw new Error(`Artifact ${artifactId} is not a CloudFormation stack`);
        }
        return artifact;
    }
    /**
     * (experimental) Returns a nested assembly artifact.
     *
     * @param artifactId The artifact ID of the nested assembly.
     * @experimental
     */
    getNestedAssemblyArtifact(artifactId) {
        const artifact = this.tryGetArtifact(artifactId);
        if (!artifact) {
            throw new Error(`Unable to find artifact with id "${artifactId}"`);
        }
        if (!(artifact instanceof nested_cloud_assembly_artifact_1.NestedCloudAssemblyArtifact)) {
            throw new Error(`Found artifact '${artifactId}' but it's not a nested cloud assembly`);
        }
        return artifact;
    }
    /**
     * (experimental) Returns a nested assembly.
     *
     * @param artifactId The artifact ID of the nested assembly.
     * @experimental
     */
    getNestedAssembly(artifactId) {
        return this.getNestedAssemblyArtifact(artifactId).nestedAssembly;
    }
    /**
     * (experimental) Returns the tree metadata artifact from this assembly.
     *
     * @returns a `TreeCloudArtifact` object if there is one defined in the manifest, `undefined` otherwise.
     * @experimental
     * @throws if there is no metadata artifact by that name
     */
    tree() {
        const trees = this.artifacts.filter(a => a.manifest.type === cxschema.ArtifactType.CDK_TREE);
        if (trees.length === 0) {
            return undefined;
        }
        else if (trees.length > 1) {
            throw new Error(`Multiple artifacts of type ${cxschema.ArtifactType.CDK_TREE} found in manifest`);
        }
        const tree = trees[0];
        if (!(tree instanceof tree_cloud_artifact_1.TreeCloudArtifact)) {
            throw new Error('"Tree" artifact is not of expected type');
        }
        return tree;
    }
    /**
     * @returns all the CloudFormation stack artifacts that are included in this assembly.
     * @experimental
     */
    get stacks() {
        return this.artifacts.filter(isCloudFormationStackArtifact);
        function isCloudFormationStackArtifact(x) {
            return x instanceof cloudformation_artifact_1.CloudFormationStackArtifact;
        }
    }
    /**
     * (experimental) The nested assembly artifacts in this assembly.
     *
     * @experimental
     */
    get nestedAssemblies() {
        return this.artifacts.filter(isNestedCloudAssemblyArtifact);
        function isNestedCloudAssemblyArtifact(x) {
            return x instanceof nested_cloud_assembly_artifact_1.NestedCloudAssemblyArtifact;
        }
    }
    validateDeps() {
        for (const artifact of this.artifacts) {
            ignore(artifact.dependencies);
        }
    }
    renderArtifacts() {
        const result = new Array();
        for (const [name, artifact] of Object.entries(this.manifest.artifacts || {})) {
            const cloudartifact = cloud_artifact_1.CloudArtifact.fromManifest(this, name, artifact);
            if (cloudartifact) {
                result.push(cloudartifact);
            }
        }
        return toposort_1.topologicalSort(result, x => x.id, x => x._dependencyIDs);
    }
}
exports.CloudAssembly = CloudAssembly;
/**
 * (experimental) Can be used to build a cloud assembly.
 *
 * @experimental
 */
class CloudAssemblyBuilder {
    /**
     * (experimental) Initializes a cloud assembly builder.
     *
     * @param outdir The output directory, uses temporary directory if undefined.
     * @experimental
     */
    constructor(outdir, props = {}) {
        var _a;
        this.artifacts = {};
        this.missing = new Array();
        this.outdir = determineOutputDirectory(outdir);
        this.assetOutdir = (_a = props.assetOutdir) !== null && _a !== void 0 ? _a : this.outdir;
        // we leverage the fact that outdir is long-lived to avoid staging assets into it
        // that were already staged (copying can be expensive). this is achieved by the fact
        // that assets use a source hash as their name. other artifacts, and the manifest itself,
        // will overwrite existing files as needed.
        ensureDirSync(this.outdir);
    }
    /**
     * (experimental) Adds an artifact into the cloud assembly.
     *
     * @param id The ID of the artifact.
     * @param manifest The artifact manifest.
     * @experimental
     */
    addArtifact(id, manifest) {
        this.artifacts[id] = filterUndefined(manifest);
    }
    /**
     * (experimental) Reports that some context is missing in order for this cloud assembly to be fully synthesized.
     *
     * @param missing Missing context information.
     * @experimental
     */
    addMissing(missing) {
        if (this.missing.every(m => m.key !== missing.key)) {
            this.missing.push(missing);
        }
    }
    /**
     * (experimental) Finalizes the cloud assembly into the output directory returns a `CloudAssembly` object that can be used to inspect the assembly.
     *
     * @experimental
     */
    buildAssembly(options = {}) {
        // explicitly initializing this type will help us detect
        // breaking changes. (For example adding a required property will break compilation).
        let manifest = {
            version: cxschema.Manifest.version(),
            artifacts: this.artifacts,
            runtime: options.runtimeInfo,
            missing: this.missing.length > 0 ? this.missing : undefined,
        };
        // now we can filter
        manifest = filterUndefined(manifest);
        const manifestFilePath = path.join(this.outdir, MANIFEST_FILE);
        cxschema.Manifest.save(manifest, manifestFilePath);
        // "backwards compatibility": in order for the old CLI to tell the user they
        // need a new version, we'll emit the legacy manifest with only "version".
        // this will result in an error "CDK Toolkit >= CLOUD_ASSEMBLY_VERSION is required in order to interact with this program."
        fs.writeFileSync(path.join(this.outdir, 'cdk.out'), JSON.stringify({ version: manifest.version }));
        return new CloudAssembly(this.outdir);
    }
    /**
     * (experimental) Creates a nested cloud assembly.
     *
     * @experimental
     */
    createNestedAssembly(artifactId, displayName) {
        const directoryName = artifactId;
        const innerAsmDir = path.join(this.outdir, directoryName);
        this.addArtifact(artifactId, {
            type: cxschema.ArtifactType.NESTED_CLOUD_ASSEMBLY,
            properties: {
                directoryName,
                displayName,
            },
        });
        return new CloudAssemblyBuilder(innerAsmDir, {
            // Reuse the same asset output directory as the current Casm builder
            assetOutdir: this.assetOutdir,
        });
    }
}
exports.CloudAssemblyBuilder = CloudAssemblyBuilder;
/**
 * Returns a copy of `obj` without undefined values in maps or arrays.
 */
function filterUndefined(obj) {
    if (Array.isArray(obj)) {
        return obj.filter(x => x !== undefined).map(x => filterUndefined(x));
    }
    if (typeof (obj) === 'object') {
        const ret = {};
        for (const [key, value] of Object.entries(obj)) {
            if (value === undefined) {
                continue;
            }
            ret[key] = filterUndefined(value);
        }
        return ret;
    }
    return obj;
}
function ignore(_x) {
    return;
}
/**
 * Turn the given optional output directory into a fixed output directory
 */
function determineOutputDirectory(outdir) {
    return outdir !== null && outdir !== void 0 ? outdir : fs.mkdtempSync(path.join(fs.realpathSync(os.tmpdir()), 'cdk.out'));
}
function ensureDirSync(dir) {
    if (fs.existsSync(dir)) {
        if (!fs.statSync(dir).isDirectory()) {
            throw new Error(`${dir} must be a directory`);
        }
    }
    else {
        fs.mkdirSync(dir, { recursive: true });
    }
}
//# sourceMappingURL=data:application/json;base64,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