"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsCustomResource = exports.AwsCustomResourcePolicy = exports.PhysicalResourceId = exports.PhysicalResourceIdReference = void 0;
const fs = require("fs");
const path = require("path");
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const lambda = require("../../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const runtime_1 = require("./runtime");
// don't use "require" since the typescript compiler emits errors since this
// file is not listed in tsconfig.json.
const metadata = JSON.parse(fs.readFileSync(path.join(__dirname, 'sdk-api-metadata.json'), 'utf-8'));
const awsSdkMetadata = metadata;
/**
 * (experimental) Reference to the physical resource id that can be passed to the AWS operation as a parameter.
 *
 * @experimental
 */
class PhysicalResourceIdReference {
    /**
     * (experimental) toJSON serialization to replace `PhysicalResourceIdReference` with a magic string.
     *
     * @experimental
     */
    toJSON() {
        return runtime_1.PHYSICAL_RESOURCE_ID_REFERENCE;
    }
}
exports.PhysicalResourceIdReference = PhysicalResourceIdReference;
/**
 * (experimental) Physical ID of the custom resource.
 *
 * @experimental
 */
class PhysicalResourceId {
    /**
     * @param responsePath Path to a response data element to be used as the physical id.
     * @param id Literal string to be used as the physical id.
     */
    constructor(responsePath, id) {
        this.responsePath = responsePath;
        this.id = id;
    }
    /**
     * (experimental) Extract the physical resource id from the path (dot notation) to the data in the API call response.
     *
     * @experimental
     */
    static fromResponse(responsePath) {
        return new PhysicalResourceId(responsePath, undefined);
    }
    /**
     * (experimental) Explicit physical resource id.
     *
     * @experimental
     */
    static of(id) {
        return new PhysicalResourceId(undefined, id);
    }
}
exports.PhysicalResourceId = PhysicalResourceId;
/**
 * (experimental) The IAM Policy that will be applied to the different calls.
 *
 * @experimental
 */
class AwsCustomResourcePolicy {
    /**
     * @param statements statements for explicit policy.
     * @param resources resources for auto-generated from SDK calls.
     */
    constructor(statements, resources) {
        this.statements = statements;
        this.resources = resources;
    }
    /**
     * (experimental) Explicit IAM Policy Statements.
     *
     * @param statements the statements to propagate to the SDK calls.
     * @experimental
     */
    static fromStatements(statements) {
        return new AwsCustomResourcePolicy(statements, undefined);
    }
    /**
     * (experimental) Generate IAM Policy Statements from the configured SDK calls.
     *
     * Each SDK call with be translated to an IAM Policy Statement in the form of: `call.service:call.action` (e.g `s3:PutObject`).
     *
     * @param options options for the policy generation.
     * @experimental
     */
    static fromSdkCalls(options) {
        return new AwsCustomResourcePolicy([], options.resources);
    }
}
exports.AwsCustomResourcePolicy = AwsCustomResourcePolicy;
/**
 * (experimental) Use this constant to configure access to any resource.
 *
 * @experimental
 */
AwsCustomResourcePolicy.ANY_RESOURCE = ['*'];
/**
 * (experimental) Defines a custom resource that is materialized using specific AWS API calls.
 *
 * Use this to bridge any gap that might exist in the CloudFormation Coverage.
 * You can specify exactly which calls are invoked for the 'CREATE', 'UPDATE' and 'DELETE' life cycle events.
 *
 * @experimental
 */
class AwsCustomResource extends cdk.Construct {
    // 'props' cannot be optional, even though all its properties are optional.
    // this is because at least one sdk call must be provided.
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id);
        if (!props.onCreate && !props.onUpdate && !props.onDelete) {
            throw new Error('At least `onCreate`, `onUpdate` or `onDelete` must be specified.');
        }
        for (const call of [props.onCreate, props.onUpdate]) {
            if (call && !call.physicalResourceId) {
                throw new Error('`physicalResourceId` must be specified for onCreate and onUpdate calls.');
            }
        }
        for (const call of [props.onCreate, props.onUpdate, props.onDelete]) {
            if ((_a = call === null || call === void 0 ? void 0 : call.physicalResourceId) === null || _a === void 0 ? void 0 : _a.responsePath) {
                AwsCustomResource.breakIgnoreErrorsCircuit([call], 'PhysicalResourceId.fromResponse');
            }
        }
        if ((_b = props.onCreate) === null || _b === void 0 ? void 0 : _b.parameters) {
            const flattenedOnCreateParams = runtime_1.flatten(JSON.parse(JSON.stringify(props.onCreate.parameters)));
            for (const param in flattenedOnCreateParams) {
                if (flattenedOnCreateParams[param] === runtime_1.PHYSICAL_RESOURCE_ID_REFERENCE) {
                    throw new Error('`PhysicalResourceIdReference` must not be specified in `onCreate` parameters.');
                }
            }
        }
        this.props = props;
        const provider = new lambda.SingletonFunction(this, 'Provider', {
            code: lambda.Code.fromAsset(path.join(__dirname, 'runtime')),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler',
            uuid: '679f53fa-c002-430c-b0da-5b7982bd2287',
            lambdaPurpose: 'AWS',
            timeout: props.timeout || cdk.Duration.minutes(2),
            role: props.role,
            logRetention: props.logRetention,
            functionName: props.functionName,
        });
        this.grantPrincipal = provider.grantPrincipal;
        // Create the policy statements for the custom resource function role, or use the user-provided ones
        const statements = [];
        if (props.policy.statements.length !== 0) {
            // Use custom statements provided by the user
            for (const statement of props.policy.statements) {
                statements.push(statement);
            }
        }
        else {
            // Derive statements from AWS SDK calls
            for (const call of [props.onCreate, props.onUpdate, props.onDelete]) {
                if (call) {
                    const statement = new iam.PolicyStatement({
                        actions: [awsSdkToIamAction(call.service, call.action)],
                        resources: props.policy.resources,
                    });
                    statements.push(statement);
                }
            }
        }
        const policy = new iam.Policy(this, 'CustomResourcePolicy', {
            statements: statements,
        });
        if (provider.role !== undefined) {
            policy.attachToRole(provider.role);
        }
        const create = props.onCreate || props.onUpdate;
        this.customResource = new cdk.CustomResource(this, 'Resource', {
            resourceType: props.resourceType || 'Custom::AWS',
            serviceToken: provider.functionArn,
            pascalCaseProperties: true,
            properties: {
                create: create && encodeBooleans(create),
                update: props.onUpdate && encodeBooleans(props.onUpdate),
                delete: props.onDelete && encodeBooleans(props.onDelete),
                installLatestAwsSdk: (_c = props.installLatestAwsSdk) !== null && _c !== void 0 ? _c : true,
            },
        });
        // If the policy was deleted first, then the function might lose permissions to delete the custom resource
        // This is here so that the policy doesn't get removed before onDelete is called
        this.customResource.node.addDependency(policy);
    }
    static breakIgnoreErrorsCircuit(sdkCalls, caller) {
        for (const call of sdkCalls) {
            if (call === null || call === void 0 ? void 0 : call.ignoreErrorCodesMatching) {
                throw new Error(`\`${caller}\`` + ' cannot be called along with `ignoreErrorCodesMatching`.');
            }
        }
    }
    /**
     * (experimental) Returns response data for the AWS SDK call.
     *
     * Example for S3 / listBucket : 'Buckets.0.Name'
     *
     * Use `Token.asXxx` to encode the returned `Reference` as a specific type or
     * use the convenience `getDataString` for string attributes.
     *
     * Note that you cannot use this method if `ignoreErrorCodesMatching`
     * is configured for any of the SDK calls. This is because in such a case,
     * the response data might not exist, and will cause a CloudFormation deploy time error.
     *
     * @param dataPath the path to the data.
     * @experimental
     */
    getResponseFieldReference(dataPath) {
        AwsCustomResource.breakIgnoreErrorsCircuit([this.props.onCreate, this.props.onUpdate], 'getData');
        return this.customResource.getAtt(dataPath);
    }
    /**
     * (experimental) Returns response data for the AWS SDK call as string.
     *
     * Example for S3 / listBucket : 'Buckets.0.Name'
     *
     * Note that you cannot use this method if `ignoreErrorCodesMatching`
     * is configured for any of the SDK calls. This is because in such a case,
     * the response data might not exist, and will cause a CloudFormation deploy time error.
     *
     * @param dataPath the path to the data.
     * @experimental
     */
    getResponseField(dataPath) {
        AwsCustomResource.breakIgnoreErrorsCircuit([this.props.onCreate, this.props.onUpdate], 'getDataString');
        return this.customResource.getAttString(dataPath);
    }
}
exports.AwsCustomResource = AwsCustomResource;
/**
 * Transform SDK service/action to IAM action using metadata from aws-sdk module.
 * Example: CloudWatchLogs with putRetentionPolicy => logs:PutRetentionPolicy
 *
 * TODO: is this mapping correct for all services?
 */
function awsSdkToIamAction(service, action) {
    const srv = service.toLowerCase();
    const iamService = (awsSdkMetadata[srv] && awsSdkMetadata[srv].prefix) || srv;
    const iamAction = action.charAt(0).toUpperCase() + action.slice(1);
    return `${iamService}:${iamAction}`;
}
/**
 * Encodes booleans as special strings
 */
function encodeBooleans(object) {
    return JSON.parse(JSON.stringify(object), (_k, v) => {
        switch (v) {
            case true:
                return 'TRUE:BOOLEAN';
            case false:
                return 'FALSE:BOOLEAN';
            default:
                return v;
        }
    });
}
//# sourceMappingURL=data:application/json;base64,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