"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.captureStackTrace = void 0;
const debug_1 = require("./debug");
/**
 * Captures the current process' stack trace.
 *
 * Stack traces are often invaluable tools to help diagnose problems, however
 * their capture is a rather expensive operation, and the stack traces can be
 * large. Consequently, users are stronly advised to condition capturing stack
 * traces to specific user opt-in.
 *
 * If the `CDK_DISABLE_STACK_TRACE` environment variable is set (to any value,
 * except for an empty string), no stack traces will be captured, and instead
 * the literal value `['stack traces disabled']` will be returned instead. This
 * is only true if the `CDK_DEBUG` environment variable is not set to `'true'`
 * or '1', in which case stack traces are *always* captured.
 *
 * @param below an optional function starting from which stack frames will be
 *              ignored. Defaults to the `captureStackTrace` function itself.
 * @param limit and optional upper bound to the number of stack frames to be
 *              captured. If not provided, this defaults to
 *              `Number.MAX_SAFE_INTEGER`, effectively meaning "no limit".
 *
 * @returns the captured stack trace, as an array of stack frames.
 */
function captureStackTrace(below = captureStackTrace, limit = Number.MAX_SAFE_INTEGER) {
    if (process.env.CDK_DISABLE_STACK_TRACE && !debug_1.debugModeEnabled()) {
        return ['stack traces disabled'];
    }
    const object = {};
    const previousLimit = Error.stackTraceLimit;
    try {
        Error.stackTraceLimit = limit;
        Error.captureStackTrace(object, below);
    }
    finally {
        Error.stackTraceLimit = previousLimit;
    }
    if (!object.stack) {
        return [];
    }
    return object.stack.split('\n').slice(1).map(s => s.replace(/^\s*at\s+/, ''));
}
exports.captureStackTrace = captureStackTrace;
//# sourceMappingURL=data:application/json;base64,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