"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.collectRuntimeInformation = void 0;
const path_1 = require("path");
const node_version_1 = require("./node-version");
// list of NPM scopes included in version reporting e.g. @aws-cdk and @aws-solutions-konstruk
const WHITELIST_SCOPES = ['@aws-cdk', '@aws-cdk-containers', '@aws-solutions-konstruk', '@aws-solutions-constructs', '@amzn'];
// list of NPM packages included in version reporting
const WHITELIST_PACKAGES = ['aws-rfdk'];
/**
 * Returns a list of loaded modules and their versions.
 */
function collectRuntimeInformation() {
    const libraries = {};
    for (const fileName of Object.keys(require.cache)) {
        const pkg = findNpmPackage(fileName);
        if (pkg && !pkg.private) {
            libraries[pkg.name] = pkg.version;
        }
    }
    // include only libraries that are in the whitelistLibraries list
    for (const name of Object.keys(libraries)) {
        let foundMatch = false;
        for (const scope of WHITELIST_SCOPES) {
            if (name.startsWith(`${scope}/`)) {
                foundMatch = true;
            }
        }
        foundMatch = foundMatch || WHITELIST_PACKAGES.includes(name);
        if (!foundMatch) {
            delete libraries[name];
        }
    }
    // add jsii runtime version
    libraries['jsii-runtime'] = getJsiiAgentVersion();
    return { libraries };
}
exports.collectRuntimeInformation = collectRuntimeInformation;
/**
 * Determines which NPM module a given loaded javascript file is from.
 *
 * The only infromation that is available locally is a list of Javascript files,
 * and every source file is associated with a search path to resolve the further
 * ``require`` calls made from there, which includes its own directory on disk,
 * and parent directories - for example:
 *
 * [ '...repo/packages/aws-cdk-resources/lib/cfn/node_modules',
 *   '...repo/packages/aws-cdk-resources/lib/node_modules',
 *   '...repo/packages/aws-cdk-resources/node_modules',
 *   '...repo/packages/node_modules',
 *   // etc...
 * ]
 *
 * We are looking for ``package.json`` that is anywhere in the tree, except it's
 * in the parent directory, not in the ``node_modules`` directory. For this
 * reason, we strip the ``/node_modules`` suffix off each path and use regular
 * module resolution to obtain a reference to ``package.json``.
 *
 * @param fileName a javascript file name.
 * @returns the NPM module infos (aka ``package.json`` contents), or
 *      ``undefined`` if the lookup was unsuccessful.
 */
function findNpmPackage(fileName) {
    const mod = require.cache[fileName];
    if (!(mod === null || mod === void 0 ? void 0 : mod.paths)) {
        // sometimes this can be undefined. for example when querying for .json modules
        // inside a jest runtime environment.
        // see https://github.com/aws/aws-cdk/issues/7657
        // potentially we can remove this if it turns out to be a bug in how jest implemented the 'require' module.
        return undefined;
    }
    // For any path in ``mod.paths`` that is a node_modules folder, use its parent directory instead.
    const paths = mod === null || mod === void 0 ? void 0 : mod.paths.map((path) => path_1.basename(path) === 'node_modules' ? path_1.dirname(path) : path);
    try {
        const packagePath = require.resolve(
        // Resolution behavior changed in node 12.0.0 - https://github.com/nodejs/node/issues/27583
        node_version_1.major >= 12 ? './package.json' : 'package.json', { paths });
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        return require(packagePath);
    }
    catch (e) {
        return undefined;
    }
}
function getJsiiAgentVersion() {
    let jsiiAgent = process.env.JSII_AGENT;
    // if JSII_AGENT is not specified, we will assume this is a node.js runtime
    // and plug in our node.js version
    if (!jsiiAgent) {
        jsiiAgent = `node.js/${process.version}`;
    }
    return jsiiAgent;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicnVudGltZS1pbmZvLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsicnVudGltZS1pbmZvLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLCtCQUF5QztBQUV6QyxpREFBMkQ7QUFDM0QsNkZBQTZGO0FBQzdGLE1BQU0sZ0JBQWdCLEdBQUcsQ0FBQyxVQUFVLEVBQUUscUJBQXFCLEVBQUUseUJBQXlCLEVBQUUsMkJBQTJCLEVBQUUsT0FBTyxDQUFDLENBQUM7QUFDOUgscURBQXFEO0FBQ3JELE1BQU0sa0JBQWtCLEdBQUcsQ0FBQyxVQUFVLENBQUMsQ0FBQztBQUN4Qzs7R0FFRztBQUNILFNBQWdCLHlCQUF5QjtJQUNyQyxNQUFNLFNBQVMsR0FFWCxFQUFFLENBQUM7SUFDUCxLQUFLLE1BQU0sUUFBUSxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxFQUFFO1FBQy9DLE1BQU0sR0FBRyxHQUFHLGNBQWMsQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUNyQyxJQUFJLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQyxPQUFPLEVBQUU7WUFDckIsU0FBUyxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDO1NBQ3JDO0tBQ0o7SUFDRCxpRUFBaUU7SUFDakUsS0FBSyxNQUFNLElBQUksSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxFQUFFO1FBQ3ZDLElBQUksVUFBVSxHQUFHLEtBQUssQ0FBQztRQUN2QixLQUFLLE1BQU0sS0FBSyxJQUFJLGdCQUFnQixFQUFFO1lBQ2xDLElBQUksSUFBSSxDQUFDLFVBQVUsQ0FBQyxHQUFHLEtBQUssR0FBRyxDQUFDLEVBQUU7Z0JBQzlCLFVBQVUsR0FBRyxJQUFJLENBQUM7YUFDckI7U0FDSjtRQUNELFVBQVUsR0FBRyxVQUFVLElBQUksa0JBQWtCLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQzdELElBQUksQ0FBQyxVQUFVLEVBQUU7WUFDYixPQUFPLFNBQVMsQ0FBQyxJQUFJLENBQUMsQ0FBQztTQUMxQjtLQUNKO0lBQ0QsMkJBQTJCO0lBQzNCLFNBQVMsQ0FBQyxjQUFjLENBQUMsR0FBRyxtQkFBbUIsRUFBRSxDQUFDO0lBQ2xELE9BQU8sRUFBRSxTQUFTLEVBQUUsQ0FBQztBQUN6QixDQUFDO0FBMUJELDhEQTBCQztBQUNEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztHQXVCRztBQUNILFNBQVMsY0FBYyxDQUFDLFFBQWdCO0lBS3BDLE1BQU0sR0FBRyxHQUFHLE9BQU8sQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLENBQUM7SUFDcEMsSUFBSSxFQUFDLEdBQUcsYUFBSCxHQUFHLHVCQUFILEdBQUcsQ0FBRSxLQUFLLENBQUEsRUFBRTtRQUNiLCtFQUErRTtRQUMvRSxxQ0FBcUM7UUFDckMsaURBQWlEO1FBQ2pELDJHQUEyRztRQUMzRyxPQUFPLFNBQVMsQ0FBQztLQUNwQjtJQUNELGlHQUFpRztJQUNqRyxNQUFNLEtBQUssR0FBRyxHQUFHLGFBQUgsR0FBRyx1QkFBSCxHQUFHLENBQUUsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDLElBQVksRUFBRSxFQUFFLENBQUMsZUFBUSxDQUFDLElBQUksQ0FBQyxLQUFLLGNBQWMsQ0FBQyxDQUFDLENBQUMsY0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUN6RyxJQUFJO1FBQ0EsTUFBTSxXQUFXLEdBQUcsT0FBTyxDQUFDLE9BQU87UUFDbkMsMkZBQTJGO1FBQzNGLG9CQUFnQixJQUFJLEVBQUUsQ0FBQyxDQUFDLENBQUMsZ0JBQWdCLENBQUMsQ0FBQyxDQUFDLGNBQWMsRUFBRSxFQUFFLEtBQUssRUFBRSxDQUFDLENBQUM7UUFDdkUsaUVBQWlFO1FBQ2pFLE9BQU8sT0FBTyxDQUFDLFdBQVcsQ0FBQyxDQUFDO0tBQy9CO0lBQ0QsT0FBTyxDQUFDLEVBQUU7UUFDTixPQUFPLFNBQVMsQ0FBQztLQUNwQjtBQUNMLENBQUM7QUFDRCxTQUFTLG1CQUFtQjtJQUN4QixJQUFJLFNBQVMsR0FBRyxPQUFPLENBQUMsR0FBRyxDQUFDLFVBQVUsQ0FBQztJQUN2QywyRUFBMkU7SUFDM0Usa0NBQWtDO0lBQ2xDLElBQUksQ0FBQyxTQUFTLEVBQUU7UUFDWixTQUFTLEdBQUcsV0FBVyxPQUFPLENBQUMsT0FBTyxFQUFFLENBQUM7S0FDNUM7SUFDRCxPQUFPLFNBQVMsQ0FBQztBQUNyQixDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgYmFzZW5hbWUsIGRpcm5hbWUgfSBmcm9tICdwYXRoJztcbmltcG9ydCAqIGFzIGN4c2NoZW1hIGZyb20gXCIuLi8uLi8uLi9jbG91ZC1hc3NlbWJseS1zY2hlbWFcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2Nsb3VkLWFzc2VtYmx5LXNjaGVtYSdcbmltcG9ydCB7IG1ham9yIGFzIG5vZGVNYWpvclZlcnNpb24gfSBmcm9tICcuL25vZGUtdmVyc2lvbic7XG4vLyBsaXN0IG9mIE5QTSBzY29wZXMgaW5jbHVkZWQgaW4gdmVyc2lvbiByZXBvcnRpbmcgZS5nLiBAYXdzLWNkayBhbmQgQGF3cy1zb2x1dGlvbnMta29uc3RydWtcbmNvbnN0IFdISVRFTElTVF9TQ09QRVMgPSBbJ0Bhd3MtY2RrJywgJ0Bhd3MtY2RrLWNvbnRhaW5lcnMnLCAnQGF3cy1zb2x1dGlvbnMta29uc3RydWsnLCAnQGF3cy1zb2x1dGlvbnMtY29uc3RydWN0cycsICdAYW16biddO1xuLy8gbGlzdCBvZiBOUE0gcGFja2FnZXMgaW5jbHVkZWQgaW4gdmVyc2lvbiByZXBvcnRpbmdcbmNvbnN0IFdISVRFTElTVF9QQUNLQUdFUyA9IFsnYXdzLXJmZGsnXTtcbi8qKlxuICogUmV0dXJucyBhIGxpc3Qgb2YgbG9hZGVkIG1vZHVsZXMgYW5kIHRoZWlyIHZlcnNpb25zLlxuICovXG5leHBvcnQgZnVuY3Rpb24gY29sbGVjdFJ1bnRpbWVJbmZvcm1hdGlvbigpOiBjeHNjaGVtYS5SdW50aW1lSW5mbyB7XG4gICAgY29uc3QgbGlicmFyaWVzOiB7XG4gICAgICAgIFtuYW1lOiBzdHJpbmddOiBzdHJpbmc7XG4gICAgfSA9IHt9O1xuICAgIGZvciAoY29uc3QgZmlsZU5hbWUgb2YgT2JqZWN0LmtleXMocmVxdWlyZS5jYWNoZSkpIHtcbiAgICAgICAgY29uc3QgcGtnID0gZmluZE5wbVBhY2thZ2UoZmlsZU5hbWUpO1xuICAgICAgICBpZiAocGtnICYmICFwa2cucHJpdmF0ZSkge1xuICAgICAgICAgICAgbGlicmFyaWVzW3BrZy5uYW1lXSA9IHBrZy52ZXJzaW9uO1xuICAgICAgICB9XG4gICAgfVxuICAgIC8vIGluY2x1ZGUgb25seSBsaWJyYXJpZXMgdGhhdCBhcmUgaW4gdGhlIHdoaXRlbGlzdExpYnJhcmllcyBsaXN0XG4gICAgZm9yIChjb25zdCBuYW1lIG9mIE9iamVjdC5rZXlzKGxpYnJhcmllcykpIHtcbiAgICAgICAgbGV0IGZvdW5kTWF0Y2ggPSBmYWxzZTtcbiAgICAgICAgZm9yIChjb25zdCBzY29wZSBvZiBXSElURUxJU1RfU0NPUEVTKSB7XG4gICAgICAgICAgICBpZiAobmFtZS5zdGFydHNXaXRoKGAke3Njb3BlfS9gKSkge1xuICAgICAgICAgICAgICAgIGZvdW5kTWF0Y2ggPSB0cnVlO1xuICAgICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICAgIGZvdW5kTWF0Y2ggPSBmb3VuZE1hdGNoIHx8IFdISVRFTElTVF9QQUNLQUdFUy5pbmNsdWRlcyhuYW1lKTtcbiAgICAgICAgaWYgKCFmb3VuZE1hdGNoKSB7XG4gICAgICAgICAgICBkZWxldGUgbGlicmFyaWVzW25hbWVdO1xuICAgICAgICB9XG4gICAgfVxuICAgIC8vIGFkZCBqc2lpIHJ1bnRpbWUgdmVyc2lvblxuICAgIGxpYnJhcmllc1snanNpaS1ydW50aW1lJ10gPSBnZXRKc2lpQWdlbnRWZXJzaW9uKCk7XG4gICAgcmV0dXJuIHsgbGlicmFyaWVzIH07XG59XG4vKipcbiAqIERldGVybWluZXMgd2hpY2ggTlBNIG1vZHVsZSBhIGdpdmVuIGxvYWRlZCBqYXZhc2NyaXB0IGZpbGUgaXMgZnJvbS5cbiAqXG4gKiBUaGUgb25seSBpbmZyb21hdGlvbiB0aGF0IGlzIGF2YWlsYWJsZSBsb2NhbGx5IGlzIGEgbGlzdCBvZiBKYXZhc2NyaXB0IGZpbGVzLFxuICogYW5kIGV2ZXJ5IHNvdXJjZSBmaWxlIGlzIGFzc29jaWF0ZWQgd2l0aCBhIHNlYXJjaCBwYXRoIHRvIHJlc29sdmUgdGhlIGZ1cnRoZXJcbiAqIGBgcmVxdWlyZWBgIGNhbGxzIG1hZGUgZnJvbSB0aGVyZSwgd2hpY2ggaW5jbHVkZXMgaXRzIG93biBkaXJlY3Rvcnkgb24gZGlzayxcbiAqIGFuZCBwYXJlbnQgZGlyZWN0b3JpZXMgLSBmb3IgZXhhbXBsZTpcbiAqXG4gKiBbICcuLi5yZXBvL3BhY2thZ2VzL2F3cy1jZGstcmVzb3VyY2VzL2xpYi9jZm4vbm9kZV9tb2R1bGVzJyxcbiAqICAgJy4uLnJlcG8vcGFja2FnZXMvYXdzLWNkay1yZXNvdXJjZXMvbGliL25vZGVfbW9kdWxlcycsXG4gKiAgICcuLi5yZXBvL3BhY2thZ2VzL2F3cy1jZGstcmVzb3VyY2VzL25vZGVfbW9kdWxlcycsXG4gKiAgICcuLi5yZXBvL3BhY2thZ2VzL25vZGVfbW9kdWxlcycsXG4gKiAgIC8vIGV0Yy4uLlxuICogXVxuICpcbiAqIFdlIGFyZSBsb29raW5nIGZvciBgYHBhY2thZ2UuanNvbmBgIHRoYXQgaXMgYW55d2hlcmUgaW4gdGhlIHRyZWUsIGV4Y2VwdCBpdCdzXG4gKiBpbiB0aGUgcGFyZW50IGRpcmVjdG9yeSwgbm90IGluIHRoZSBgYG5vZGVfbW9kdWxlc2BgIGRpcmVjdG9yeS4gRm9yIHRoaXNcbiAqIHJlYXNvbiwgd2Ugc3RyaXAgdGhlIGBgL25vZGVfbW9kdWxlc2BgIHN1ZmZpeCBvZmYgZWFjaCBwYXRoIGFuZCB1c2UgcmVndWxhclxuICogbW9kdWxlIHJlc29sdXRpb24gdG8gb2J0YWluIGEgcmVmZXJlbmNlIHRvIGBgcGFja2FnZS5qc29uYGAuXG4gKlxuICogQHBhcmFtIGZpbGVOYW1lIGEgamF2YXNjcmlwdCBmaWxlIG5hbWUuXG4gKiBAcmV0dXJucyB0aGUgTlBNIG1vZHVsZSBpbmZvcyAoYWthIGBgcGFja2FnZS5qc29uYGAgY29udGVudHMpLCBvclxuICogICAgICBgYHVuZGVmaW5lZGBgIGlmIHRoZSBsb29rdXAgd2FzIHVuc3VjY2Vzc2Z1bC5cbiAqL1xuZnVuY3Rpb24gZmluZE5wbVBhY2thZ2UoZmlsZU5hbWU6IHN0cmluZyk6IHtcbiAgICBuYW1lOiBzdHJpbmc7XG4gICAgdmVyc2lvbjogc3RyaW5nO1xuICAgIHByaXZhdGU/OiBib29sZWFuO1xufSB8IHVuZGVmaW5lZCB7XG4gICAgY29uc3QgbW9kID0gcmVxdWlyZS5jYWNoZVtmaWxlTmFtZV07XG4gICAgaWYgKCFtb2Q/LnBhdGhzKSB7XG4gICAgICAgIC8vIHNvbWV0aW1lcyB0aGlzIGNhbiBiZSB1bmRlZmluZWQuIGZvciBleGFtcGxlIHdoZW4gcXVlcnlpbmcgZm9yIC5qc29uIG1vZHVsZXNcbiAgICAgICAgLy8gaW5zaWRlIGEgamVzdCBydW50aW1lIGVudmlyb25tZW50LlxuICAgICAgICAvLyBzZWUgaHR0cHM6Ly9naXRodWIuY29tL2F3cy9hd3MtY2RrL2lzc3Vlcy83NjU3XG4gICAgICAgIC8vIHBvdGVudGlhbGx5IHdlIGNhbiByZW1vdmUgdGhpcyBpZiBpdCB0dXJucyBvdXQgdG8gYmUgYSBidWcgaW4gaG93IGplc3QgaW1wbGVtZW50ZWQgdGhlICdyZXF1aXJlJyBtb2R1bGUuXG4gICAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgfVxuICAgIC8vIEZvciBhbnkgcGF0aCBpbiBgYG1vZC5wYXRoc2BgIHRoYXQgaXMgYSBub2RlX21vZHVsZXMgZm9sZGVyLCB1c2UgaXRzIHBhcmVudCBkaXJlY3RvcnkgaW5zdGVhZC5cbiAgICBjb25zdCBwYXRocyA9IG1vZD8ucGF0aHMubWFwKChwYXRoOiBzdHJpbmcpID0+IGJhc2VuYW1lKHBhdGgpID09PSAnbm9kZV9tb2R1bGVzJyA/IGRpcm5hbWUocGF0aCkgOiBwYXRoKTtcbiAgICB0cnkge1xuICAgICAgICBjb25zdCBwYWNrYWdlUGF0aCA9IHJlcXVpcmUucmVzb2x2ZShcbiAgICAgICAgLy8gUmVzb2x1dGlvbiBiZWhhdmlvciBjaGFuZ2VkIGluIG5vZGUgMTIuMC4wIC0gaHR0cHM6Ly9naXRodWIuY29tL25vZGVqcy9ub2RlL2lzc3Vlcy8yNzU4M1xuICAgICAgICBub2RlTWFqb3JWZXJzaW9uID49IDEyID8gJy4vcGFja2FnZS5qc29uJyA6ICdwYWNrYWdlLmpzb24nLCB7IHBhdGhzIH0pO1xuICAgICAgICAvLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmUgQHR5cGVzY3JpcHQtZXNsaW50L25vLXJlcXVpcmUtaW1wb3J0c1xuICAgICAgICByZXR1cm4gcmVxdWlyZShwYWNrYWdlUGF0aCk7XG4gICAgfVxuICAgIGNhdGNoIChlKSB7XG4gICAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgfVxufVxuZnVuY3Rpb24gZ2V0SnNpaUFnZW50VmVyc2lvbigpIHtcbiAgICBsZXQganNpaUFnZW50ID0gcHJvY2Vzcy5lbnYuSlNJSV9BR0VOVDtcbiAgICAvLyBpZiBKU0lJX0FHRU5UIGlzIG5vdCBzcGVjaWZpZWQsIHdlIHdpbGwgYXNzdW1lIHRoaXMgaXMgYSBub2RlLmpzIHJ1bnRpbWVcbiAgICAvLyBhbmQgcGx1ZyBpbiBvdXIgbm9kZS5qcyB2ZXJzaW9uXG4gICAgaWYgKCFqc2lpQWdlbnQpIHtcbiAgICAgICAganNpaUFnZW50ID0gYG5vZGUuanMvJHtwcm9jZXNzLnZlcnNpb259YDtcbiAgICB9XG4gICAgcmV0dXJuIGpzaWlBZ2VudDtcbn1cbiJdfQ==