"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NestedStack = void 0;
const crypto = require("crypto");
const constructs_1 = require("constructs");
const assets_1 = require("./assets");
const cfn_fn_1 = require("./cfn-fn");
const cfn_pseudo_1 = require("./cfn-pseudo");
const cloudformation_generated_1 = require("./cloudformation.generated");
const lazy_1 = require("./lazy");
const names_1 = require("./names");
const stack_1 = require("./stack");
const stack_synthesizers_1 = require("./stack-synthesizers");
const token_1 = require("./token");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const construct_compat_1 = require("./construct-compat");
const NESTED_STACK_SYMBOL = Symbol.for('@aws-cdk/core.NestedStack');
/**
 * (experimental) A CloudFormation nested stack.
 *
 * When you apply template changes to update a top-level stack, CloudFormation
 * updates the top-level stack and initiates an update to its nested stacks.
 * CloudFormation updates the resources of modified nested stacks, but does not
 * update the resources of unmodified nested stacks.
 *
 * Furthermore, this stack will not be treated as an independent deployment
 * artifact (won't be listed in "cdk list" or deployable through "cdk deploy"),
 * but rather only synthesized as a template and uploaded as an asset to S3.
 *
 * Cross references of resource attributes between the parent stack and the
 * nested stack will automatically be translated to stack parameters and
 * outputs.
 *
 * @experimental
 */
class NestedStack extends stack_1.Stack {
    /**
     * @experimental
     */
    constructor(scope, id, props = {}) {
        const parentStack = findParentStack(scope);
        super(scope, id, {
            env: { account: parentStack.account, region: parentStack.region },
            synthesizer: new stack_synthesizers_1.NestedStackSynthesizer(parentStack.synthesizer),
        });
        this._parentStack = parentStack;
        // @deprecate: remove this in v2.0 (redundent)
        const parentScope = new construct_compat_1.Construct(scope, id + '.NestedStack');
        Object.defineProperty(this, NESTED_STACK_SYMBOL, { value: true });
        // this is the file name of the synthesized template file within the cloud assembly
        this.templateFile = `${names_1.Names.uniqueId(this)}.nested.template.json`;
        this.parameters = props.parameters || {};
        this.resource = new cloudformation_generated_1.CfnStack(parentScope, `${id}.NestedStackResource`, {
            templateUrl: lazy_1.Lazy.stringValue({ produce: () => this._templateUrl || '<unresolved>' }),
            parameters: lazy_1.Lazy.anyValue({ produce: () => Object.keys(this.parameters).length > 0 ? this.parameters : undefined }),
            notificationArns: props.notificationArns,
            timeoutInMinutes: props.timeout ? props.timeout.toMinutes() : undefined,
        });
        this.nestedStackResource = this.resource;
        // context-aware stack name: if resolved from within this stack, return AWS::StackName
        // if resolved from the outer stack, use the { Ref } of the AWS::CloudFormation::Stack resource
        // which resolves the ARN of the stack. We need to extract the stack name, which is the second
        // component after splitting by "/"
        this._contextualStackName = this.contextualAttribute(cfn_pseudo_1.Aws.STACK_NAME, cfn_fn_1.Fn.select(1, cfn_fn_1.Fn.split('/', this.resource.ref)));
        this._contextualStackId = this.contextualAttribute(cfn_pseudo_1.Aws.STACK_ID, this.resource.ref);
    }
    /**
     * (experimental) Checks if `x` is an object of type `NestedStack`.
     *
     * @experimental
     */
    static isNestedStack(x) {
        return x != null && typeof (x) === 'object' && NESTED_STACK_SYMBOL in x;
    }
    /**
     * (experimental) An attribute that represents the name of the nested stack.
     *
     * This is a context aware attribute:
     * - If this is referenced from the parent stack, it will return a token that parses the name from the stack ID.
     * - If this is referenced from the context of the nested stack, it will return `{ "Ref": "AWS::StackName" }`
     *
     * @experimental
     * @attribute true
     * @example
     *
     * mystack-mynestedstack-sggfrhxhum7w
     */
    get stackName() {
        return this._contextualStackName;
    }
    /**
     * (experimental) An attribute that represents the ID of the stack.
     *
     * This is a context aware attribute:
     * - If this is referenced from the parent stack, it will return `{ "Ref": "LogicalIdOfNestedStackResource" }`.
     * - If this is referenced from the context of the nested stack, it will return `{ "Ref": "AWS::StackId" }`
     *
     * @experimental
     * @attribute true
     * @example
     *
     * "arn:aws:cloudformation:us-east-2:123456789012:stack/mystack-mynestedstack-sggfrhxhum7w/f449b250-b969-11e0-a185-5081d0136786"
     */
    get stackId() {
        return this._contextualStackId;
    }
    /**
     * (experimental) Assign a value to one of the nested stack parameters.
     *
     * @param name The parameter name (ID).
     * @param value The value to assign.
     * @experimental
     */
    setParameter(name, value) {
        this.parameters[name] = value;
    }
    /**
     * Defines an asset at the parent stack which represents the template of this
     * nested stack.
     *
     * This private API is used by `App.prepare()` within a loop that rectifies
     * references every time an asset is added. This is because (at the moment)
     * assets are addressed using CloudFormation parameters.
     *
     * @returns `true` if a new asset was added or `false` if an asset was
     * previously added. When this returns `true`, App will do another reference
     * rectification cycle.
     *
     * @internal
     */
    _prepareTemplateAsset() {
        if (this._templateUrl) {
            return false;
        }
        const cfn = JSON.stringify(this._toCloudFormation());
        const templateHash = crypto.createHash('sha256').update(cfn).digest('hex');
        const templateLocation = this._parentStack.addFileAsset({
            packaging: assets_1.FileAssetPackaging.FILE,
            sourceHash: templateHash,
            fileName: this.templateFile,
        });
        // if bucketName/objectKey are cfn parameters from a stack other than the parent stack, they will
        // be resolved as cross-stack references like any other (see "multi" tests).
        this._templateUrl = `https://s3.${this._parentStack.region}.${this._parentStack.urlSuffix}/${templateLocation.bucketName}/${templateLocation.objectKey}`;
        return true;
    }
    contextualAttribute(innerValue, outerValue) {
        return token_1.Token.asString({
            resolve: (context) => {
                if (stack_1.Stack.of(context.scope) === this) {
                    return innerValue;
                }
                else {
                    return outerValue;
                }
            },
        });
    }
}
exports.NestedStack = NestedStack;
/**
 * Validates the scope for a nested stack. Nested stacks must be defined within the scope of another `Stack`.
 */
function findParentStack(scope) {
    if (!scope) {
        throw new Error('Nested stacks cannot be defined as a root construct');
    }
    const parentStack = constructs_1.Node.of(scope).scopes.reverse().find(p => stack_1.Stack.isStack(p));
    if (!parentStack) {
        throw new Error('Nested stacks must be defined within scope of another non-nested stack');
    }
    return parentStack;
}
//# sourceMappingURL=data:application/json;base64,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