"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Lazy = void 0;
const debug_1 = require("./debug");
const stack_trace_1 = require("./stack-trace");
const token_1 = require("./token");
/**
 * (experimental) Lazily produce a value.
 *
 * Can be used to return a string, list or numeric value whose actual value
 * will only be calculated later, during synthesis.
 *
 * @experimental
 */
class Lazy {
    /**
     * @experimental
     */
    static stringValue(producer, options = {}) {
        return token_1.Token.asString(new LazyString(producer), options);
    }
    /**
     * @experimental
     */
    static numberValue(producer) {
        return token_1.Token.asNumber(new LazyNumber(producer));
    }
    /**
     * @experimental
     */
    static listValue(producer, options = {}) {
        return token_1.Token.asList(new LazyList(producer, options), options);
    }
    /**
     * @experimental
     */
    static anyValue(producer, options = {}) {
        return new LazyAny(producer, options);
    }
    constructor() {
    }
}
exports.Lazy = Lazy;
class LazyBase {
    constructor() {
        // Stack trace capture is conditionned to `debugModeEnabled()`, because
        // lazies can be created in a fairly thrashy way, and the stack traces are
        // large and slow to obtain; but are mostly useful only when debugging a
        // resolution issue.
        this.creationStack = debug_1.debugModeEnabled()
            ? stack_trace_1.captureStackTrace(this.constructor)
            : [`Execute again with ${debug_1.CDK_DEBUG}=true to capture stack traces`];
    }
    toString() {
        return token_1.Token.asString(this);
    }
    /**
     * Turn this Token into JSON
     *
     * Called automatically when JSON.stringify() is called on a Token.
     */
    toJSON() {
        return '<unresolved-lazy>';
    }
}
class LazyString extends LazyBase {
    constructor(producer) {
        super();
        this.producer = producer;
    }
    resolve(context) {
        return this.producer.produce(context);
    }
}
class LazyNumber extends LazyBase {
    constructor(producer) {
        super();
        this.producer = producer;
    }
    resolve(context) {
        return this.producer.produce(context);
    }
}
class LazyList extends LazyBase {
    constructor(producer, options = {}) {
        super();
        this.producer = producer;
        this.options = options;
    }
    resolve(context) {
        const ret = this.producer.produce(context);
        if (ret !== undefined && ret.length === 0 && this.options.omitEmpty) {
            return undefined;
        }
        return ret;
    }
}
class LazyAny extends LazyBase {
    constructor(producer, options = {}) {
        super();
        this.producer = producer;
        this.options = options;
    }
    resolve(context) {
        const ret = this.producer.produce(context);
        if (Array.isArray(ret) && ret.length === 0 && this.options.omitEmptyArray) {
            return undefined;
        }
        return ret;
    }
}
//# sourceMappingURL=data:application/json;base64,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