"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnParser = exports.CfnParsingContext = exports.FromCloudFormation = void 0;
const cfn_fn_1 = require("./cfn-fn");
const cfn_pseudo_1 = require("./cfn-pseudo");
const cfn_resource_policy_1 = require("./cfn-resource-policy");
const lazy_1 = require("./lazy");
const cfn_reference_1 = require("./private/cfn-reference");
const token_1 = require("./token");
const util_1 = require("./util");
/**
 * This class contains static methods called when going from
 * translated values received from {@link CfnParser.parseValue}
 * to the actual L1 properties -
 * things like changing IResolvable to the appropriate type
 * (string, string array, or number), etc.
 *
 * While this file not exported from the module
 * (to not make it part of the public API),
 * it is directly referenced in the generated L1 code.
 *
 * @experimental
 */
class FromCloudFormation {
    // nothing to for any but return it
    static getAny(value) { return value; }
    static getBoolean(value) {
        if (typeof value === 'string') {
            // CloudFormation allows passing strings as boolean
            switch (value) {
                case 'true': return true;
                case 'false': return false;
                default: throw new Error(`Expected 'true' or 'false' for boolean value, got: '${value}'`);
            }
        }
        // in all other cases, just return the value,
        // and let a validator handle if it's not a boolean
        return value;
    }
    static getDate(value) {
        // if the date is a deploy-time value, just return it
        if (token_1.isResolvableObject(value)) {
            return value;
        }
        // if the date has been given as a string, convert it
        if (typeof value === 'string') {
            return new Date(value);
        }
        // all other cases - just return the value,
        // if it's not a Date, a validator should catch it
        return value;
    }
    // won't always return a string; if the input can't be resolved to a string,
    // the input will be returned.
    static getString(value) {
        // if the string is a deploy-time value, serialize it to a Token
        if (token_1.isResolvableObject(value)) {
            return value.toString();
        }
        // CloudFormation treats numbers and strings interchangeably;
        // so, if we get a number here, convert it to a string
        if (typeof value === 'number') {
            return value.toString();
        }
        // in all other cases, just return the input,
        // and let a validator handle it if it's not a string
        return value;
    }
    // won't always return a number; if the input can't be parsed to a number,
    // the input will be returned.
    static getNumber(value) {
        // if the string is a deploy-time value, serialize it to a Token
        if (token_1.isResolvableObject(value)) {
            return token_1.Token.asNumber(value);
        }
        // return a number, if the input can be parsed as one
        if (typeof value === 'string') {
            const parsedValue = parseFloat(value);
            if (!isNaN(parsedValue)) {
                return parsedValue;
            }
        }
        // otherwise return the input,
        // and let a validator handle it if it's not a number
        return value;
    }
    static getStringArray(value) {
        // if the array is a deploy-time value, serialize it to a Token
        if (token_1.isResolvableObject(value)) {
            return token_1.Token.asList(value);
        }
        // in all other cases, delegate to the standard mapping logic
        return this.getArray(this.getString)(value);
    }
    static getArray(mapper) {
        return (value) => {
            if (!Array.isArray(value)) {
                // break the type system, and just return the given value,
                // which hopefully will be reported as invalid by the validator
                // of the property we're transforming
                // (unless it's a deploy-time value,
                // which we can't map over at build time anyway)
                return value;
            }
            return value.map(mapper);
        };
    }
    static getMap(mapper) {
        return (value) => {
            if (typeof value !== 'object') {
                // if the input is not a map (= object in JS land),
                // just return it, and let the validator of this property handle it
                // (unless it's a deploy-time value,
                // which we can't map over at build time anyway)
                return value;
            }
            const ret = {};
            for (const [key, val] of Object.entries(value)) {
                ret[key] = mapper(val);
            }
            return ret;
        };
    }
    static getCfnTag(tag) {
        return tag == null
            ? {} // break the type system - this should be detected at runtime by a tag validator
            : {
                key: tag.Key,
                value: tag.Value,
            };
    }
    /**
     * Return a function that, when applied to a value, will return the first validly deserialized one
     */
    static getTypeUnion(validators, mappers) {
        return (value) => {
            for (let i = 0; i < validators.length; i++) {
                const candidate = mappers[i](value);
                if (validators[i](candidate).isSuccess) {
                    return candidate;
                }
            }
            // if nothing matches, just return the input unchanged, and let validators catch it
            return value;
        };
    }
}
exports.FromCloudFormation = FromCloudFormation;
/**
 * The context in which the parsing is taking place.
 *
 * Some fragments of CloudFormation templates behave differently than others
 * (for example, the 'Conditions' sections treats { "Condition": "NameOfCond" }
 * differently than the 'Resources' section).
 * This enum can be used to change the created {@link CfnParser} behavior,
 * based on the template context.
 */
var CfnParsingContext;
(function (CfnParsingContext) {
    /** We're currently parsing the 'Conditions' section. */
    CfnParsingContext[CfnParsingContext["CONDITIONS"] = 0] = "CONDITIONS";
    /** We're currently parsing the 'Rules' section. */
    CfnParsingContext[CfnParsingContext["RULES"] = 1] = "RULES";
})(CfnParsingContext = exports.CfnParsingContext || (exports.CfnParsingContext = {}));
/**
 * This class contains methods for translating from a pure CFN value
 * (like a JS object { "Ref": "Bucket" })
 * to a form CDK understands
 * (like Fn.ref('Bucket')).
 *
 * While this file not exported from the module
 * (to not make it part of the public API),
 * it is directly referenced in the generated L1 code,
 * so any renames of it need to be reflected in cfn2ts/codegen.ts as well.
 *
 * @experimental
 */
class CfnParser {
    constructor(options) {
        this.options = options;
    }
    handleAttributes(resource, resourceAttributes, logicalId) {
        var _a;
        const cfnOptions = resource.cfnOptions;
        cfnOptions.creationPolicy = this.parseCreationPolicy(resourceAttributes.CreationPolicy);
        cfnOptions.updatePolicy = this.parseUpdatePolicy(resourceAttributes.UpdatePolicy);
        cfnOptions.deletionPolicy = this.parseDeletionPolicy(resourceAttributes.DeletionPolicy);
        cfnOptions.updateReplacePolicy = this.parseDeletionPolicy(resourceAttributes.UpdateReplacePolicy);
        cfnOptions.version = this.parseValue(resourceAttributes.Version);
        cfnOptions.description = this.parseValue(resourceAttributes.Description);
        cfnOptions.metadata = this.parseValue(resourceAttributes.Metadata);
        // handle Condition
        if (resourceAttributes.Condition) {
            const condition = this.finder.findCondition(resourceAttributes.Condition);
            if (!condition) {
                throw new Error(`Resource '${logicalId}' uses Condition '${resourceAttributes.Condition}' that doesn't exist`);
            }
            cfnOptions.condition = condition;
        }
        // handle DependsOn
        resourceAttributes.DependsOn = (_a = resourceAttributes.DependsOn) !== null && _a !== void 0 ? _a : [];
        const dependencies = Array.isArray(resourceAttributes.DependsOn) ?
            resourceAttributes.DependsOn : [resourceAttributes.DependsOn];
        for (const dep of dependencies) {
            const depResource = this.finder.findResource(dep);
            if (!depResource) {
                throw new Error(`Resource '${logicalId}' depends on '${dep}' that doesn't exist`);
            }
            resource.node.addDependency(depResource);
        }
    }
    parseCreationPolicy(policy) {
        if (typeof policy !== 'object') {
            return undefined;
        }
        // change simple JS values to their CDK equivalents
        policy = this.parseValue(policy);
        return util_1.undefinedIfAllValuesAreEmpty({
            autoScalingCreationPolicy: parseAutoScalingCreationPolicy(policy.AutoScalingCreationPolicy),
            resourceSignal: parseResourceSignal(policy.ResourceSignal),
        });
        function parseAutoScalingCreationPolicy(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return util_1.undefinedIfAllValuesAreEmpty({
                minSuccessfulInstancesPercent: FromCloudFormation.getNumber(p.MinSuccessfulInstancesPercent),
            });
        }
        function parseResourceSignal(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return util_1.undefinedIfAllValuesAreEmpty({
                count: FromCloudFormation.getNumber(p.Count),
                timeout: FromCloudFormation.getString(p.Timeout),
            });
        }
    }
    parseUpdatePolicy(policy) {
        if (typeof policy !== 'object') {
            return undefined;
        }
        // change simple JS values to their CDK equivalents
        policy = this.parseValue(policy);
        return util_1.undefinedIfAllValuesAreEmpty({
            autoScalingReplacingUpdate: parseAutoScalingReplacingUpdate(policy.AutoScalingReplacingUpdate),
            autoScalingRollingUpdate: parseAutoScalingRollingUpdate(policy.AutoScalingRollingUpdate),
            autoScalingScheduledAction: parseAutoScalingScheduledAction(policy.AutoScalingScheduledAction),
            codeDeployLambdaAliasUpdate: parseCodeDeployLambdaAliasUpdate(policy.CodeDeployLambdaAliasUpdate),
            enableVersionUpgrade: FromCloudFormation.getBoolean(policy.EnableVersionUpgrade),
            useOnlineResharding: FromCloudFormation.getBoolean(policy.UseOnlineResharding),
        });
        function parseAutoScalingReplacingUpdate(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return util_1.undefinedIfAllValuesAreEmpty({
                willReplace: p.WillReplace,
            });
        }
        function parseAutoScalingRollingUpdate(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return util_1.undefinedIfAllValuesAreEmpty({
                maxBatchSize: FromCloudFormation.getNumber(p.MaxBatchSize),
                minInstancesInService: FromCloudFormation.getNumber(p.MinInstancesInService),
                minSuccessfulInstancesPercent: FromCloudFormation.getNumber(p.MinSuccessfulInstancesPercent),
                pauseTime: FromCloudFormation.getString(p.PauseTime),
                suspendProcesses: FromCloudFormation.getStringArray(p.SuspendProcesses),
                waitOnResourceSignals: FromCloudFormation.getBoolean(p.WaitOnResourceSignals),
            });
        }
        function parseCodeDeployLambdaAliasUpdate(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return {
                beforeAllowTrafficHook: FromCloudFormation.getString(p.BeforeAllowTrafficHook),
                afterAllowTrafficHook: FromCloudFormation.getString(p.AfterAllowTrafficHook),
                applicationName: FromCloudFormation.getString(p.ApplicationName),
                deploymentGroupName: FromCloudFormation.getString(p.DeploymentGroupName),
            };
        }
        function parseAutoScalingScheduledAction(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return util_1.undefinedIfAllValuesAreEmpty({
                ignoreUnmodifiedGroupSizeProperties: p.IgnoreUnmodifiedGroupSizeProperties,
            });
        }
    }
    parseDeletionPolicy(policy) {
        switch (policy) {
            case null: return undefined;
            case undefined: return undefined;
            case 'Delete': return cfn_resource_policy_1.CfnDeletionPolicy.DELETE;
            case 'Retain': return cfn_resource_policy_1.CfnDeletionPolicy.RETAIN;
            case 'Snapshot': return cfn_resource_policy_1.CfnDeletionPolicy.SNAPSHOT;
            default: throw new Error(`Unrecognized DeletionPolicy '${policy}'`);
        }
    }
    parseValue(cfnValue) {
        // == null captures undefined as well
        if (cfnValue == null) {
            return undefined;
        }
        // if we have any late-bound values,
        // just return them
        if (token_1.isResolvableObject(cfnValue)) {
            return cfnValue;
        }
        if (Array.isArray(cfnValue)) {
            return cfnValue.map(el => this.parseValue(el));
        }
        if (typeof cfnValue === 'object') {
            // an object can be either a CFN intrinsic, or an actual object
            const cfnIntrinsic = this.parseIfCfnIntrinsic(cfnValue);
            if (cfnIntrinsic !== undefined) {
                return cfnIntrinsic;
            }
            const ret = {};
            for (const [key, val] of Object.entries(cfnValue)) {
                ret[key] = this.parseValue(val);
            }
            return ret;
        }
        // in all other cases, just return the input
        return cfnValue;
    }
    get finder() {
        return this.options.finder;
    }
    parseIfCfnIntrinsic(object) {
        const key = this.looksLikeCfnIntrinsic(object);
        switch (key) {
            case undefined:
                return undefined;
            case 'Ref': {
                const refTarget = object[key];
                const specialRef = this.specialCaseRefs(refTarget);
                if (specialRef !== undefined) {
                    return specialRef;
                }
                else {
                    const refElement = this.finder.findRefTarget(refTarget);
                    if (!refElement) {
                        throw new Error(`Element used in Ref expression with logical ID: '${refTarget}' not found`);
                    }
                    return cfn_reference_1.CfnReference.for(refElement, 'Ref');
                }
            }
            case 'Fn::GetAtt': {
                const value = object[key];
                let logicalId, attributeName, stringForm;
                // Fn::GetAtt takes as arguments either a string...
                if (typeof value === 'string') {
                    // ...in which case the logical ID and the attribute name are separated with '.'
                    const dotIndex = value.indexOf('.');
                    if (dotIndex === -1) {
                        throw new Error(`Short-form Fn::GetAtt must contain a '.' in its string argument, got: '${value}'`);
                    }
                    logicalId = value.substr(0, dotIndex);
                    attributeName = value.substr(dotIndex + 1); // the +1 is to skip the actual '.'
                    stringForm = true;
                }
                else {
                    // ...or a 2-element list
                    logicalId = value[0];
                    attributeName = value[1];
                    stringForm = false;
                }
                const target = this.finder.findResource(logicalId);
                if (!target) {
                    throw new Error(`Resource used in GetAtt expression with logical ID: '${logicalId}' not found`);
                }
                return cfn_reference_1.CfnReference.for(target, attributeName, stringForm ? cfn_reference_1.ReferenceRendering.GET_ATT_STRING : undefined);
            }
            case 'Fn::Join': {
                // Fn::Join takes a 2-element list as its argument,
                // where the first element is the delimiter,
                // and the second is the list of elements to join
                const value = this.parseValue(object[key]);
                // wrap the array as a Token,
                // as otherwise Fn.join() will try to concatenate
                // the non-token parts,
                // causing a diff with the original template
                return cfn_fn_1.Fn.join(value[0], lazy_1.Lazy.listValue({ produce: () => value[1] }));
            }
            case 'Fn::Cidr': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.cidr(value[0], value[1], value[2]);
            }
            case 'Fn::FindInMap': {
                const value = this.parseValue(object[key]);
                // the first argument to FindInMap is the mapping name
                const mapping = this.finder.findMapping(value[0]);
                if (!mapping) {
                    throw new Error(`Mapping used in FindInMap expression with name '${value[0]}' was not found in the template`);
                }
                return cfn_fn_1.Fn.findInMap(mapping.logicalId, value[1], value[2]);
            }
            case 'Fn::Select': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.select(value[0], value[1]);
            }
            case 'Fn::GetAZs': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.getAzs(value);
            }
            case 'Fn::ImportValue': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.importValue(value);
            }
            case 'Fn::Split': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.split(value[0], value[1]);
            }
            case 'Fn::Transform': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.transform(value.Name, value.Parameters);
            }
            case 'Fn::Base64': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.base64(value);
            }
            case 'Fn::If': {
                // Fn::If takes a 3-element list as its argument,
                // where the first element is the name of a Condition
                const value = this.parseValue(object[key]);
                const condition = this.finder.findCondition(value[0]);
                if (!condition) {
                    throw new Error(`Condition '${value[0]}' used in an Fn::If expression does not exist in the template`);
                }
                return cfn_fn_1.Fn.conditionIf(condition.logicalId, value[1], value[2]);
            }
            case 'Fn::Equals': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.conditionEquals(value[0], value[1]);
            }
            case 'Fn::And': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.conditionAnd(...value);
            }
            case 'Fn::Not': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.conditionNot(value[0]);
            }
            case 'Fn::Or': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.conditionOr(...value);
            }
            case 'Fn::Sub': {
                const value = this.parseValue(object[key]);
                let fnSubString;
                let map;
                if (typeof value === 'string') {
                    fnSubString = value;
                    map = undefined;
                }
                else {
                    fnSubString = value[0];
                    map = value[1];
                }
                return cfn_fn_1.Fn.sub(this.parseFnSubString(fnSubString, map), map);
            }
            case 'Condition': {
                // a reference to a Condition from another Condition
                const condition = this.finder.findCondition(object[key]);
                if (!condition) {
                    throw new Error(`Referenced Condition with name '${object[key]}' was not found in the template`);
                }
                return { Condition: condition.logicalId };
            }
            default:
                if (this.options.context === CfnParsingContext.RULES) {
                    return this.handleRulesIntrinsic(key, object);
                }
                else {
                    throw new Error(`Unsupported CloudFormation function '${key}'`);
                }
        }
    }
    looksLikeCfnIntrinsic(object) {
        const objectKeys = Object.keys(object);
        // a CFN intrinsic is always an object with a single key
        if (objectKeys.length !== 1) {
            return undefined;
        }
        const key = objectKeys[0];
        return key === 'Ref' || key.startsWith('Fn::') ||
            // special intrinsic only available in the 'Conditions' section
            (this.options.context === CfnParsingContext.CONDITIONS && key === 'Condition')
            ? key
            : undefined;
    }
    parseFnSubString(value, map = {}) {
        const leftBrace = value.indexOf('${');
        const rightBrace = value.indexOf('}') + 1;
        // don't include left and right braces when searching for the target of the reference
        if (leftBrace === -1 || leftBrace >= rightBrace) {
            return value;
        }
        const leftHalf = value.substring(0, leftBrace);
        const rightHalf = value.substring(rightBrace);
        const refTarget = value.substring(leftBrace + 2, rightBrace - 1).trim();
        if (refTarget[0] === '!') {
            return value.substring(0, rightBrace) + this.parseFnSubString(rightHalf, map);
        }
        // lookup in map
        if (refTarget in map) {
            return leftHalf + '${' + refTarget + '}' + this.parseFnSubString(rightHalf, map);
        }
        // since it's not in the map, check if it's a pseudo parameter
        const specialRef = this.specialCaseSubRefs(refTarget);
        if (specialRef !== undefined) {
            return leftHalf + specialRef + this.parseFnSubString(rightHalf, map);
        }
        const dotIndex = refTarget.indexOf('.');
        const isRef = dotIndex === -1;
        if (isRef) {
            const refElement = this.finder.findRefTarget(refTarget);
            if (!refElement) {
                throw new Error(`Element referenced in Fn::Sub expression with logical ID: '${refTarget}' was not found in the template`);
            }
            return leftHalf + cfn_reference_1.CfnReference.for(refElement, 'Ref', cfn_reference_1.ReferenceRendering.FN_SUB).toString() + this.parseFnSubString(rightHalf, map);
        }
        else {
            const targetId = refTarget.substring(0, dotIndex);
            const refResource = this.finder.findResource(targetId);
            if (!refResource) {
                throw new Error(`Resource referenced in Fn::Sub expression with logical ID: '${targetId}' was not found in the template`);
            }
            const attribute = refTarget.substring(dotIndex + 1);
            return leftHalf + cfn_reference_1.CfnReference.for(refResource, attribute, cfn_reference_1.ReferenceRendering.FN_SUB).toString() + this.parseFnSubString(rightHalf, map);
        }
    }
    handleRulesIntrinsic(key, object) {
        // Rules have their own set of intrinsics:
        // https://docs.aws.amazon.com/servicecatalog/latest/adminguide/intrinsic-function-reference-rules.html
        switch (key) {
            case 'Fn::ValueOf': {
                // ValueOf is special,
                // as it takes the name of a Parameter as its first argument
                const value = this.parseValue(object[key]);
                const parameterName = value[0];
                if (parameterName in this.parameters) {
                    // since ValueOf returns the value of a specific attribute,
                    // fail here - this substitution is not allowed
                    throw new Error(`Cannot substitute parameter '${parameterName}' used in Fn::ValueOf expression with attribute '${value[1]}'`);
                }
                const param = this.finder.findRefTarget(parameterName);
                if (!param) {
                    throw new Error(`Rule references parameter '${parameterName}' which was not found in the template`);
                }
                // create an explicit IResolvable,
                // as Fn.valueOf() returns a string,
                // which is incorrect
                // (Fn::ValueOf can also return an array)
                return lazy_1.Lazy.anyValue({ produce: () => ({ 'Fn::ValueOf': [param.logicalId, value[1]] }) });
            }
            default:
                // I don't want to hard-code the list of supported Rules-specific intrinsics in this function;
                // so, just return undefined here,
                // and they will be treated as a regular JSON object
                return undefined;
        }
    }
    specialCaseRefs(value) {
        if (value in this.parameters) {
            return this.parameters[value];
        }
        switch (value) {
            case 'AWS::AccountId': return cfn_pseudo_1.Aws.ACCOUNT_ID;
            case 'AWS::Region': return cfn_pseudo_1.Aws.REGION;
            case 'AWS::Partition': return cfn_pseudo_1.Aws.PARTITION;
            case 'AWS::URLSuffix': return cfn_pseudo_1.Aws.URL_SUFFIX;
            case 'AWS::NotificationARNs': return cfn_pseudo_1.Aws.NOTIFICATION_ARNS;
            case 'AWS::StackId': return cfn_pseudo_1.Aws.STACK_ID;
            case 'AWS::StackName': return cfn_pseudo_1.Aws.STACK_NAME;
            case 'AWS::NoValue': return cfn_pseudo_1.Aws.NO_VALUE;
            default: return undefined;
        }
    }
    specialCaseSubRefs(value) {
        if (value in this.parameters) {
            return this.parameters[value];
        }
        return value.indexOf('::') === -1 ? undefined : '${' + value + '}';
    }
    get parameters() {
        return this.options.parameters || {};
    }
}
exports.CfnParser = CfnParser;
//# sourceMappingURL=data:application/json;base64,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