"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnParameter = void 0;
const cfn_element_1 = require("./cfn-element");
const cfn_reference_1 = require("./private/cfn-reference");
const token_1 = require("./token");
/**
 * (experimental) A CloudFormation parameter.
 *
 * Use the optional Parameters section to customize your templates.
 * Parameters enable you to input custom values to your template each time you create or
 * update a stack.
 *
 * @experimental
 */
class CfnParameter extends cfn_element_1.CfnElement {
    /**
     * (experimental) Creates a parameter construct.
     *
     * Note that the name (logical ID) of the parameter will derive from it's `coname` and location
     * within the stack. Therefore, it is recommended that parameters are defined at the stack level.
     *
     * @param scope The parent construct.
     * @param props The parameter properties.
     * @experimental
     */
    constructor(scope, id, props = {}) {
        super(scope, id);
        this._type = props.type || 'String';
        this._default = props.default;
        this._allowedPattern = props.allowedPattern;
        this._allowedValues = props.allowedValues;
        this._constraintDescription = props.constraintDescription;
        this._description = props.description;
        this._maxLength = props.maxLength;
        this._maxValue = props.maxValue;
        this._minLength = props.minLength;
        this._minValue = props.minValue;
        this._noEcho = props.noEcho;
    }
    /**
     * (experimental) The data type for the parameter (DataType).
     *
     * @default String
     * @experimental
     */
    get type() {
        return this._type;
    }
    /**
     * (experimental) The data type for the parameter (DataType).
     *
     * @default String
     * @experimental
     */
    set type(type) {
        this._type = type;
    }
    /**
     * (experimental) A value of the appropriate type for the template to use if no value is specified when a stack is created.
     *
     * If you define constraints for the parameter, you must specify
     * a value that adheres to those constraints.
     *
     * @default - No default value for parameter.
     * @experimental
     */
    get default() {
        return this._default;
    }
    /**
     * (experimental) A value of the appropriate type for the template to use if no value is specified when a stack is created.
     *
     * If you define constraints for the parameter, you must specify
     * a value that adheres to those constraints.
     *
     * @default - No default value for parameter.
     * @experimental
     */
    set default(value) {
        this._default = value;
    }
    /**
     * (experimental) A regular expression that represents the patterns to allow for String types.
     *
     * @default - No constraints on patterns allowed for parameter.
     * @experimental
     */
    get allowedPattern() {
        return this._allowedPattern;
    }
    /**
     * (experimental) A regular expression that represents the patterns to allow for String types.
     *
     * @default - No constraints on patterns allowed for parameter.
     * @experimental
     */
    set allowedPattern(pattern) {
        this._allowedPattern = pattern;
    }
    /**
     * (experimental) An array containing the list of values allowed for the parameter.
     *
     * @default - No constraints on values allowed for parameter.
     * @experimental
     */
    get allowedValues() {
        return this._allowedValues;
    }
    /**
     * (experimental) An array containing the list of values allowed for the parameter.
     *
     * @default - No constraints on values allowed for parameter.
     * @experimental
     */
    set allowedValues(values) {
        this._allowedValues = values;
    }
    /**
     * (experimental) A string that explains a constraint when the constraint is violated.
     *
     * For example, without a constraint description, a parameter that has an allowed
     * pattern of [A-Za-z0-9]+ displays the following error message when the user specifies
     * an invalid value:
     *
     * @default - No description with customized error message when user specifies invalid values.
     * @experimental
     */
    get constraintDescription() {
        return this._constraintDescription;
    }
    /**
     * (experimental) A string that explains a constraint when the constraint is violated.
     *
     * For example, without a constraint description, a parameter that has an allowed
     * pattern of [A-Za-z0-9]+ displays the following error message when the user specifies
     * an invalid value:
     *
     * @default - No description with customized error message when user specifies invalid values.
     * @experimental
     */
    set constraintDescription(desc) {
        this._constraintDescription = desc;
    }
    /**
     * (experimental) A string of up to 4000 characters that describes the parameter.
     *
     * @default - No description for the parameter.
     * @experimental
     */
    get description() {
        return this._description;
    }
    /**
     * (experimental) A string of up to 4000 characters that describes the parameter.
     *
     * @default - No description for the parameter.
     * @experimental
     */
    set description(desc) {
        this._description = desc;
    }
    /**
     * (experimental) An integer value that determines the largest number of characters you want to allow for String types.
     *
     * @default - None.
     * @experimental
     */
    get maxLength() {
        return this._maxLength;
    }
    /**
     * (experimental) An integer value that determines the largest number of characters you want to allow for String types.
     *
     * @default - None.
     * @experimental
     */
    set maxLength(len) {
        this._maxLength = len;
    }
    /**
     * (experimental) An integer value that determines the smallest number of characters you want to allow for String types.
     *
     * @default - None.
     * @experimental
     */
    get minLength() {
        return this._minLength;
    }
    /**
     * (experimental) An integer value that determines the smallest number of characters you want to allow for String types.
     *
     * @default - None.
     * @experimental
     */
    set minLength(len) {
        this._minLength = len;
    }
    /**
     * (experimental) A numeric value that determines the largest numeric value you want to allow for Number types.
     *
     * @default - None.
     * @experimental
     */
    get maxValue() {
        return this._maxValue;
    }
    /**
     * (experimental) A numeric value that determines the largest numeric value you want to allow for Number types.
     *
     * @default - None.
     * @experimental
     */
    set maxValue(len) {
        this._maxValue = len;
    }
    /**
     * (experimental) A numeric value that determines the smallest numeric value you want to allow for Number types.
     *
     * @default - None.
     * @experimental
     */
    get minValue() {
        return this._minValue;
    }
    /**
     * (experimental) A numeric value that determines the smallest numeric value you want to allow for Number types.
     *
     * @default - None.
     * @experimental
     */
    set minValue(len) {
        this._minValue = len;
    }
    /**
     * (experimental) Indicates if this parameter is configured with "NoEcho" enabled.
     *
     * @experimental
     */
    get noEcho() {
        return !!this._noEcho;
    }
    /**
     * (experimental) Indicates if this parameter is configured with "NoEcho" enabled.
     *
     * @experimental
     */
    set noEcho(echo) {
        this._noEcho = echo;
    }
    /**
     * (experimental) The parameter value as a Token.
     *
     * @experimental
     */
    get value() {
        return cfn_reference_1.CfnReference.for(this, 'Ref');
    }
    /**
     * (experimental) The parameter value, if it represents a string.
     *
     * @experimental
     */
    get valueAsString() {
        if (!isStringType(this.type) && !isNumberType(this.type)) {
            throw new Error(`Parameter type (${this.type}) is not a string or number type`);
        }
        return token_1.Token.asString(this.value);
    }
    /**
     * (experimental) The parameter value, if it represents a string list.
     *
     * @experimental
     */
    get valueAsList() {
        if (!isListType(this.type)) {
            throw new Error(`Parameter type (${this.type}) is not a string list type`);
        }
        return token_1.Token.asList(this.value);
    }
    /**
     * (experimental) The parameter value, if it represents a number.
     *
     * @experimental
     */
    get valueAsNumber() {
        if (!isNumberType(this.type)) {
            throw new Error(`Parameter type (${this.type}) is not a number type`);
        }
        return token_1.Token.asNumber(this.value);
    }
    /**
     * @internal
     */
    _toCloudFormation() {
        return {
            Parameters: {
                [this.logicalId]: {
                    Type: this.type,
                    Default: this.default,
                    AllowedPattern: this.allowedPattern,
                    AllowedValues: this.allowedValues,
                    ConstraintDescription: this.constraintDescription,
                    Description: this.description,
                    MaxLength: this.maxLength,
                    MaxValue: this.maxValue,
                    MinLength: this.minLength,
                    MinValue: this.minValue,
                    NoEcho: this._noEcho,
                },
            },
        };
    }
    /**
     * @experimental
     */
    resolve(_context) {
        return this.value;
    }
}
exports.CfnParameter = CfnParameter;
/**
 * Whether the given parameter type looks like a list type
 */
function isListType(type) {
    return type.indexOf('List<') >= 0 || type.indexOf('CommaDelimitedList') >= 0;
}
/**
 * Whether the given parameter type looks like a number type
 */
function isNumberType(type) {
    return type === 'Number';
}
/**
 * Whether the given parameter type looks like a string type
 */
function isStringType(type) {
    return !isListType(type) && !isNumberType(type);
}
//# sourceMappingURL=data:application/json;base64,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