"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnRefElement = exports.CfnElement = void 0;
const cxschema = require("../../cloud-assembly-schema"); // Automatically re-written from '@aws-cdk/cloud-assembly-schema'
const constructs_1 = require("constructs");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const construct_compat_1 = require("./construct-compat");
const lazy_1 = require("./lazy");
const CFN_ELEMENT_SYMBOL = Symbol.for('@aws-cdk/core.CfnElement');
/**
 * (experimental) An element of a CloudFormation stack.
 *
 * @experimental
 */
class CfnElement extends construct_compat_1.Construct {
    /**
     * (experimental) Creates an entity and binds it to a tree.
     *
     * Note that the root of the tree must be a Stack object (not just any Root).
     *
     * @param scope The parent construct.
     * @experimental
     */
    constructor(scope, id) {
        super(scope, id);
        Object.defineProperty(this, CFN_ELEMENT_SYMBOL, { value: true });
        this.stack = stack_1.Stack.of(this);
        this.logicalId = lazy_1.Lazy.stringValue({ produce: () => this.synthesizeLogicalId() }, {
            displayHint: `${notTooLong(constructs_1.Node.of(this).path)}.LogicalID`,
        });
        constructs_1.Node.of(this).addMetadata(cxschema.ArtifactMetadataEntryType.LOGICAL_ID, this.logicalId, this.constructor);
    }
    /**
     * (experimental) Returns `true` if a construct is a stack element (i.e. part of the synthesized cloudformation template).
     *
     * Uses duck-typing instead of `instanceof` to allow stack elements from different
     * versions of this library to be included in the same stack.
     *
     * @returns The construct as a stack element or undefined if it is not a stack element.
     * @experimental
     */
    static isCfnElement(x) {
        return CFN_ELEMENT_SYMBOL in x;
    }
    /**
     * (experimental) Overrides the auto-generated logical ID with a specific ID.
     *
     * @param newLogicalId The new logical ID to use for this stack element.
     * @experimental
     */
    overrideLogicalId(newLogicalId) {
        this._logicalIdOverride = newLogicalId;
    }
    /**
     * @returns the stack trace of the point where this Resource was created from, sourced
     * from the +metadata+ entry typed +aws:cdk:logicalId+, and with the bottom-most
     * node +internal+ entries filtered.
     * @experimental
     */
    get creationStack() {
        const trace = constructs_1.Node.of(this).metadata.find(md => md.type === cxschema.ArtifactMetadataEntryType.LOGICAL_ID).trace;
        if (!trace) {
            return [];
        }
        return filterStackTrace(trace);
        function filterStackTrace(stack) {
            const result = Array.of(...stack);
            while (result.length > 0 && shouldFilter(result[result.length - 1])) {
                result.pop();
            }
            // It's weird if we filtered everything, so return the whole stack...
            return result.length === 0 ? stack : result;
        }
        function shouldFilter(str) {
            return str.match(/[^(]+\(internal\/.*/) !== null;
        }
    }
    /**
     * Called during synthesize to render the logical ID of this element. If
     * `overrideLogicalId` was it will be used, otherwise, we will allocate the
     * logical ID through the stack.
     */
    synthesizeLogicalId() {
        if (this._logicalIdOverride) {
            return this._logicalIdOverride;
        }
        else {
            return this.stack.getLogicalId(this);
        }
    }
}
exports.CfnElement = CfnElement;
/**
 * (experimental) Base class for referenceable CloudFormation constructs which are not Resources.
 *
 * These constructs are things like Conditions and Parameters, can be
 * referenced by taking the `.ref` attribute.
 *
 * Resource constructs do not inherit from CfnRefElement because they have their
 * own, more specific types returned from the .ref attribute. Also, some
 * resources aren't referenceable at all (such as BucketPolicies or GatewayAttachments).
 *
 * @experimental
 */
class CfnRefElement extends CfnElement {
    /**
     * (experimental) Return a string that will be resolved to a CloudFormation `{ Ref }` for this element.
     *
     * If, by any chance, the intrinsic reference of a resource is not a string, you could
     * coerce it to an IResolvable through `Lazy.any({ produce: resource.ref })`.
     *
     * @experimental
     */
    get ref() {
        return token_1.Token.asString(cfn_reference_1.CfnReference.for(this, 'Ref'));
    }
}
exports.CfnRefElement = CfnRefElement;
function notTooLong(x) {
    if (x.length < 100) {
        return x;
    }
    return x.substr(0, 47) + '...' + x.substr(x.length - 47);
}
const cfn_reference_1 = require("./private/cfn-reference");
const stack_1 = require("./stack");
const token_1 = require("./token");
//# sourceMappingURL=data:application/json;base64,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