"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Manifest = exports.VERSION_MISMATCH = void 0;
const fs = require("fs");
const jsonschema = require("jsonschema");
const semver = require("semver");
const assembly = require("./cloud-assembly");
/* eslint-disable @typescript-eslint/no-var-requires */
/* eslint-disable @typescript-eslint/no-require-imports */
// this prefix is used by the CLI to identify this specific error.
// in which case we want to instruct the user to upgrade his CLI.
// see exec.ts#createAssembly
exports.VERSION_MISMATCH = 'Cloud assembly schema version mismatch';
const ASSETS_SCHEMA = require('../schema/assets.schema.json');
const ASSEMBLY_SCHEMA = require('../schema/cloud-assembly.schema.json');
/**
 * Version is shared for both manifests
 */
const SCHEMA_VERSION = require('../schema/cloud-assembly.version.json').version;
/**
 * (experimental) Protocol utility class.
 *
 * @experimental
 */
class Manifest {
    /**
     * (experimental) Validates and saves the cloud assembly manifest to file.
     *
     * @param manifest - manifest.
     * @param filePath - output file path.
     * @experimental
     */
    static saveAssemblyManifest(manifest, filePath) {
        Manifest.saveManifest(manifest, filePath, ASSEMBLY_SCHEMA, Manifest.patchStackTagsOnWrite);
    }
    /**
     * (experimental) Load and validates the cloud assembly manifest from file.
     *
     * @param filePath - path to the manifest file.
     * @experimental
     */
    static loadAssemblyManifest(filePath) {
        return Manifest.loadManifest(filePath, ASSEMBLY_SCHEMA, Manifest.patchStackTagsOnRead);
    }
    /**
     * (experimental) Validates and saves the asset manifest to file.
     *
     * @param manifest - manifest.
     * @param filePath - output file path.
     * @experimental
     */
    static saveAssetManifest(manifest, filePath) {
        Manifest.saveManifest(manifest, filePath, ASSETS_SCHEMA, Manifest.patchStackTagsOnRead);
    }
    /**
     * (experimental) Load and validates the asset manifest from file.
     *
     * @param filePath - path to the manifest file.
     * @experimental
     */
    static loadAssetManifest(filePath) {
        return this.loadManifest(filePath, ASSETS_SCHEMA);
    }
    /**
     * (experimental) Fetch the current schema version number.
     *
     * @experimental
     */
    static version() {
        return SCHEMA_VERSION;
    }
    /**
     * (deprecated) Deprecated.
     *
     * @deprecated use `saveAssemblyManifest()`
     */
    static save(manifest, filePath) { return this.saveAssemblyManifest(manifest, filePath); }
    /**
     * (deprecated) Deprecated.
     *
     * @deprecated use `loadAssemblyManifest()`
     */
    static load(filePath) { return this.loadAssemblyManifest(filePath); }
    static validate(manifest, schema) {
        function parseVersion(version) {
            const ver = semver.valid(version);
            if (!ver) {
                throw new Error(`Invalid semver string: "${version}"`);
            }
            return ver;
        }
        const maxSupported = parseVersion(Manifest.version());
        const actual = parseVersion(manifest.version);
        // first validate the version should be accepted.
        if (semver.gt(actual, maxSupported)) {
            // we use a well known error prefix so that the CLI can identify this specific error
            // and print some more context to the user.
            throw new Error(`${exports.VERSION_MISMATCH}: Maximum schema version supported is ${maxSupported}, but found ${actual}`);
        }
        // now validate the format is good.
        const validator = new jsonschema.Validator();
        const result = validator.validate(manifest, schema, {
            // does exist but is not in the TypeScript definitions
            nestedErrors: true,
            allowUnknownAttributes: false,
        });
        if (!result.valid) {
            throw new Error(`Invalid assembly manifest:\n${result}`);
        }
    }
    static saveManifest(manifest, filePath, schema, preprocess) {
        let withVersion = { ...manifest, version: Manifest.version() };
        Manifest.validate(withVersion, schema);
        if (preprocess) {
            withVersion = preprocess(withVersion);
        }
        fs.writeFileSync(filePath, JSON.stringify(withVersion, undefined, 2));
    }
    static loadManifest(filePath, schema, preprocess) {
        let obj = JSON.parse(fs.readFileSync(filePath, { encoding: 'utf-8' }));
        if (preprocess) {
            obj = preprocess(obj);
        }
        Manifest.validate(obj, schema);
        return obj;
    }
    /**
     * This requires some explaining...
     *
     * We previously used `{ Key, Value }` for the object that represents a stack tag. (Notice the casing)
     * @link https://github.com/aws/aws-cdk/blob/v1.27.0/packages/aws-cdk/lib/api/cxapp/stacks.ts#L427.
     *
     * When that object moved to this package, it had to be JSII compliant, which meant the property
     * names must be `camelCased`, and not `PascalCased`. This meant it no longer matches the structure in the `manifest.json` file.
     * In order to support current manifest files, we have to translate the `PascalCased` representation to the new `camelCased` one.
     *
     * Note that the serialization itself still writes `PascalCased` because it relates to how CloudFormation expects it.
     *
     * Ideally, we would start writing the `camelCased` and translate to how CloudFormation expects it when needed. But this requires nasty
     * backwards-compatibility code and it just doesn't seem to be worth the effort.
     */
    static patchStackTagsOnRead(manifest) {
        return Manifest.replaceStackTags(manifest, tags => tags.map((diskTag) => ({
            key: diskTag.Key,
            value: diskTag.Value,
        })));
    }
    /**
     * See explanation on `patchStackTagsOnRead`
     *
     * Translate stack tags metadata if it has the "right" casing.
     */
    static patchStackTagsOnWrite(manifest) {
        return Manifest.replaceStackTags(manifest, tags => tags.map(memTag => 
        // Might already be uppercased (because stack synthesis generates it in final form yet)
        ('Key' in memTag ? memTag : { Key: memTag.key, Value: memTag.value })));
    }
    /**
     * Recursively replace stack tags in the stack metadata
     */
    static replaceStackTags(manifest, fn) {
        // Need to add in the `noUndefined`s because otherwise jest snapshot tests are going to freak out
        // about the keys with values that are `undefined` (even though they would never be JSON.stringified)
        return noUndefined({
            ...manifest,
            artifacts: mapValues(manifest.artifacts, artifact => {
                if (artifact.type !== assembly.ArtifactType.AWS_CLOUDFORMATION_STACK) {
                    return artifact;
                }
                return noUndefined({
                    ...artifact,
                    metadata: mapValues(artifact.metadata, metadataEntries => metadataEntries.map(metadataEntry => {
                        if (metadataEntry.type !== assembly.ArtifactMetadataEntryType.STACK_TAGS || !metadataEntry.data) {
                            return metadataEntry;
                        }
                        return {
                            ...metadataEntry,
                            data: fn(metadataEntry.data),
                        };
                    })),
                });
            }),
        });
    }
    constructor() { }
}
exports.Manifest = Manifest;
function mapValues(xs, fn) {
    if (!xs) {
        return undefined;
    }
    const ret = {};
    for (const [k, v] of Object.entries(xs)) {
        ret[k] = fn(v);
    }
    return ret;
}
function noUndefined(xs) {
    const ret = {};
    for (const [k, v] of Object.entries(xs)) {
        if (v !== undefined) {
            ret[k] = v;
        }
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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