"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3Code = exports.InlineCode = exports.AssetCode = exports.Code = void 0;
const fs = require("fs");
const path = require("path");
const s3_assets = require("../../aws-s3-assets"); // Automatically re-written from '@aws-cdk/aws-s3-assets'
/**
 * (experimental) The code the canary should execute.
 *
 * @experimental
 */
class Code {
    /**
     * (experimental) Specify code inline.
     *
     * @param code The actual handler code (limited to 4KiB).
     * @returns `InlineCode` with inline code.
     * @experimental
     */
    static fromInline(code) {
        return new InlineCode(code);
    }
    /**
     * (experimental) Specify code from a local path.
     *
     * Path must include the folder structure `nodejs/node_modules/myCanaryFilename.js`.
     *
     * @param assetPath Either a directory or a .zip file.
     * @returns `AssetCode` associated with the specified path.
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_WritingCanary.html#CloudWatch_Synthetics_Canaries_write_from_scratch
     * @experimental
     */
    static fromAsset(assetPath, options) {
        return new AssetCode(assetPath, options);
    }
    /**
     * (experimental) Specify code from an s3 bucket.
     *
     * The object in the s3 bucket must be a .zip file that contains
     * the structure `nodejs/node_modules/myCanaryFilename.js`.
     *
     * @param bucket The S3 bucket.
     * @param key The object key.
     * @param objectVersion Optional S3 object version.
     * @returns `S3Code` associated with the specified S3 object.
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_WritingCanary.html#CloudWatch_Synthetics_Canaries_write_from_scratch
     * @experimental
     */
    static fromBucket(bucket, key, objectVersion) {
        return new S3Code(bucket, key, objectVersion);
    }
}
exports.Code = Code;
/**
 * (experimental) Canary code from an Asset.
 *
 * @experimental
 */
class AssetCode extends Code {
    /**
     * @param assetPath The path to the asset file or directory.
     * @experimental
     */
    constructor(assetPath, options) {
        super();
        this.assetPath = assetPath;
        this.options = options;
        if (!fs.existsSync(this.assetPath)) {
            throw new Error(`${this.assetPath} is not a valid path`);
        }
    }
    /**
     * (experimental) Called when the canary is initialized to allow this object to bind to the stack, add resources and have fun.
     *
     * @experimental
     */
    bind(scope, handler) {
        this.validateCanaryAsset(handler);
        // If the same AssetCode is used multiple times, retain only the first instantiation.
        if (!this.asset) {
            this.asset = new s3_assets.Asset(scope, 'Code', {
                path: this.assetPath,
                ...this.options,
            });
        }
        return {
            s3Location: {
                bucketName: this.asset.s3BucketName,
                objectKey: this.asset.s3ObjectKey,
            },
        };
    }
    /**
     * Validates requirements specified by the canary resource. For example, the canary code with handler `index.handler`
     * must be found in the file structure `nodejs/node_modules/index.js`.
     *
     * Requires path to be either zip file or directory.
     * Requires asset directory to have the structure 'nodejs/node_modules'.
     * Requires canary file to be directly inside node_modules folder.
     * Requires canary file name matches the handler name.
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_WritingCanary.html
     *
     * @param handler the canary handler
     */
    validateCanaryAsset(handler) {
        if (path.extname(this.assetPath) !== '.zip') {
            if (!fs.lstatSync(this.assetPath).isDirectory()) {
                throw new Error(`Asset must be a .zip file or a directory (${this.assetPath})`);
            }
            const filename = `${handler.split('.')[0]}.js`;
            if (!fs.existsSync(path.join(this.assetPath, 'nodejs', 'node_modules', filename))) {
                throw new Error(`The canary resource requires that the handler is present at "nodejs/node_modules/${filename}" but not found at ${this.assetPath} (https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_WritingCanary.html#CloudWatch_Synthetics_Canaries_write_from_scratch)`);
            }
        }
    }
}
exports.AssetCode = AssetCode;
/**
 * (experimental) Canary code from an inline string.
 *
 * @experimental
 */
class InlineCode extends Code {
    /**
     * @experimental
     */
    constructor(code) {
        super();
        this.code = code;
        if (code.length === 0) {
            throw new Error('Canary inline code cannot be empty');
        }
    }
    /**
     * (experimental) Called when the canary is initialized to allow this object to bind to the stack, add resources and have fun.
     *
     * @experimental
     */
    bind(_scope, handler) {
        if (handler !== 'index.handler') {
            throw new Error(`The handler for inline code must be "index.handler" (got "${handler}")`);
        }
        return {
            inlineCode: this.code,
        };
    }
}
exports.InlineCode = InlineCode;
/**
 * (experimental) S3 bucket path to the code zip file.
 *
 * @experimental
 */
class S3Code extends Code {
    /**
     * @experimental
     */
    constructor(bucket, key, objectVersion) {
        super();
        this.bucket = bucket;
        this.key = key;
        this.objectVersion = objectVersion;
    }
    /**
     * (experimental) Called when the canary is initialized to allow this object to bind to the stack, add resources and have fun.
     *
     * @experimental
     */
    bind(_scope, _handler) {
        return {
            s3Location: {
                bucketName: this.bucket.bucketName,
                objectKey: this.key,
                objectVersion: this.objectVersion,
            },
        };
    }
}
exports.S3Code = S3Code;
//# sourceMappingURL=data:application/json;base64,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