"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Canary = exports.Runtime = exports.Test = void 0;
const crypto = require("crypto");
const aws_cloudwatch_1 = require("../../aws-cloudwatch"); // Automatically re-written from '@aws-cdk/aws-cloudwatch'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const s3 = require("../../aws-s3"); // Automatically re-written from '@aws-cdk/aws-s3'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const synthetics_generated_1 = require("./synthetics.generated");
/**
 * (experimental) Specify a test that the canary should run.
 *
 * @experimental
 */
class Test {
    /**
     * Construct a Test property
     *
     * @param code The code that the canary should run
     * @param handler The handler of the canary
     */
    constructor(code, handler) {
        this.code = code;
        this.handler = handler;
    }
    /**
     * (experimental) Specify a custom test with your own code.
     *
     * @param options The configuration options.
     * @returns `Test` associated with the specified Code object
     * @experimental
     */
    static custom(options) {
        Test.validateHandler(options.handler);
        return new Test(options.code, options.handler);
    }
    /**
     * Verifies that the given handler ends in '.handler'. Returns the handler if successful and
     * throws an error if not.
     *
     * @param handler - the handler given by the user
     */
    static validateHandler(handler) {
        if (!handler.endsWith('.handler')) {
            throw new Error(`Canary Handler must end in '.handler' (${handler})`);
        }
        if (handler.length > 21) {
            throw new Error(`Canary Handler must be less than 21 characters (${handler})`);
        }
    }
}
exports.Test = Test;
/**
 * (experimental) Runtime options for a canary.
 *
 * @experimental
 */
class Runtime {
    /**
     * @param name The name of the runtime version.
     * @experimental
     */
    constructor(name) {
        this.name = name;
    }
}
exports.Runtime = Runtime;
/**
 * (experimental) `syn-1.0` includes the following:.
 *
 * - Synthetics library 1.0
 * - Synthetics handler code 1.0
 * - Lambda runtime Node.js 10.x
 * - Puppeteer-core version 1.14.0
 * - The Chromium version that matches Puppeteer-core 1.14.0
 *
 * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_Library.html#CloudWatch_Synthetics_runtimeversion-1.0
 * @experimental
 */
Runtime.SYNTHETICS_1_0 = new Runtime('syn-1.0');
/**
 * (experimental) `syn-nodejs-2.0` includes the following: - Lambda runtime Node.js 10.x - Puppeteer-core version 3.3.0 - Chromium version 81.0.4044.0.
 *
 * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_Library.html#CloudWatch_Synthetics_runtimeversion-2.0
 * @experimental
 */
Runtime.SYNTHETICS_NODEJS_2_0 = new Runtime('syn-nodejs-2.0');
/**
 * (experimental) Define a new Canary.
 *
 * @experimental
 */
class Canary extends cdk.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g, _h;
        if (props.canaryName && !cdk.Token.isUnresolved(props.canaryName)) {
            validateName(props.canaryName);
        }
        super(scope, id, {
            physicalName: props.canaryName || cdk.Lazy.stringValue({
                produce: () => this.generateUniqueName(),
            }),
        });
        this.artifactsBucket = (_b = (_a = props.artifactsBucketLocation) === null || _a === void 0 ? void 0 : _a.bucket) !== null && _b !== void 0 ? _b : new s3.Bucket(this, 'ArtifactsBucket', {
            encryption: s3.BucketEncryption.KMS_MANAGED,
        });
        this.role = (_c = props.role) !== null && _c !== void 0 ? _c : this.createDefaultRole((_d = props.artifactsBucketLocation) === null || _d === void 0 ? void 0 : _d.prefix);
        const resource = new synthetics_generated_1.CfnCanary(this, 'Resource', {
            artifactS3Location: this.artifactsBucket.s3UrlForObject((_e = props.artifactsBucketLocation) === null || _e === void 0 ? void 0 : _e.prefix),
            executionRoleArn: this.role.roleArn,
            startCanaryAfterCreation: (_f = props.startAfterCreation) !== null && _f !== void 0 ? _f : true,
            runtimeVersion: props.runtime.name,
            name: this.physicalName,
            schedule: this.createSchedule(props),
            failureRetentionPeriod: (_g = props.failureRetentionPeriod) === null || _g === void 0 ? void 0 : _g.toDays(),
            successRetentionPeriod: (_h = props.successRetentionPeriod) === null || _h === void 0 ? void 0 : _h.toDays(),
            code: this.createCode(props),
        });
        this.canaryId = resource.attrId;
        this.canaryState = resource.attrState;
        this.canaryName = this.getResourceNameAttribute(resource.ref);
    }
    /**
     * (experimental) Measure the Duration of a single canary run, in seconds.
     *
     * @param options - configuration options for the metric.
     * @default avg over 5 minutes
     * @experimental
     */
    metricDuration(options) {
        return this.metric('Duration', options);
    }
    /**
     * (experimental) Measure the percentage of successful canary runs.
     *
     * @param options - configuration options for the metric.
     * @default avg over 5 minutes
     * @experimental
     */
    metricSuccessPercent(options) {
        return this.metric('SuccessPercent', options);
    }
    /**
     * (experimental) Measure the number of failed canary runs over a given time period.
     *
     * @param options - configuration options for the metric.
     * @default avg over 5 minutes
     * @experimental
     */
    metricFailed(options) {
        return this.metric('Failed', options);
    }
    /**
     * @param metricName - the name of the metric
     * @param options - configuration options for the metric
     *
     * @returns a CloudWatch metric associated with the canary.
     * @default avg over 5 minutes
     */
    metric(metricName, options) {
        return new aws_cloudwatch_1.Metric({
            metricName,
            namespace: 'CloudWatchSynthetics',
            dimensions: { CanaryName: this.canaryName },
            statistic: 'avg',
            ...options,
        }).attachTo(this);
    }
    /**
     * Returns a default role for the canary
     */
    createDefaultRole(prefix) {
        // Created role will need these policies to run the Canary.
        // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-synthetics-canary.html#cfn-synthetics-canary-executionrolearn
        const policy = new iam.PolicyDocument({
            statements: [
                new iam.PolicyStatement({
                    resources: ['*'],
                    actions: ['s3:ListAllMyBuckets'],
                }),
                new iam.PolicyStatement({
                    resources: [this.artifactsBucket.arnForObjects(`${prefix ? prefix + '/*' : '*'}`)],
                    actions: ['s3:PutObject', 's3:GetBucketLocation'],
                }),
                new iam.PolicyStatement({
                    resources: ['*'],
                    actions: ['cloudwatch:PutMetricData'],
                    conditions: { StringEquals: { 'cloudwatch:namespace': 'CloudWatchSynthetics' } },
                }),
                new iam.PolicyStatement({
                    resources: ['arn:aws:logs:::*'],
                    actions: ['logs:CreateLogStream', 'logs:CreateLogGroup', 'logs:PutLogEvents'],
                }),
            ],
        });
        return new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            inlinePolicies: {
                canaryPolicy: policy,
            },
        });
    }
    /**
     * Returns the code object taken in by the canary resource.
     */
    createCode(props) {
        var _a, _b, _c;
        const codeConfig = {
            handler: props.test.handler,
            ...props.test.code.bind(this, props.test.handler),
        };
        return {
            handler: codeConfig.handler,
            script: codeConfig.inlineCode,
            s3Bucket: (_a = codeConfig.s3Location) === null || _a === void 0 ? void 0 : _a.bucketName,
            s3Key: (_b = codeConfig.s3Location) === null || _b === void 0 ? void 0 : _b.objectKey,
            s3ObjectVersion: (_c = codeConfig.s3Location) === null || _c === void 0 ? void 0 : _c.objectVersion,
        };
    }
    /**
     * Returns a canary schedule object
     */
    createSchedule(props) {
        var _a, _b, _c, _d;
        return {
            durationInSeconds: String(`${(_b = (_a = props.timeToLive) === null || _a === void 0 ? void 0 : _a.toSeconds()) !== null && _b !== void 0 ? _b : 0}`),
            expression: (_d = (_c = props.schedule) === null || _c === void 0 ? void 0 : _c.expressionString) !== null && _d !== void 0 ? _d : 'rate(5 minutes)',
        };
    }
    /**
     * Creates a unique name for the canary. The generated name is the physical ID of the canary.
     */
    generateUniqueName() {
        const name = cdk.Names.uniqueId(this).toLowerCase().replace(' ', '-');
        if (name.length <= 21) {
            return name;
        }
        else {
            return name.substring(0, 15) + nameHash(name);
        }
    }
}
exports.Canary = Canary;
/**
 * Take a hash of the given name.
 *
 * @param name the name to be hashed
 */
function nameHash(name) {
    const md5 = crypto.createHash('sha256').update(name).digest('hex');
    return md5.slice(0, 6);
}
const nameRegex = /^[0-9a-z_\-]+$/;
/**
 * Verifies that the name fits the regex expression: ^[0-9a-z_\-]+$.
 *
 * @param name - the given name of the canary
 */
function validateName(name) {
    if (name.length > 21) {
        throw new Error(`Canary name is too large, must be between 1 and 21 characters, but is ${name.length} (got "${name}")`);
    }
    if (!nameRegex.test(name)) {
        throw new Error(`Canary name must be lowercase, numbers, hyphens, or underscores (got "${name}")`);
    }
}
//# sourceMappingURL=data:application/json;base64,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