"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.renderJsonPath = exports.renderList = exports.State = void 0;
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const constructs_1 = require("constructs");
const fields_1 = require("../fields");
const types_1 = require("../types");
/**
 * (experimental) Base class for all other state classes.
 *
 * @experimental
 */
class State extends cdk.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * @experimental
         */
        this.branches = [];
        this.retries = [];
        this.catches = [];
        this.choices = [];
        this.prefixes = [];
        /**
         * States with references to this state.
         *
         * Used for finding complete connected graph that a state is part of.
         */
        this.incomingStates = [];
        this.startState = this;
        this.comment = props.comment;
        this.inputPath = props.inputPath;
        this.parameters = props.parameters;
        this.outputPath = props.outputPath;
        this.resultPath = props.resultPath;
    }
    /**
     * (experimental) Add a prefix to the stateId of all States found in a construct tree.
     *
     * @experimental
     */
    static prefixStates(root, prefix) {
        const queue = [root];
        while (queue.length > 0) {
            const el = queue.splice(0, 1)[0];
            if (isPrefixable(el)) {
                el.addPrefix(prefix);
            }
            queue.push(...constructs_1.Node.of(el).children);
        }
    }
    /**
     * (experimental) Find the set of states reachable through transitions from the given start state.
     *
     * This does not retrieve states from within sub-graphs, such as states within a Parallel state's branch.
     *
     * @experimental
     */
    static findReachableStates(start, options = {}) {
        const visited = new Set();
        const ret = new Set();
        const queue = [start];
        while (queue.length > 0) {
            const state = queue.splice(0, 1)[0];
            if (visited.has(state)) {
                continue;
            }
            visited.add(state);
            const outgoing = state.outgoingTransitions(options);
            queue.push(...outgoing);
            ret.add(state);
        }
        return Array.from(ret);
    }
    /**
     * (experimental) Find the set of end states states reachable through transitions from the given start state.
     *
     * @experimental
     */
    static findReachableEndStates(start, options = {}) {
        const visited = new Set();
        const ret = new Set();
        const queue = [start];
        while (queue.length > 0) {
            const state = queue.splice(0, 1)[0];
            if (visited.has(state)) {
                continue;
            }
            visited.add(state);
            const outgoing = state.outgoingTransitions(options);
            if (outgoing.length > 0) {
                // We can continue
                queue.push(...outgoing);
            }
            else {
                // Terminal state
                ret.add(state);
            }
        }
        return Array.from(ret);
    }
    /**
     * (experimental) Return only the states that allow chaining from an array of states.
     *
     * @experimental
     */
    static filterNextables(states) {
        return states.filter(isNextable);
    }
    /**
     * (experimental) Descriptive identifier for this chainable.
     *
     * @experimental
     */
    get id() {
        return this.node.id;
    }
    /**
     * (experimental) Tokenized string that evaluates to the state's ID.
     *
     * @experimental
     */
    get stateId() {
        return this.prefixes.concat(this.id).join('');
    }
    /**
     * (experimental) Add a prefix to the stateId of this state.
     *
     * @experimental
     */
    addPrefix(x) {
        if (x !== '') {
            this.prefixes.splice(0, 0, x);
        }
    }
    /**
     * (experimental) Register this state as part of the given graph.
     *
     * Don't call this. It will be called automatically when you work
     * with states normally.
     *
     * @experimental
     */
    bindToGraph(graph) {
        if (this.containingGraph === graph) {
            return;
        }
        if (this.containingGraph) {
            // eslint-disable-next-line max-len
            throw new Error(`Trying to use state '${this.stateId}' in ${graph}, but is already in ${this.containingGraph}. Every state can only be used in one graph.`);
        }
        this.containingGraph = graph;
        this.whenBoundToGraph(graph);
        for (const incoming of this.incomingStates) {
            incoming.bindToGraph(graph);
        }
        for (const outgoing of this.outgoingTransitions({ includeErrorHandlers: true })) {
            outgoing.bindToGraph(graph);
        }
        for (const branch of this.branches) {
            branch.registerSuperGraph(this.containingGraph);
        }
        if (!!this.iteration) {
            this.iteration.registerSuperGraph(this.containingGraph);
        }
    }
    /**
     * Add a retrier to the retry list of this state
     * @internal
     */
    _addRetry(props = {}) {
        validateErrors(props.errors);
        this.retries.push({
            ...props,
            errors: props.errors ? props.errors : [types_1.Errors.ALL],
        });
    }
    /**
     * Add an error handler to the catch list of this state
     * @internal
     */
    _addCatch(handler, props = {}) {
        validateErrors(props.errors);
        this.catches.push({
            next: handler,
            props: {
                errors: props.errors ? props.errors : [types_1.Errors.ALL],
                resultPath: props.resultPath,
            },
        });
        handler.addIncoming(this);
        if (this.containingGraph) {
            handler.bindToGraph(this.containingGraph);
        }
    }
    /**
     * (experimental) Make the indicated state the default transition of this state.
     *
     * @experimental
     */
    makeNext(next) {
        // Can't be called 'setNext' because of JSII
        if (this._next) {
            throw new Error(`State '${this.id}' already has a next state`);
        }
        this._next = next;
        next.addIncoming(this);
        if (this.containingGraph) {
            next.bindToGraph(this.containingGraph);
        }
    }
    /**
     * (experimental) Add a choice branch to this state.
     *
     * @experimental
     */
    addChoice(condition, next) {
        this.choices.push({ condition, next });
        next.startState.addIncoming(this);
        if (this.containingGraph) {
            next.startState.bindToGraph(this.containingGraph);
        }
    }
    /**
     * (experimental) Add a paralle branch to this state.
     *
     * @experimental
     */
    addBranch(branch) {
        this.branches.push(branch);
        if (this.containingGraph) {
            branch.registerSuperGraph(this.containingGraph);
        }
    }
    /**
     * (experimental) Add a map iterator to this state.
     *
     * @experimental
     */
    addIterator(iteration) {
        this.iteration = iteration;
        if (this.containingGraph) {
            iteration.registerSuperGraph(this.containingGraph);
        }
    }
    /**
     * (experimental) Make the indicated state the default choice transition of this state.
     *
     * @experimental
     */
    makeDefault(def) {
        // Can't be called 'setDefault' because of JSII
        if (this.defaultChoice) {
            throw new Error(`Choice '${this.id}' already has a default next state`);
        }
        this.defaultChoice = def;
    }
    /**
     * (experimental) Render the default next state in ASL JSON format.
     *
     * @experimental
     */
    renderNextEnd() {
        if (this._next) {
            return { Next: this._next.stateId };
        }
        else {
            return { End: true };
        }
    }
    /**
     * (experimental) Render the choices in ASL JSON format.
     *
     * @experimental
     */
    renderChoices() {
        return {
            Choices: renderList(this.choices, renderChoice),
            Default: this.defaultChoice ? this.defaultChoice.stateId : undefined,
        };
    }
    /**
     * (experimental) Render InputPath/Parameters/OutputPath in ASL JSON format.
     *
     * @experimental
     */
    renderInputOutput() {
        return {
            InputPath: renderJsonPath(this.inputPath),
            Parameters: this.parameters,
            OutputPath: renderJsonPath(this.outputPath),
        };
    }
    /**
     * (experimental) Render parallel branches in ASL JSON format.
     *
     * @experimental
     */
    renderBranches() {
        return {
            Branches: this.branches.map(b => b.toGraphJson()),
        };
    }
    /**
     * (experimental) Render map iterator in ASL JSON format.
     *
     * @experimental
     */
    renderIterator() {
        if (!this.iteration) {
            throw new Error('Iterator must not be undefined !');
        }
        return {
            Iterator: this.iteration.toGraphJson(),
        };
    }
    /**
     * (experimental) Render error recovery options in ASL JSON format.
     *
     * @experimental
     */
    renderRetryCatch() {
        return {
            Retry: renderList(this.retries, renderRetry, (a, b) => compareErrors(a.errors, b.errors)),
            Catch: renderList(this.catches, renderCatch, (a, b) => compareErrors(a.props.errors, b.props.errors)),
        };
    }
    /**
     * (experimental) Called whenever this state is bound to a graph.
     *
     * Can be overridden by subclasses.
     *
     * @experimental
     */
    whenBoundToGraph(graph) {
        graph.registerState(this);
    }
    /**
     * Add a state to the incoming list
     */
    addIncoming(source) {
        this.incomingStates.push(source);
    }
    /**
     * Return all states this state can transition to
     */
    outgoingTransitions(options) {
        const ret = new Array();
        if (this._next) {
            ret.push(this._next);
        }
        if (this.defaultChoice) {
            ret.push(this.defaultChoice);
        }
        for (const c of this.choices) {
            ret.push(c.next);
        }
        if (options.includeErrorHandlers) {
            for (const c of this.catches) {
                ret.push(c.next);
            }
        }
        return ret;
    }
}
exports.State = State;
/**
 * Render a choice transition
 */
function renderChoice(c) {
    return {
        ...c.condition.renderCondition(),
        Next: c.next.stateId,
    };
}
/**
 * Render a Retry object to ASL
 */
function renderRetry(retry) {
    return {
        ErrorEquals: retry.errors,
        IntervalSeconds: retry.interval && retry.interval.toSeconds(),
        MaxAttempts: retry.maxAttempts,
        BackoffRate: retry.backoffRate,
    };
}
/**
 * Render a Catch object to ASL
 */
function renderCatch(c) {
    return {
        ErrorEquals: c.props.errors,
        ResultPath: renderJsonPath(c.props.resultPath),
        Next: c.next.stateId,
    };
}
/**
 * Compares a list of Errors to move Errors.ALL last in a sort function
 */
function compareErrors(a, b) {
    if (a === null || a === void 0 ? void 0 : a.includes(types_1.Errors.ALL)) {
        return 1;
    }
    if (b === null || b === void 0 ? void 0 : b.includes(types_1.Errors.ALL)) {
        return -1;
    }
    return 0;
}
/**
 * Validates an errors list
 */
function validateErrors(errors) {
    if ((errors === null || errors === void 0 ? void 0 : errors.includes(types_1.Errors.ALL)) && errors.length > 1) {
        throw new Error(`${types_1.Errors.ALL} must appear alone in an error list`);
    }
}
/**
 * Render a list or return undefined for an empty list
 */
function renderList(xs, mapFn, sortFn) {
    if (xs.length === 0) {
        return undefined;
    }
    let list = xs;
    if (sortFn) {
        list = xs.sort(sortFn);
    }
    return list.map(mapFn);
}
exports.renderList = renderList;
/**
 * Render JSON path, respecting the special value DISCARD
 */
function renderJsonPath(jsonPath) {
    if (jsonPath === undefined) {
        return undefined;
    }
    if (jsonPath === fields_1.JsonPath.DISCARD) {
        return null;
    }
    if (!jsonPath.startsWith('$')) {
        throw new Error(`Expected JSON path to start with '$', got: ${jsonPath}`);
    }
    return jsonPath;
}
exports.renderJsonPath = renderJsonPath;
/**
 * Whether an object is a Prefixable
 */
function isPrefixable(x) {
    return typeof (x) === 'object' && x.addPrefix;
}
/**
 * Whether an object is INextable
 */
function isNextable(x) {
    return typeof (x) === 'object' && x.next;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic3RhdGUuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJzdGF0ZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSxxQ0FBcUMsQ0FBQyxnREFBZ0Q7QUFDdEYsMkNBQXlEO0FBRXpELHNDQUFxQztBQUVyQyxvQ0FBaUY7Ozs7OztBQXFEakYsTUFBc0IsS0FBTSxTQUFRLEdBQUcsQ0FBQyxTQUFTOzs7O0lBMEc3QyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQWlCO1FBQ3ZELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7Ozs7UUF4QkYsYUFBUSxHQUFpQixFQUFFLENBQUM7UUFPOUIsWUFBTyxHQUFpQixFQUFFLENBQUM7UUFDM0IsWUFBTyxHQUFzQixFQUFFLENBQUM7UUFDaEMsWUFBTyxHQUF1QixFQUFFLENBQUM7UUFDakMsYUFBUSxHQUFhLEVBQUUsQ0FBQztRQU96Qzs7OztXQUlHO1FBQ2MsbUJBQWMsR0FBWSxFQUFFLENBQUM7UUFHMUMsSUFBSSxDQUFDLFVBQVUsR0FBRyxJQUFJLENBQUM7UUFDdkIsSUFBSSxDQUFDLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDO1FBQzdCLElBQUksQ0FBQyxTQUFTLEdBQUcsS0FBSyxDQUFDLFNBQVMsQ0FBQztRQUNqQyxJQUFJLENBQUMsVUFBVSxHQUFHLEtBQUssQ0FBQyxVQUFVLENBQUM7UUFDbkMsSUFBSSxDQUFDLFVBQVUsR0FBRyxLQUFLLENBQUMsVUFBVSxDQUFDO1FBQ25DLElBQUksQ0FBQyxVQUFVLEdBQUcsS0FBSyxDQUFDLFVBQVUsQ0FBQztJQUN2QyxDQUFDOzs7Ozs7SUE5R00sTUFBTSxDQUFDLFlBQVksQ0FBQyxJQUFnQixFQUFFLE1BQWM7UUFDdkQsTUFBTSxLQUFLLEdBQUcsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUNyQixPQUFPLEtBQUssQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO1lBQ3JCLE1BQU0sRUFBRSxHQUFHLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBRSxDQUFDO1lBQ2xDLElBQUksWUFBWSxDQUFDLEVBQUUsQ0FBQyxFQUFFO2dCQUNsQixFQUFFLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxDQUFDO2FBQ3hCO1lBQ0QsS0FBSyxDQUFDLElBQUksQ0FBQyxHQUFHLGlCQUFJLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1NBQ3ZDO0lBQ0wsQ0FBQzs7Ozs7Ozs7SUFLTSxNQUFNLENBQUMsbUJBQW1CLENBQUMsS0FBWSxFQUFFLFVBQTRCLEVBQUU7UUFDMUUsTUFBTSxPQUFPLEdBQUcsSUFBSSxHQUFHLEVBQVMsQ0FBQztRQUNqQyxNQUFNLEdBQUcsR0FBRyxJQUFJLEdBQUcsRUFBUyxDQUFDO1FBQzdCLE1BQU0sS0FBSyxHQUFHLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDdEIsT0FBTyxLQUFLLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtZQUNyQixNQUFNLEtBQUssR0FBRyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUUsQ0FBQztZQUNyQyxJQUFJLE9BQU8sQ0FBQyxHQUFHLENBQUMsS0FBSyxDQUFDLEVBQUU7Z0JBQ3BCLFNBQVM7YUFDWjtZQUNELE9BQU8sQ0FBQyxHQUFHLENBQUMsS0FBSyxDQUFDLENBQUM7WUFDbkIsTUFBTSxRQUFRLEdBQUcsS0FBSyxDQUFDLG1CQUFtQixDQUFDLE9BQU8sQ0FBQyxDQUFDO1lBQ3BELEtBQUssQ0FBQyxJQUFJLENBQUMsR0FBRyxRQUFRLENBQUMsQ0FBQztZQUN4QixHQUFHLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxDQUFDO1NBQ2xCO1FBQ0QsT0FBTyxLQUFLLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQzNCLENBQUM7Ozs7OztJQUlNLE1BQU0sQ0FBQyxzQkFBc0IsQ0FBQyxLQUFZLEVBQUUsVUFBNEIsRUFBRTtRQUM3RSxNQUFNLE9BQU8sR0FBRyxJQUFJLEdBQUcsRUFBUyxDQUFDO1FBQ2pDLE1BQU0sR0FBRyxHQUFHLElBQUksR0FBRyxFQUFTLENBQUM7UUFDN0IsTUFBTSxLQUFLLEdBQUcsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUN0QixPQUFPLEtBQUssQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO1lBQ3JCLE1BQU0sS0FBSyxHQUFHLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBRSxDQUFDO1lBQ3JDLElBQUksT0FBTyxDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRTtnQkFDcEIsU0FBUzthQUNaO1lBQ0QsT0FBTyxDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsQ0FBQztZQUNuQixNQUFNLFFBQVEsR0FBRyxLQUFLLENBQUMsbUJBQW1CLENBQUMsT0FBTyxDQUFDLENBQUM7WUFDcEQsSUFBSSxRQUFRLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtnQkFDckIsa0JBQWtCO2dCQUNsQixLQUFLLENBQUMsSUFBSSxDQUFDLEdBQUcsUUFBUSxDQUFDLENBQUM7YUFDM0I7aUJBQ0k7Z0JBQ0QsaUJBQWlCO2dCQUNqQixHQUFHLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxDQUFDO2FBQ2xCO1NBQ0o7UUFDRCxPQUFPLEtBQUssQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDM0IsQ0FBQzs7Ozs7O0lBSU0sTUFBTSxDQUFDLGVBQWUsQ0FBQyxNQUFlO1FBQ3pDLE9BQU8sTUFBTSxDQUFDLE1BQU0sQ0FBQyxVQUFVLENBQVEsQ0FBQztJQUM1QyxDQUFDOzs7Ozs7SUFtREQsSUFBVyxFQUFFO1FBQ1QsT0FBTyxJQUFJLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQztJQUN4QixDQUFDOzs7Ozs7SUFJRCxJQUFXLE9BQU87UUFDZCxPQUFPLElBQUksQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUM7SUFDbEQsQ0FBQzs7Ozs7O0lBSU0sU0FBUyxDQUFDLENBQVM7UUFDdEIsSUFBSSxDQUFDLEtBQUssRUFBRSxFQUFFO1lBQ1YsSUFBSSxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQztTQUNqQztJQUNMLENBQUM7Ozs7Ozs7OztJQU9NLFdBQVcsQ0FBQyxLQUFpQjtRQUNoQyxJQUFJLElBQUksQ0FBQyxlQUFlLEtBQUssS0FBSyxFQUFFO1lBQ2hDLE9BQU87U0FDVjtRQUNELElBQUksSUFBSSxDQUFDLGVBQWUsRUFBRTtZQUN0QixtQ0FBbUM7WUFDbkMsTUFBTSxJQUFJLEtBQUssQ0FBQyx3QkFBd0IsSUFBSSxDQUFDLE9BQU8sUUFBUSxLQUFLLHVCQUF1QixJQUFJLENBQUMsZUFBZSw4Q0FBOEMsQ0FBQyxDQUFDO1NBQy9KO1FBQ0QsSUFBSSxDQUFDLGVBQWUsR0FBRyxLQUFLLENBQUM7UUFDN0IsSUFBSSxDQUFDLGdCQUFnQixDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQzdCLEtBQUssTUFBTSxRQUFRLElBQUksSUFBSSxDQUFDLGNBQWMsRUFBRTtZQUN4QyxRQUFRLENBQUMsV0FBVyxDQUFDLEtBQUssQ0FBQyxDQUFDO1NBQy9CO1FBQ0QsS0FBSyxNQUFNLFFBQVEsSUFBSSxJQUFJLENBQUMsbUJBQW1CLENBQUMsRUFBRSxvQkFBb0IsRUFBRSxJQUFJLEVBQUUsQ0FBQyxFQUFFO1lBQzdFLFFBQVEsQ0FBQyxXQUFXLENBQUMsS0FBSyxDQUFDLENBQUM7U0FDL0I7UUFDRCxLQUFLLE1BQU0sTUFBTSxJQUFJLElBQUksQ0FBQyxRQUFRLEVBQUU7WUFDaEMsTUFBTSxDQUFDLGtCQUFrQixDQUFDLElBQUksQ0FBQyxlQUFlLENBQUMsQ0FBQztTQUNuRDtRQUNELElBQUksQ0FBQyxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUU7WUFDbEIsSUFBSSxDQUFDLFNBQVMsQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLENBQUMsZUFBZSxDQUFDLENBQUM7U0FDM0Q7SUFDTCxDQUFDO0lBS0Q7OztPQUdHO0lBQ08sU0FBUyxDQUFDLFFBQW9CLEVBQUU7UUFDdEMsY0FBYyxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUM3QixJQUFJLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQztZQUNkLEdBQUcsS0FBSztZQUNSLE1BQU0sRUFBRSxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLGNBQU0sQ0FBQyxHQUFHLENBQUM7U0FDckQsQ0FBQyxDQUFDO0lBQ1AsQ0FBQztJQUNEOzs7T0FHRztJQUNPLFNBQVMsQ0FBQyxPQUFjLEVBQUUsUUFBb0IsRUFBRTtRQUN0RCxjQUFjLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQzdCLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDO1lBQ2QsSUFBSSxFQUFFLE9BQU87WUFDYixLQUFLLEVBQUU7Z0JBQ0gsTUFBTSxFQUFFLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUMsY0FBTSxDQUFDLEdBQUcsQ0FBQztnQkFDbEQsVUFBVSxFQUFFLEtBQUssQ0FBQyxVQUFVO2FBQy9CO1NBQ0osQ0FBQyxDQUFDO1FBQ0gsT0FBTyxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUMxQixJQUFJLElBQUksQ0FBQyxlQUFlLEVBQUU7WUFDdEIsT0FBTyxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsZUFBZSxDQUFDLENBQUM7U0FDN0M7SUFDTCxDQUFDOzs7Ozs7SUFJUyxRQUFRLENBQUMsSUFBVztRQUMxQiw0Q0FBNEM7UUFDNUMsSUFBSSxJQUFJLENBQUMsS0FBSyxFQUFFO1lBQ1osTUFBTSxJQUFJLEtBQUssQ0FBQyxVQUFVLElBQUksQ0FBQyxFQUFFLDRCQUE0QixDQUFDLENBQUM7U0FDbEU7UUFDRCxJQUFJLENBQUMsS0FBSyxHQUFHLElBQUksQ0FBQztRQUNsQixJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ3ZCLElBQUksSUFBSSxDQUFDLGVBQWUsRUFBRTtZQUN0QixJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksQ0FBQyxlQUFlLENBQUMsQ0FBQztTQUMxQztJQUNMLENBQUM7Ozs7OztJQUlTLFNBQVMsQ0FBQyxTQUFvQixFQUFFLElBQVc7UUFDakQsSUFBSSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsRUFBRSxTQUFTLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztRQUN2QyxJQUFJLENBQUMsVUFBVSxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUNsQyxJQUFJLElBQUksQ0FBQyxlQUFlLEVBQUU7WUFDdEIsSUFBSSxDQUFDLFVBQVUsQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLGVBQWUsQ0FBQyxDQUFDO1NBQ3JEO0lBQ0wsQ0FBQzs7Ozs7O0lBSVMsU0FBUyxDQUFDLE1BQWtCO1FBQ2xDLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQzNCLElBQUksSUFBSSxDQUFDLGVBQWUsRUFBRTtZQUN0QixNQUFNLENBQUMsa0JBQWtCLENBQUMsSUFBSSxDQUFDLGVBQWUsQ0FBQyxDQUFDO1NBQ25EO0lBQ0wsQ0FBQzs7Ozs7O0lBSVMsV0FBVyxDQUFDLFNBQXFCO1FBQ3ZDLElBQUksQ0FBQyxTQUFTLEdBQUcsU0FBUyxDQUFDO1FBQzNCLElBQUksSUFBSSxDQUFDLGVBQWUsRUFBRTtZQUN0QixTQUFTLENBQUMsa0JBQWtCLENBQUMsSUFBSSxDQUFDLGVBQWUsQ0FBQyxDQUFDO1NBQ3REO0lBQ0wsQ0FBQzs7Ozs7O0lBSVMsV0FBVyxDQUFDLEdBQVU7UUFDNUIsK0NBQStDO1FBQy9DLElBQUksSUFBSSxDQUFDLGFBQWEsRUFBRTtZQUNwQixNQUFNLElBQUksS0FBSyxDQUFDLFdBQVcsSUFBSSxDQUFDLEVBQUUsb0NBQW9DLENBQUMsQ0FBQztTQUMzRTtRQUNELElBQUksQ0FBQyxhQUFhLEdBQUcsR0FBRyxDQUFDO0lBQzdCLENBQUM7Ozs7OztJQUlTLGFBQWE7UUFDbkIsSUFBSSxJQUFJLENBQUMsS0FBSyxFQUFFO1lBQ1osT0FBTyxFQUFFLElBQUksRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sRUFBRSxDQUFDO1NBQ3ZDO2FBQ0k7WUFDRCxPQUFPLEVBQUUsR0FBRyxFQUFFLElBQUksRUFBRSxDQUFDO1NBQ3hCO0lBQ0wsQ0FBQzs7Ozs7O0lBSVMsYUFBYTtRQUNuQixPQUFPO1lBQ0gsT0FBTyxFQUFFLFVBQVUsQ0FBQyxJQUFJLENBQUMsT0FBTyxFQUFFLFlBQVksQ0FBQztZQUMvQyxPQUFPLEVBQUUsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLFNBQVM7U0FDdkUsQ0FBQztJQUNOLENBQUM7Ozs7OztJQUlTLGlCQUFpQjtRQUN2QixPQUFPO1lBQ0gsU0FBUyxFQUFFLGNBQWMsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ3pDLFVBQVUsRUFBRSxJQUFJLENBQUMsVUFBVTtZQUMzQixVQUFVLEVBQUUsY0FBYyxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUM7U0FDOUMsQ0FBQztJQUNOLENBQUM7Ozs7OztJQUlTLGNBQWM7UUFDcEIsT0FBTztZQUNILFFBQVEsRUFBRSxJQUFJLENBQUMsUUFBUSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxXQUFXLEVBQUUsQ0FBQztTQUNwRCxDQUFDO0lBQ04sQ0FBQzs7Ozs7O0lBSVMsY0FBYztRQUNwQixJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRTtZQUNqQixNQUFNLElBQUksS0FBSyxDQUFDLGtDQUFrQyxDQUFDLENBQUM7U0FDdkQ7UUFDRCxPQUFPO1lBQ0gsUUFBUSxFQUFFLElBQUksQ0FBQyxTQUFTLENBQUMsV0FBVyxFQUFFO1NBQ3pDLENBQUM7SUFDTixDQUFDOzs7Ozs7SUFJUyxnQkFBZ0I7UUFDdEIsT0FBTztZQUNILEtBQUssRUFBRSxVQUFVLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRSxXQUFXLEVBQUUsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUM7WUFDekYsS0FBSyxFQUFFLFVBQVUsQ0FBQyxJQUFJLENBQUMsT0FBTyxFQUFFLFdBQVcsRUFBRSxDQUFDLENBQUMsRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1NBQ3hHLENBQUM7SUFDTixDQUFDOzs7Ozs7OztJQU1TLGdCQUFnQixDQUFDLEtBQWlCO1FBQ3hDLEtBQUssQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDOUIsQ0FBQztJQUNEOztPQUVHO0lBQ0ssV0FBVyxDQUFDLE1BQWE7UUFDN0IsSUFBSSxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUM7SUFDckMsQ0FBQztJQUNEOztPQUVHO0lBQ0ssbUJBQW1CLENBQUMsT0FBeUI7UUFDakQsTUFBTSxHQUFHLEdBQUcsSUFBSSxLQUFLLEVBQVMsQ0FBQztRQUMvQixJQUFJLElBQUksQ0FBQyxLQUFLLEVBQUU7WUFDWixHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztTQUN4QjtRQUNELElBQUksSUFBSSxDQUFDLGFBQWEsRUFBRTtZQUNwQixHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQztTQUNoQztRQUNELEtBQUssTUFBTSxDQUFDLElBQUksSUFBSSxDQUFDLE9BQU8sRUFBRTtZQUMxQixHQUFHLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQztTQUNwQjtRQUNELElBQUksT0FBTyxDQUFDLG9CQUFvQixFQUFFO1lBQzlCLEtBQUssTUFBTSxDQUFDLElBQUksSUFBSSxDQUFDLE9BQU8sRUFBRTtnQkFDMUIsR0FBRyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUM7YUFDcEI7U0FDSjtRQUNELE9BQU8sR0FBRyxDQUFDO0lBQ2YsQ0FBQztDQUNKO0FBblZELHNCQW1WQztBQXlCRDs7R0FFRztBQUNILFNBQVMsWUFBWSxDQUFDLENBQW1CO0lBQ3JDLE9BQU87UUFDSCxHQUFHLENBQUMsQ0FBQyxTQUFTLENBQUMsZUFBZSxFQUFFO1FBQ2hDLElBQUksRUFBRSxDQUFDLENBQUMsSUFBSSxDQUFDLE9BQU87S0FDdkIsQ0FBQztBQUNOLENBQUM7QUFjRDs7R0FFRztBQUNILFNBQVMsV0FBVyxDQUFDLEtBQWlCO0lBQ2xDLE9BQU87UUFDSCxXQUFXLEVBQUUsS0FBSyxDQUFDLE1BQU07UUFDekIsZUFBZSxFQUFFLEtBQUssQ0FBQyxRQUFRLElBQUksS0FBSyxDQUFDLFFBQVEsQ0FBQyxTQUFTLEVBQUU7UUFDN0QsV0FBVyxFQUFFLEtBQUssQ0FBQyxXQUFXO1FBQzlCLFdBQVcsRUFBRSxLQUFLLENBQUMsV0FBVztLQUNqQyxDQUFDO0FBQ04sQ0FBQztBQUNEOztHQUVHO0FBQ0gsU0FBUyxXQUFXLENBQUMsQ0FBa0I7SUFDbkMsT0FBTztRQUNILFdBQVcsRUFBRSxDQUFDLENBQUMsS0FBSyxDQUFDLE1BQU07UUFDM0IsVUFBVSxFQUFFLGNBQWMsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQztRQUM5QyxJQUFJLEVBQUUsQ0FBQyxDQUFDLElBQUksQ0FBQyxPQUFPO0tBQ3ZCLENBQUM7QUFDTixDQUFDO0FBQ0Q7O0dBRUc7QUFDSCxTQUFTLGFBQWEsQ0FBQyxDQUFZLEVBQUUsQ0FBWTtJQUM3QyxJQUFJLENBQUMsYUFBRCxDQUFDLHVCQUFELENBQUMsQ0FBRSxRQUFRLENBQUMsY0FBTSxDQUFDLEdBQUcsR0FBRztRQUN6QixPQUFPLENBQUMsQ0FBQztLQUNaO0lBQ0QsSUFBSSxDQUFDLGFBQUQsQ0FBQyx1QkFBRCxDQUFDLENBQUUsUUFBUSxDQUFDLGNBQU0sQ0FBQyxHQUFHLEdBQUc7UUFDekIsT0FBTyxDQUFDLENBQUMsQ0FBQztLQUNiO0lBQ0QsT0FBTyxDQUFDLENBQUM7QUFDYixDQUFDO0FBQ0Q7O0dBRUc7QUFDSCxTQUFTLGNBQWMsQ0FBQyxNQUFpQjtJQUNyQyxJQUFJLENBQUEsTUFBTSxhQUFOLE1BQU0sdUJBQU4sTUFBTSxDQUFFLFFBQVEsQ0FBQyxjQUFNLENBQUMsR0FBRyxNQUFLLE1BQU0sQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO1FBQ25ELE1BQU0sSUFBSSxLQUFLLENBQUMsR0FBRyxjQUFNLENBQUMsR0FBRyxxQ0FBcUMsQ0FBQyxDQUFDO0tBQ3ZFO0FBQ0wsQ0FBQztBQUNEOztHQUVHO0FBQ0gsU0FBZ0IsVUFBVSxDQUFJLEVBQU8sRUFBRSxLQUFvQixFQUFFLE1BQStCO0lBQ3hGLElBQUksRUFBRSxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7UUFDakIsT0FBTyxTQUFTLENBQUM7S0FDcEI7SUFDRCxJQUFJLElBQUksR0FBRyxFQUFFLENBQUM7SUFDZCxJQUFJLE1BQU0sRUFBRTtRQUNSLElBQUksR0FBRyxFQUFFLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDO0tBQzFCO0lBQ0QsT0FBTyxJQUFJLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxDQUFDO0FBQzNCLENBQUM7QUFURCxnQ0FTQztBQUNEOztHQUVHO0FBQ0gsU0FBZ0IsY0FBYyxDQUFDLFFBQWlCO0lBQzVDLElBQUksUUFBUSxLQUFLLFNBQVMsRUFBRTtRQUN4QixPQUFPLFNBQVMsQ0FBQztLQUNwQjtJQUNELElBQUksUUFBUSxLQUFLLGlCQUFRLENBQUMsT0FBTyxFQUFFO1FBQy9CLE9BQU8sSUFBSSxDQUFDO0tBQ2Y7SUFDRCxJQUFJLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQyxHQUFHLENBQUMsRUFBRTtRQUMzQixNQUFNLElBQUksS0FBSyxDQUFDLDhDQUE4QyxRQUFRLEVBQUUsQ0FBQyxDQUFDO0tBQzdFO0lBQ0QsT0FBTyxRQUFRLENBQUM7QUFDcEIsQ0FBQztBQVhELHdDQVdDO0FBT0Q7O0dBRUc7QUFDSCxTQUFTLFlBQVksQ0FBQyxDQUFNO0lBQ3hCLE9BQU8sT0FBTyxDQUFDLENBQUMsQ0FBQyxLQUFLLFFBQVEsSUFBSSxDQUFDLENBQUMsU0FBUyxDQUFDO0FBQ2xELENBQUM7QUFDRDs7R0FFRztBQUNILFNBQVMsVUFBVSxDQUFDLENBQU07SUFDdEIsT0FBTyxPQUFPLENBQUMsQ0FBQyxDQUFDLEtBQUssUUFBUSxJQUFJLENBQUMsQ0FBQyxJQUFJLENBQUM7QUFDN0MsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGNkayBmcm9tIFwiLi4vLi4vLi4vY29yZVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvY29yZSdcbmltcG9ydCB7IElDb25zdHJ1Y3QsIENvbnN0cnVjdCwgTm9kZSB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuaW1wb3J0IHsgQ29uZGl0aW9uIH0gZnJvbSAnLi4vY29uZGl0aW9uJztcbmltcG9ydCB7IEpzb25QYXRoIH0gZnJvbSAnLi4vZmllbGRzJztcbmltcG9ydCB7IFN0YXRlR3JhcGggfSBmcm9tICcuLi9zdGF0ZS1ncmFwaCc7XG5pbXBvcnQgeyBDYXRjaFByb3BzLCBFcnJvcnMsIElDaGFpbmFibGUsIElOZXh0YWJsZSwgUmV0cnlQcm9wcyB9IGZyb20gJy4uL3R5cGVzJztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBTdGF0ZVByb3BzIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBjb21tZW50Pzogc3RyaW5nO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IGlucHV0UGF0aD86IHN0cmluZztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBwYXJhbWV0ZXJzPzoge1xuICAgICAgICBbbmFtZTogc3RyaW5nXTogYW55O1xuICAgIH07XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBvdXRwdXRQYXRoPzogc3RyaW5nO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgcmVzdWx0UGF0aD86IHN0cmluZztcbn1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBhYnN0cmFjdCBjbGFzcyBTdGF0ZSBleHRlbmRzIGNkay5Db25zdHJ1Y3QgaW1wbGVtZW50cyBJQ2hhaW5hYmxlIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcHJlZml4U3RhdGVzKHJvb3Q6IElDb25zdHJ1Y3QsIHByZWZpeDogc3RyaW5nKSB7XG4gICAgICAgIGNvbnN0IHF1ZXVlID0gW3Jvb3RdO1xuICAgICAgICB3aGlsZSAocXVldWUubGVuZ3RoID4gMCkge1xuICAgICAgICAgICAgY29uc3QgZWwgPSBxdWV1ZS5zcGxpY2UoMCwgMSlbMF0hO1xuICAgICAgICAgICAgaWYgKGlzUHJlZml4YWJsZShlbCkpIHtcbiAgICAgICAgICAgICAgICBlbC5hZGRQcmVmaXgocHJlZml4KTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIHF1ZXVlLnB1c2goLi4uTm9kZS5vZihlbCkuY2hpbGRyZW4pO1xuICAgICAgICB9XG4gICAgfVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyBmaW5kUmVhY2hhYmxlU3RhdGVzKHN0YXJ0OiBTdGF0ZSwgb3B0aW9uczogRmluZFN0YXRlT3B0aW9ucyA9IHt9KTogU3RhdGVbXSB7XG4gICAgICAgIGNvbnN0IHZpc2l0ZWQgPSBuZXcgU2V0PFN0YXRlPigpO1xuICAgICAgICBjb25zdCByZXQgPSBuZXcgU2V0PFN0YXRlPigpO1xuICAgICAgICBjb25zdCBxdWV1ZSA9IFtzdGFydF07XG4gICAgICAgIHdoaWxlIChxdWV1ZS5sZW5ndGggPiAwKSB7XG4gICAgICAgICAgICBjb25zdCBzdGF0ZSA9IHF1ZXVlLnNwbGljZSgwLCAxKVswXSE7XG4gICAgICAgICAgICBpZiAodmlzaXRlZC5oYXMoc3RhdGUpKSB7XG4gICAgICAgICAgICAgICAgY29udGludWU7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICB2aXNpdGVkLmFkZChzdGF0ZSk7XG4gICAgICAgICAgICBjb25zdCBvdXRnb2luZyA9IHN0YXRlLm91dGdvaW5nVHJhbnNpdGlvbnMob3B0aW9ucyk7XG4gICAgICAgICAgICBxdWV1ZS5wdXNoKC4uLm91dGdvaW5nKTtcbiAgICAgICAgICAgIHJldC5hZGQoc3RhdGUpO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiBBcnJheS5mcm9tKHJldCk7XG4gICAgfVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIGZpbmRSZWFjaGFibGVFbmRTdGF0ZXMoc3RhcnQ6IFN0YXRlLCBvcHRpb25zOiBGaW5kU3RhdGVPcHRpb25zID0ge30pOiBTdGF0ZVtdIHtcbiAgICAgICAgY29uc3QgdmlzaXRlZCA9IG5ldyBTZXQ8U3RhdGU+KCk7XG4gICAgICAgIGNvbnN0IHJldCA9IG5ldyBTZXQ8U3RhdGU+KCk7XG4gICAgICAgIGNvbnN0IHF1ZXVlID0gW3N0YXJ0XTtcbiAgICAgICAgd2hpbGUgKHF1ZXVlLmxlbmd0aCA+IDApIHtcbiAgICAgICAgICAgIGNvbnN0IHN0YXRlID0gcXVldWUuc3BsaWNlKDAsIDEpWzBdITtcbiAgICAgICAgICAgIGlmICh2aXNpdGVkLmhhcyhzdGF0ZSkpIHtcbiAgICAgICAgICAgICAgICBjb250aW51ZTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIHZpc2l0ZWQuYWRkKHN0YXRlKTtcbiAgICAgICAgICAgIGNvbnN0IG91dGdvaW5nID0gc3RhdGUub3V0Z29pbmdUcmFuc2l0aW9ucyhvcHRpb25zKTtcbiAgICAgICAgICAgIGlmIChvdXRnb2luZy5sZW5ndGggPiAwKSB7XG4gICAgICAgICAgICAgICAgLy8gV2UgY2FuIGNvbnRpbnVlXG4gICAgICAgICAgICAgICAgcXVldWUucHVzaCguLi5vdXRnb2luZyk7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICBlbHNlIHtcbiAgICAgICAgICAgICAgICAvLyBUZXJtaW5hbCBzdGF0ZVxuICAgICAgICAgICAgICAgIHJldC5hZGQoc3RhdGUpO1xuICAgICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICAgIHJldHVybiBBcnJheS5mcm9tKHJldCk7XG4gICAgfVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIGZpbHRlck5leHRhYmxlcyhzdGF0ZXM6IFN0YXRlW10pOiBJTmV4dGFibGVbXSB7XG4gICAgICAgIHJldHVybiBzdGF0ZXMuZmlsdGVyKGlzTmV4dGFibGUpIGFzIGFueTtcbiAgICB9XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHJlYWRvbmx5IHN0YXJ0U3RhdGU6IFN0YXRlO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgZW5kU3RhdGVzOiBJTmV4dGFibGVbXTtcbiAgICAvLyBUaGlzIGNsYXNzIGhhcyBhIHN1cGVyc2V0IG9mIG1vc3Qgb2YgdGhlIGZlYXR1cmVzIG9mIHRoZSBvdGhlciBzdGF0ZXMsXG4gICAgLy8gYW5kIHRoZSBzdWJjbGFzc2VzIGRlY2lkZSB3aGljaCBwYXJ0IG9mIHRoZSBmZWF0dXJlcyB0byBleHBvc2UuIE1vc3RcbiAgICAvLyBmZWF0dXJlcyBhcmUgc2hhcmVkIGJ5IGEgY291cGxlIG9mIHN0YXRlcywgYW5kIGl0IGJlY29tZXMgY3VtYmVyc29tZSB0b1xuICAgIC8vIHNsaWNlIGl0IG91dCBhY3Jvc3MgYWxsIHN0YXRlcy4gVGhpcyBpcyBub3QgZ3JlYXQgZGVzaWduLCBidXQgaXQgaXNcbiAgICAvLyBwcmFnbWF0aWMhXG4gICAgcHJvdGVjdGVkIHJlYWRvbmx5IGNvbW1lbnQ/OiBzdHJpbmc7XG4gICAgcHJvdGVjdGVkIHJlYWRvbmx5IGlucHV0UGF0aD86IHN0cmluZztcbiAgICBwcm90ZWN0ZWQgcmVhZG9ubHkgcGFyYW1ldGVycz86IG9iamVjdDtcbiAgICBwcm90ZWN0ZWQgcmVhZG9ubHkgb3V0cHV0UGF0aD86IHN0cmluZztcbiAgICBwcm90ZWN0ZWQgcmVhZG9ubHkgcmVzdWx0UGF0aD86IHN0cmluZztcbiAgICBwcm90ZWN0ZWQgcmVhZG9ubHkgYnJhbmNoZXM6IFN0YXRlR3JhcGhbXSA9IFtdO1xuICAgIHByb3RlY3RlZCBpdGVyYXRpb24/OiBTdGF0ZUdyYXBoO1xuICAgIHByb3RlY3RlZCBkZWZhdWx0Q2hvaWNlPzogU3RhdGU7XG4gICAgLyoqXG4gICAgICogQGludGVybmFsXG4gICAgICovXG4gICAgcHJvdGVjdGVkIF9uZXh0PzogU3RhdGU7XG4gICAgcHJpdmF0ZSByZWFkb25seSByZXRyaWVzOiBSZXRyeVByb3BzW10gPSBbXTtcbiAgICBwcml2YXRlIHJlYWRvbmx5IGNhdGNoZXM6IENhdGNoVHJhbnNpdGlvbltdID0gW107XG4gICAgcHJpdmF0ZSByZWFkb25seSBjaG9pY2VzOiBDaG9pY2VUcmFuc2l0aW9uW10gPSBbXTtcbiAgICBwcml2YXRlIHJlYWRvbmx5IHByZWZpeGVzOiBzdHJpbmdbXSA9IFtdO1xuICAgIC8qKlxuICAgICAqIFRoZSBncmFwaCB0aGF0IHRoaXMgc3RhdGUgaXMgcGFydCBvZi5cbiAgICAgKlxuICAgICAqIFVzZWQgZm9yIGd1YXJhbnRlZWluZyBjb25zaXN0ZW5jeSBiZXR3ZWVuIGdyYXBocyBhbmQgZ3JhcGggY29tcG9uZW50cy5cbiAgICAgKi9cbiAgICBwcml2YXRlIGNvbnRhaW5pbmdHcmFwaD86IFN0YXRlR3JhcGg7XG4gICAgLyoqXG4gICAgICogU3RhdGVzIHdpdGggcmVmZXJlbmNlcyB0byB0aGlzIHN0YXRlLlxuICAgICAqXG4gICAgICogVXNlZCBmb3IgZmluZGluZyBjb21wbGV0ZSBjb25uZWN0ZWQgZ3JhcGggdGhhdCBhIHN0YXRlIGlzIHBhcnQgb2YuXG4gICAgICovXG4gICAgcHJpdmF0ZSByZWFkb25seSBpbmNvbWluZ1N0YXRlczogU3RhdGVbXSA9IFtdO1xuICAgIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBTdGF0ZVByb3BzKSB7XG4gICAgICAgIHN1cGVyKHNjb3BlLCBpZCk7XG4gICAgICAgIHRoaXMuc3RhcnRTdGF0ZSA9IHRoaXM7XG4gICAgICAgIHRoaXMuY29tbWVudCA9IHByb3BzLmNvbW1lbnQ7XG4gICAgICAgIHRoaXMuaW5wdXRQYXRoID0gcHJvcHMuaW5wdXRQYXRoO1xuICAgICAgICB0aGlzLnBhcmFtZXRlcnMgPSBwcm9wcy5wYXJhbWV0ZXJzO1xuICAgICAgICB0aGlzLm91dHB1dFBhdGggPSBwcm9wcy5vdXRwdXRQYXRoO1xuICAgICAgICB0aGlzLnJlc3VsdFBhdGggPSBwcm9wcy5yZXN1bHRQYXRoO1xuICAgIH1cbiAgICBwdWJsaWMgZ2V0IGlkKCkge1xuICAgICAgICByZXR1cm4gdGhpcy5ub2RlLmlkO1xuICAgIH1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBnZXQgc3RhdGVJZCgpOiBzdHJpbmcge1xuICAgICAgICByZXR1cm4gdGhpcy5wcmVmaXhlcy5jb25jYXQodGhpcy5pZCkuam9pbignJyk7XG4gICAgfVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBhZGRQcmVmaXgoeDogc3RyaW5nKSB7XG4gICAgICAgIGlmICh4ICE9PSAnJykge1xuICAgICAgICAgICAgdGhpcy5wcmVmaXhlcy5zcGxpY2UoMCwgMCwgeCk7XG4gICAgICAgIH1cbiAgICB9XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIGJpbmRUb0dyYXBoKGdyYXBoOiBTdGF0ZUdyYXBoKSB7XG4gICAgICAgIGlmICh0aGlzLmNvbnRhaW5pbmdHcmFwaCA9PT0gZ3JhcGgpIHtcbiAgICAgICAgICAgIHJldHVybjtcbiAgICAgICAgfVxuICAgICAgICBpZiAodGhpcy5jb250YWluaW5nR3JhcGgpIHtcbiAgICAgICAgICAgIC8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBtYXgtbGVuXG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYFRyeWluZyB0byB1c2Ugc3RhdGUgJyR7dGhpcy5zdGF0ZUlkfScgaW4gJHtncmFwaH0sIGJ1dCBpcyBhbHJlYWR5IGluICR7dGhpcy5jb250YWluaW5nR3JhcGh9LiBFdmVyeSBzdGF0ZSBjYW4gb25seSBiZSB1c2VkIGluIG9uZSBncmFwaC5gKTtcbiAgICAgICAgfVxuICAgICAgICB0aGlzLmNvbnRhaW5pbmdHcmFwaCA9IGdyYXBoO1xuICAgICAgICB0aGlzLndoZW5Cb3VuZFRvR3JhcGgoZ3JhcGgpO1xuICAgICAgICBmb3IgKGNvbnN0IGluY29taW5nIG9mIHRoaXMuaW5jb21pbmdTdGF0ZXMpIHtcbiAgICAgICAgICAgIGluY29taW5nLmJpbmRUb0dyYXBoKGdyYXBoKTtcbiAgICAgICAgfVxuICAgICAgICBmb3IgKGNvbnN0IG91dGdvaW5nIG9mIHRoaXMub3V0Z29pbmdUcmFuc2l0aW9ucyh7IGluY2x1ZGVFcnJvckhhbmRsZXJzOiB0cnVlIH0pKSB7XG4gICAgICAgICAgICBvdXRnb2luZy5iaW5kVG9HcmFwaChncmFwaCk7XG4gICAgICAgIH1cbiAgICAgICAgZm9yIChjb25zdCBicmFuY2ggb2YgdGhpcy5icmFuY2hlcykge1xuICAgICAgICAgICAgYnJhbmNoLnJlZ2lzdGVyU3VwZXJHcmFwaCh0aGlzLmNvbnRhaW5pbmdHcmFwaCk7XG4gICAgICAgIH1cbiAgICAgICAgaWYgKCEhdGhpcy5pdGVyYXRpb24pIHtcbiAgICAgICAgICAgIHRoaXMuaXRlcmF0aW9uLnJlZ2lzdGVyU3VwZXJHcmFwaCh0aGlzLmNvbnRhaW5pbmdHcmFwaCk7XG4gICAgICAgIH1cbiAgICB9XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBhYnN0cmFjdCB0b1N0YXRlSnNvbigpOiBvYmplY3Q7XG4gICAgLyoqXG4gICAgICogQWRkIGEgcmV0cmllciB0byB0aGUgcmV0cnkgbGlzdCBvZiB0aGlzIHN0YXRlXG4gICAgICogQGludGVybmFsXG4gICAgICovXG4gICAgcHJvdGVjdGVkIF9hZGRSZXRyeShwcm9wczogUmV0cnlQcm9wcyA9IHt9KSB7XG4gICAgICAgIHZhbGlkYXRlRXJyb3JzKHByb3BzLmVycm9ycyk7XG4gICAgICAgIHRoaXMucmV0cmllcy5wdXNoKHtcbiAgICAgICAgICAgIC4uLnByb3BzLFxuICAgICAgICAgICAgZXJyb3JzOiBwcm9wcy5lcnJvcnMgPyBwcm9wcy5lcnJvcnMgOiBbRXJyb3JzLkFMTF0sXG4gICAgICAgIH0pO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBBZGQgYW4gZXJyb3IgaGFuZGxlciB0byB0aGUgY2F0Y2ggbGlzdCBvZiB0aGlzIHN0YXRlXG4gICAgICogQGludGVybmFsXG4gICAgICovXG4gICAgcHJvdGVjdGVkIF9hZGRDYXRjaChoYW5kbGVyOiBTdGF0ZSwgcHJvcHM6IENhdGNoUHJvcHMgPSB7fSkge1xuICAgICAgICB2YWxpZGF0ZUVycm9ycyhwcm9wcy5lcnJvcnMpO1xuICAgICAgICB0aGlzLmNhdGNoZXMucHVzaCh7XG4gICAgICAgICAgICBuZXh0OiBoYW5kbGVyLFxuICAgICAgICAgICAgcHJvcHM6IHtcbiAgICAgICAgICAgICAgICBlcnJvcnM6IHByb3BzLmVycm9ycyA/IHByb3BzLmVycm9ycyA6IFtFcnJvcnMuQUxMXSxcbiAgICAgICAgICAgICAgICByZXN1bHRQYXRoOiBwcm9wcy5yZXN1bHRQYXRoLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgfSk7XG4gICAgICAgIGhhbmRsZXIuYWRkSW5jb21pbmcodGhpcyk7XG4gICAgICAgIGlmICh0aGlzLmNvbnRhaW5pbmdHcmFwaCkge1xuICAgICAgICAgICAgaGFuZGxlci5iaW5kVG9HcmFwaCh0aGlzLmNvbnRhaW5pbmdHcmFwaCk7XG4gICAgICAgIH1cbiAgICB9XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwcm90ZWN0ZWQgbWFrZU5leHQobmV4dDogU3RhdGUpIHtcbiAgICAgICAgLy8gQ2FuJ3QgYmUgY2FsbGVkICdzZXROZXh0JyBiZWNhdXNlIG9mIEpTSUlcbiAgICAgICAgaWYgKHRoaXMuX25leHQpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgU3RhdGUgJyR7dGhpcy5pZH0nIGFscmVhZHkgaGFzIGEgbmV4dCBzdGF0ZWApO1xuICAgICAgICB9XG4gICAgICAgIHRoaXMuX25leHQgPSBuZXh0O1xuICAgICAgICBuZXh0LmFkZEluY29taW5nKHRoaXMpO1xuICAgICAgICBpZiAodGhpcy5jb250YWluaW5nR3JhcGgpIHtcbiAgICAgICAgICAgIG5leHQuYmluZFRvR3JhcGgodGhpcy5jb250YWluaW5nR3JhcGgpO1xuICAgICAgICB9XG4gICAgfVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwcm90ZWN0ZWQgYWRkQ2hvaWNlKGNvbmRpdGlvbjogQ29uZGl0aW9uLCBuZXh0OiBTdGF0ZSkge1xuICAgICAgICB0aGlzLmNob2ljZXMucHVzaCh7IGNvbmRpdGlvbiwgbmV4dCB9KTtcbiAgICAgICAgbmV4dC5zdGFydFN0YXRlLmFkZEluY29taW5nKHRoaXMpO1xuICAgICAgICBpZiAodGhpcy5jb250YWluaW5nR3JhcGgpIHtcbiAgICAgICAgICAgIG5leHQuc3RhcnRTdGF0ZS5iaW5kVG9HcmFwaCh0aGlzLmNvbnRhaW5pbmdHcmFwaCk7XG4gICAgICAgIH1cbiAgICB9XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwcm90ZWN0ZWQgYWRkQnJhbmNoKGJyYW5jaDogU3RhdGVHcmFwaCkge1xuICAgICAgICB0aGlzLmJyYW5jaGVzLnB1c2goYnJhbmNoKTtcbiAgICAgICAgaWYgKHRoaXMuY29udGFpbmluZ0dyYXBoKSB7XG4gICAgICAgICAgICBicmFuY2gucmVnaXN0ZXJTdXBlckdyYXBoKHRoaXMuY29udGFpbmluZ0dyYXBoKTtcbiAgICAgICAgfVxuICAgIH1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwcm90ZWN0ZWQgYWRkSXRlcmF0b3IoaXRlcmF0aW9uOiBTdGF0ZUdyYXBoKSB7XG4gICAgICAgIHRoaXMuaXRlcmF0aW9uID0gaXRlcmF0aW9uO1xuICAgICAgICBpZiAodGhpcy5jb250YWluaW5nR3JhcGgpIHtcbiAgICAgICAgICAgIGl0ZXJhdGlvbi5yZWdpc3RlclN1cGVyR3JhcGgodGhpcy5jb250YWluaW5nR3JhcGgpO1xuICAgICAgICB9XG4gICAgfVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHByb3RlY3RlZCBtYWtlRGVmYXVsdChkZWY6IFN0YXRlKSB7XG4gICAgICAgIC8vIENhbid0IGJlIGNhbGxlZCAnc2V0RGVmYXVsdCcgYmVjYXVzZSBvZiBKU0lJXG4gICAgICAgIGlmICh0aGlzLmRlZmF1bHRDaG9pY2UpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgQ2hvaWNlICcke3RoaXMuaWR9JyBhbHJlYWR5IGhhcyBhIGRlZmF1bHQgbmV4dCBzdGF0ZWApO1xuICAgICAgICB9XG4gICAgICAgIHRoaXMuZGVmYXVsdENob2ljZSA9IGRlZjtcbiAgICB9XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHByb3RlY3RlZCByZW5kZXJOZXh0RW5kKCk6IGFueSB7XG4gICAgICAgIGlmICh0aGlzLl9uZXh0KSB7XG4gICAgICAgICAgICByZXR1cm4geyBOZXh0OiB0aGlzLl9uZXh0LnN0YXRlSWQgfTtcbiAgICAgICAgfVxuICAgICAgICBlbHNlIHtcbiAgICAgICAgICAgIHJldHVybiB7IEVuZDogdHJ1ZSB9O1xuICAgICAgICB9XG4gICAgfVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHJvdGVjdGVkIHJlbmRlckNob2ljZXMoKTogYW55IHtcbiAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICAgIENob2ljZXM6IHJlbmRlckxpc3QodGhpcy5jaG9pY2VzLCByZW5kZXJDaG9pY2UpLFxuICAgICAgICAgICAgRGVmYXVsdDogdGhpcy5kZWZhdWx0Q2hvaWNlID8gdGhpcy5kZWZhdWx0Q2hvaWNlLnN0YXRlSWQgOiB1bmRlZmluZWQsXG4gICAgICAgIH07XG4gICAgfVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwcm90ZWN0ZWQgcmVuZGVySW5wdXRPdXRwdXQoKTogYW55IHtcbiAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICAgIElucHV0UGF0aDogcmVuZGVySnNvblBhdGgodGhpcy5pbnB1dFBhdGgpLFxuICAgICAgICAgICAgUGFyYW1ldGVyczogdGhpcy5wYXJhbWV0ZXJzLFxuICAgICAgICAgICAgT3V0cHV0UGF0aDogcmVuZGVySnNvblBhdGgodGhpcy5vdXRwdXRQYXRoKSxcbiAgICAgICAgfTtcbiAgICB9XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwcm90ZWN0ZWQgcmVuZGVyQnJhbmNoZXMoKTogYW55IHtcbiAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICAgIEJyYW5jaGVzOiB0aGlzLmJyYW5jaGVzLm1hcChiID0+IGIudG9HcmFwaEpzb24oKSksXG4gICAgICAgIH07XG4gICAgfVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHByb3RlY3RlZCByZW5kZXJJdGVyYXRvcigpOiBhbnkge1xuICAgICAgICBpZiAoIXRoaXMuaXRlcmF0aW9uKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0l0ZXJhdG9yIG11c3Qgbm90IGJlIHVuZGVmaW5lZCAhJyk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICAgIEl0ZXJhdG9yOiB0aGlzLml0ZXJhdGlvbi50b0dyYXBoSnNvbigpLFxuICAgICAgICB9O1xuICAgIH1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHJvdGVjdGVkIHJlbmRlclJldHJ5Q2F0Y2goKTogYW55IHtcbiAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICAgIFJldHJ5OiByZW5kZXJMaXN0KHRoaXMucmV0cmllcywgcmVuZGVyUmV0cnksIChhLCBiKSA9PiBjb21wYXJlRXJyb3JzKGEuZXJyb3JzLCBiLmVycm9ycykpLFxuICAgICAgICAgICAgQ2F0Y2g6IHJlbmRlckxpc3QodGhpcy5jYXRjaGVzLCByZW5kZXJDYXRjaCwgKGEsIGIpID0+IGNvbXBhcmVFcnJvcnMoYS5wcm9wcy5lcnJvcnMsIGIucHJvcHMuZXJyb3JzKSksXG4gICAgICAgIH07XG4gICAgfVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwcm90ZWN0ZWQgd2hlbkJvdW5kVG9HcmFwaChncmFwaDogU3RhdGVHcmFwaCkge1xuICAgICAgICBncmFwaC5yZWdpc3RlclN0YXRlKHRoaXMpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBBZGQgYSBzdGF0ZSB0byB0aGUgaW5jb21pbmcgbGlzdFxuICAgICAqL1xuICAgIHByaXZhdGUgYWRkSW5jb21pbmcoc291cmNlOiBTdGF0ZSkge1xuICAgICAgICB0aGlzLmluY29taW5nU3RhdGVzLnB1c2goc291cmNlKTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogUmV0dXJuIGFsbCBzdGF0ZXMgdGhpcyBzdGF0ZSBjYW4gdHJhbnNpdGlvbiB0b1xuICAgICAqL1xuICAgIHByaXZhdGUgb3V0Z29pbmdUcmFuc2l0aW9ucyhvcHRpb25zOiBGaW5kU3RhdGVPcHRpb25zKTogU3RhdGVbXSB7XG4gICAgICAgIGNvbnN0IHJldCA9IG5ldyBBcnJheTxTdGF0ZT4oKTtcbiAgICAgICAgaWYgKHRoaXMuX25leHQpIHtcbiAgICAgICAgICAgIHJldC5wdXNoKHRoaXMuX25leHQpO1xuICAgICAgICB9XG4gICAgICAgIGlmICh0aGlzLmRlZmF1bHRDaG9pY2UpIHtcbiAgICAgICAgICAgIHJldC5wdXNoKHRoaXMuZGVmYXVsdENob2ljZSk7XG4gICAgICAgIH1cbiAgICAgICAgZm9yIChjb25zdCBjIG9mIHRoaXMuY2hvaWNlcykge1xuICAgICAgICAgICAgcmV0LnB1c2goYy5uZXh0KTtcbiAgICAgICAgfVxuICAgICAgICBpZiAob3B0aW9ucy5pbmNsdWRlRXJyb3JIYW5kbGVycykge1xuICAgICAgICAgICAgZm9yIChjb25zdCBjIG9mIHRoaXMuY2F0Y2hlcykge1xuICAgICAgICAgICAgICAgIHJldC5wdXNoKGMubmV4dCk7XG4gICAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIHJldDtcbiAgICB9XG59XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBGaW5kU3RhdGVPcHRpb25zIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBpbmNsdWRlRXJyb3JIYW5kbGVycz86IGJvb2xlYW47XG59XG4vKipcbiAqIEEgQ2hvaWNlIFRyYW5zaXRpb25cbiAqL1xuaW50ZXJmYWNlIENob2ljZVRyYW5zaXRpb24ge1xuICAgIC8qKlxuICAgICAqIFN0YXRlIHRvIHRyYW5zaXRpb24gdG9cbiAgICAgKi9cbiAgICBuZXh0OiBTdGF0ZTtcbiAgICAvKipcbiAgICAgKiBDb25kaXRpb24gZm9yIHRoaXMgdHJhbnNpdGlvblxuICAgICAqL1xuICAgIGNvbmRpdGlvbjogQ29uZGl0aW9uO1xufVxuLyoqXG4gKiBSZW5kZXIgYSBjaG9pY2UgdHJhbnNpdGlvblxuICovXG5mdW5jdGlvbiByZW5kZXJDaG9pY2UoYzogQ2hvaWNlVHJhbnNpdGlvbikge1xuICAgIHJldHVybiB7XG4gICAgICAgIC4uLmMuY29uZGl0aW9uLnJlbmRlckNvbmRpdGlvbigpLFxuICAgICAgICBOZXh0OiBjLm5leHQuc3RhdGVJZCxcbiAgICB9O1xufVxuLyoqXG4gKiBBIENhdGNoIFRyYW5zaXRpb25cbiAqL1xuaW50ZXJmYWNlIENhdGNoVHJhbnNpdGlvbiB7XG4gICAgLyoqXG4gICAgICogU3RhdGUgdG8gdHJhbnNpdGlvbiB0b1xuICAgICAqL1xuICAgIG5leHQ6IFN0YXRlO1xuICAgIC8qKlxuICAgICAqIEFkZGl0aW9uYWwgcHJvcGVydGllcyBmb3IgdGhpcyB0cmFuc2l0aW9uXG4gICAgICovXG4gICAgcHJvcHM6IENhdGNoUHJvcHM7XG59XG4vKipcbiAqIFJlbmRlciBhIFJldHJ5IG9iamVjdCB0byBBU0xcbiAqL1xuZnVuY3Rpb24gcmVuZGVyUmV0cnkocmV0cnk6IFJldHJ5UHJvcHMpIHtcbiAgICByZXR1cm4ge1xuICAgICAgICBFcnJvckVxdWFsczogcmV0cnkuZXJyb3JzLFxuICAgICAgICBJbnRlcnZhbFNlY29uZHM6IHJldHJ5LmludGVydmFsICYmIHJldHJ5LmludGVydmFsLnRvU2Vjb25kcygpLFxuICAgICAgICBNYXhBdHRlbXB0czogcmV0cnkubWF4QXR0ZW1wdHMsXG4gICAgICAgIEJhY2tvZmZSYXRlOiByZXRyeS5iYWNrb2ZmUmF0ZSxcbiAgICB9O1xufVxuLyoqXG4gKiBSZW5kZXIgYSBDYXRjaCBvYmplY3QgdG8gQVNMXG4gKi9cbmZ1bmN0aW9uIHJlbmRlckNhdGNoKGM6IENhdGNoVHJhbnNpdGlvbikge1xuICAgIHJldHVybiB7XG4gICAgICAgIEVycm9yRXF1YWxzOiBjLnByb3BzLmVycm9ycyxcbiAgICAgICAgUmVzdWx0UGF0aDogcmVuZGVySnNvblBhdGgoYy5wcm9wcy5yZXN1bHRQYXRoKSxcbiAgICAgICAgTmV4dDogYy5uZXh0LnN0YXRlSWQsXG4gICAgfTtcbn1cbi8qKlxuICogQ29tcGFyZXMgYSBsaXN0IG9mIEVycm9ycyB0byBtb3ZlIEVycm9ycy5BTEwgbGFzdCBpbiBhIHNvcnQgZnVuY3Rpb25cbiAqL1xuZnVuY3Rpb24gY29tcGFyZUVycm9ycyhhPzogc3RyaW5nW10sIGI/OiBzdHJpbmdbXSkge1xuICAgIGlmIChhPy5pbmNsdWRlcyhFcnJvcnMuQUxMKSkge1xuICAgICAgICByZXR1cm4gMTtcbiAgICB9XG4gICAgaWYgKGI/LmluY2x1ZGVzKEVycm9ycy5BTEwpKSB7XG4gICAgICAgIHJldHVybiAtMTtcbiAgICB9XG4gICAgcmV0dXJuIDA7XG59XG4vKipcbiAqIFZhbGlkYXRlcyBhbiBlcnJvcnMgbGlzdFxuICovXG5mdW5jdGlvbiB2YWxpZGF0ZUVycm9ycyhlcnJvcnM/OiBzdHJpbmdbXSkge1xuICAgIGlmIChlcnJvcnM/LmluY2x1ZGVzKEVycm9ycy5BTEwpICYmIGVycm9ycy5sZW5ndGggPiAxKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihgJHtFcnJvcnMuQUxMfSBtdXN0IGFwcGVhciBhbG9uZSBpbiBhbiBlcnJvciBsaXN0YCk7XG4gICAgfVxufVxuLyoqXG4gKiBSZW5kZXIgYSBsaXN0IG9yIHJldHVybiB1bmRlZmluZWQgZm9yIGFuIGVtcHR5IGxpc3RcbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIHJlbmRlckxpc3Q8VD4oeHM6IFRbXSwgbWFwRm46ICh4OiBUKSA9PiBhbnksIHNvcnRGbj86IChhOiBULCBiOiBUKSA9PiBudW1iZXIpOiBhbnkge1xuICAgIGlmICh4cy5sZW5ndGggPT09IDApIHtcbiAgICAgICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgICB9XG4gICAgbGV0IGxpc3QgPSB4cztcbiAgICBpZiAoc29ydEZuKSB7XG4gICAgICAgIGxpc3QgPSB4cy5zb3J0KHNvcnRGbik7XG4gICAgfVxuICAgIHJldHVybiBsaXN0Lm1hcChtYXBGbik7XG59XG4vKipcbiAqIFJlbmRlciBKU09OIHBhdGgsIHJlc3BlY3RpbmcgdGhlIHNwZWNpYWwgdmFsdWUgRElTQ0FSRFxuICovXG5leHBvcnQgZnVuY3Rpb24gcmVuZGVySnNvblBhdGgoanNvblBhdGg/OiBzdHJpbmcpOiB1bmRlZmluZWQgfCBudWxsIHwgc3RyaW5nIHtcbiAgICBpZiAoanNvblBhdGggPT09IHVuZGVmaW5lZCkge1xuICAgICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgIH1cbiAgICBpZiAoanNvblBhdGggPT09IEpzb25QYXRoLkRJU0NBUkQpIHtcbiAgICAgICAgcmV0dXJuIG51bGw7XG4gICAgfVxuICAgIGlmICghanNvblBhdGguc3RhcnRzV2l0aCgnJCcpKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihgRXhwZWN0ZWQgSlNPTiBwYXRoIHRvIHN0YXJ0IHdpdGggJyQnLCBnb3Q6ICR7anNvblBhdGh9YCk7XG4gICAgfVxuICAgIHJldHVybiBqc29uUGF0aDtcbn1cbi8qKlxuICogSW50ZXJmYWNlIGZvciBzdHJ1Y3R1cmFsIGZlYXR1cmUgdGVzdGluZyAodG8gbWFrZSBUeXBlU2NyaXB0IGhhcHB5KVxuICovXG5pbnRlcmZhY2UgUHJlZml4YWJsZSB7XG4gICAgYWRkUHJlZml4KHg6IHN0cmluZyk6IHZvaWQ7XG59XG4vKipcbiAqIFdoZXRoZXIgYW4gb2JqZWN0IGlzIGEgUHJlZml4YWJsZVxuICovXG5mdW5jdGlvbiBpc1ByZWZpeGFibGUoeDogYW55KTogeCBpcyBQcmVmaXhhYmxlIHtcbiAgICByZXR1cm4gdHlwZW9mICh4KSA9PT0gJ29iamVjdCcgJiYgeC5hZGRQcmVmaXg7XG59XG4vKipcbiAqIFdoZXRoZXIgYW4gb2JqZWN0IGlzIElOZXh0YWJsZVxuICovXG5mdW5jdGlvbiBpc05leHRhYmxlKHg6IGFueSk6IHggaXMgSU5leHRhYmxlIHtcbiAgICByZXR1cm4gdHlwZW9mICh4KSA9PT0gJ29iamVjdCcgJiYgeC5uZXh0O1xufVxuIl19