"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SageMakerCreateEndpointConfig = void 0;
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const sfn = require("../../../aws-stepfunctions"); // Automatically re-written from '@aws-cdk/aws-stepfunctions'
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const task_utils_1 = require("../private/task-utils");
/**
 * (experimental) A Step Functions Task to create a SageMaker endpoint configuration.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-sagemaker.html
 * @experimental
 */
class SageMakerCreateEndpointConfig extends sfn.TaskStateBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        this.integrationPattern = props.integrationPattern || sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, SageMakerCreateEndpointConfig.SUPPORTED_INTEGRATION_PATTERNS);
        this.validateProductionVariants();
        this.taskPolicies = this.makePolicyStatements();
    }
    /**
     * @internal
     */
    _renderTask() {
        return {
            Resource: task_utils_1.integrationResourceArn('sagemaker', 'createEndpointConfig', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject(this.renderParameters()),
        };
    }
    renderParameters() {
        var _a, _b;
        return {
            EndpointConfigName: this.props.endpointConfigName,
            Tags: (_a = this.props.tags) === null || _a === void 0 ? void 0 : _a.value,
            KmsKeyId: (_b = this.props.kmsKey) === null || _b === void 0 ? void 0 : _b.keyId,
            ProductionVariants: this.props.productionVariants.map((variant) => ({
                InitialInstanceCount: variant.initialInstanceCount ? variant.initialInstanceCount : 1,
                InstanceType: `ml.${variant.instanceType}`,
                ModelName: variant.modelName,
                VariantName: variant.variantName,
                AcceleratorType: variant.acceleratorType,
                InitialVariantWeight: variant.initialVariantWeight,
            })),
        };
    }
    makePolicyStatements() {
        const stack = cdk.Stack.of(this);
        // https://docs.aws.amazon.com/sagemaker/latest/dg/api-permissions-reference.html
        return [
            new iam.PolicyStatement({
                actions: ['sagemaker:CreateEndpointConfig'],
                resources: [
                    stack.formatArn({
                        service: 'sagemaker',
                        resource: 'endpoint-config',
                        // If the endpoint configuration name comes from input, we cannot target the policy to a particular ARN prefix reliably.
                        // SageMaker uses lowercase for resource name in the arn
                        resourceName: sfn.JsonPath.isEncodedJsonPath(this.props.endpointConfigName) ? '*' : `${this.props.endpointConfigName.toLowerCase()}`,
                    }),
                ],
            }),
            new iam.PolicyStatement({
                actions: ['sagemaker:ListTags'],
                // https://docs.aws.amazon.com/step-functions/latest/dg/sagemaker-iam.html
                resources: ['*'],
            }),
        ];
    }
    validateProductionVariants() {
        if (this.props.productionVariants.length < 1 || this.props.productionVariants.length > 10) {
            throw new Error('Must specify from 1 to 10 production variants per endpoint configuration');
        }
        this.props.productionVariants.forEach((variant) => {
            if (variant.initialInstanceCount && variant.initialInstanceCount < 1)
                throw new Error('Must define at least one instance');
            if (variant.initialVariantWeight && variant.initialVariantWeight <= 0) {
                throw new Error('InitialVariantWeight has minimum value of 0');
            }
        });
    }
}
exports.SageMakerCreateEndpointConfig = SageMakerCreateEndpointConfig;
SageMakerCreateEndpointConfig.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
];
//# sourceMappingURL=data:application/json;base64,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