"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.integrationResourceArn = exports.validatePatternSupported = void 0;
const aws_stepfunctions_1 = require("../../../aws-stepfunctions"); // Automatically re-written from '@aws-cdk/aws-stepfunctions'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * Verifies that a validation pattern is supported for a service integration
 *
 */
function validatePatternSupported(integrationPattern, supportedPatterns) {
    if (!supportedPatterns.includes(integrationPattern)) {
        throw new Error(`Unsupported service integration pattern. Supported Patterns: ${supportedPatterns}. Received: ${integrationPattern}`);
    }
}
exports.validatePatternSupported = validatePatternSupported;
/**
 * Suffixes corresponding to different service integration patterns
 *
 * Key is the service integration pattern, value is the resource ARN suffix.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html
 */
const resourceArnSuffix = {
    [aws_stepfunctions_1.IntegrationPattern.REQUEST_RESPONSE]: '',
    [aws_stepfunctions_1.IntegrationPattern.RUN_JOB]: '.sync',
    [aws_stepfunctions_1.IntegrationPattern.WAIT_FOR_TASK_TOKEN]: '.waitForTaskToken',
};
function integrationResourceArn(service, api, integrationPattern) {
    if (!service || !api) {
        throw new Error("Both 'service' and 'api' must be provided to build the resource ARN.");
    }
    return `arn:${core_1.Aws.PARTITION}:states:::${service}:${api}` +
        (integrationPattern ? resourceArnSuffix[integrationPattern] : '');
}
exports.integrationResourceArn = integrationResourceArn;
//# sourceMappingURL=data:application/json;base64,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