"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DynamoAttributeValue = exports.DynamoProjectionExpression = exports.DynamoReturnValues = exports.DynamoItemCollectionMetrics = exports.DynamoConsumedCapacity = void 0;
const utils_1 = require("./private/utils");
/**
 * (experimental) Determines the level of detail about provisioned throughput consumption that is returned.
 *
 * @experimental
 */
var DynamoConsumedCapacity;
(function (DynamoConsumedCapacity) {
    DynamoConsumedCapacity["INDEXES"] = "INDEXES";
    DynamoConsumedCapacity["TOTAL"] = "TOTAL";
    DynamoConsumedCapacity["NONE"] = "NONE";
})(DynamoConsumedCapacity = exports.DynamoConsumedCapacity || (exports.DynamoConsumedCapacity = {}));
/**
 * (experimental) Determines whether item collection metrics are returned.
 *
 * @experimental
 */
var DynamoItemCollectionMetrics;
(function (DynamoItemCollectionMetrics) {
    DynamoItemCollectionMetrics["SIZE"] = "SIZE";
    DynamoItemCollectionMetrics["NONE"] = "NONE";
})(DynamoItemCollectionMetrics = exports.DynamoItemCollectionMetrics || (exports.DynamoItemCollectionMetrics = {}));
/**
 * (experimental) Use ReturnValues if you want to get the item attributes as they appear before or after they are changed.
 *
 * @experimental
 */
var DynamoReturnValues;
(function (DynamoReturnValues) {
    DynamoReturnValues["NONE"] = "NONE";
    DynamoReturnValues["ALL_OLD"] = "ALL_OLD";
    DynamoReturnValues["UPDATED_OLD"] = "UPDATED_OLD";
    DynamoReturnValues["ALL_NEW"] = "ALL_NEW";
    DynamoReturnValues["UPDATED_NEW"] = "UPDATED_NEW";
})(DynamoReturnValues = exports.DynamoReturnValues || (exports.DynamoReturnValues = {}));
/**
 * (experimental) Class to generate projection expression.
 *
 * @experimental
 */
class DynamoProjectionExpression {
    constructor() {
        this.expression = [];
    }
    /**
     * (experimental) Adds the passed attribute to the chain.
     *
     * @param attr Attribute name.
     * @experimental
     */
    withAttribute(attr) {
        if (this.expression.length) {
            this.expression.push(`.${attr}`);
        }
        else {
            this.expression.push(attr);
        }
        return this;
    }
    /**
     * (experimental) Adds the array literal access for passed index.
     *
     * @param index array index.
     * @experimental
     */
    atIndex(index) {
        if (!this.expression.length) {
            throw new Error('Expression must start with an attribute');
        }
        this.expression.push(`[${index}]`);
        return this;
    }
    /**
     * (experimental) converts and return the string expression.
     *
     * @experimental
     */
    toString() {
        return this.expression.join('');
    }
}
exports.DynamoProjectionExpression = DynamoProjectionExpression;
/**
 * (experimental) Represents the data for an attribute.
 *
 * Each attribute value is described as a name-value pair.
 * The name is the data type, and the value is the data itself.
 *
 * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_AttributeValue.html
 * @experimental
 */
class DynamoAttributeValue {
    constructor(value) {
        this.attributeValue = value;
    }
    /**
     * (experimental) Sets an attribute of type String.
     *
     * For example:  "S": "Hello"
     * Strings may be literal values or as JsonPath
     *
     * @experimental
     */
    static fromString(value) {
        return new DynamoAttributeValue({ S: value });
    }
    /**
     * (experimental) Sets a literal number.
     *
     * For example: 1234
     * Numbers are sent across the network to DynamoDB as strings,
     * to maximize compatibility across languages and libraries.
     * However, DynamoDB treats them as number type attributes for mathematical operations.
     *
     * @experimental
     */
    static fromNumber(value) {
        return new DynamoAttributeValue({ N: value.toString() });
    }
    /**
     * (experimental) Sets an attribute of type Number.
     *
     * For example:  "N": "123.45"
     * Numbers are sent across the network to DynamoDB as strings,
     * to maximize compatibility across languages and libraries.
     * However, DynamoDB treats them as number type attributes for mathematical operations.
     *
     * Numbers may be expressed as literal strings or as JsonPath
     *
     * @experimental
     */
    static numberFromString(value) {
        return new DynamoAttributeValue({ N: value.toString() });
    }
    /**
     * (experimental) Sets an attribute of type Binary.
     *
     * For example:  "B": "dGhpcyB0ZXh0IGlzIGJhc2U2NC1lbmNvZGVk"
     *
     * @param value base-64 encoded string.
     * @experimental
     */
    static fromBinary(value) {
        return new DynamoAttributeValue({ B: value });
    }
    /**
     * (experimental) Sets an attribute of type String Set.
     *
     * For example:  "SS": ["Giraffe", "Hippo" ,"Zebra"]
     *
     * @experimental
     */
    static fromStringSet(value) {
        return new DynamoAttributeValue({ SS: value });
    }
    /**
     * (experimental) Sets an attribute of type Number Set.
     *
     * For example:  "NS": ["42.2", "-19", "7.5", "3.14"]
     * Numbers are sent across the network to DynamoDB as strings,
     * to maximize compatibility across languages and libraries.
     * However, DynamoDB treats them as number type attributes for mathematical operations.
     *
     * @experimental
     */
    static fromNumberSet(value) {
        return new DynamoAttributeValue({ NS: value.map(String) });
    }
    /**
     * (experimental) Sets an attribute of type Number Set.
     *
     * For example:  "NS": ["42.2", "-19", "7.5", "3.14"]
     * Numbers are sent across the network to DynamoDB as strings,
     * to maximize compatibility across languages and libraries.
     * However, DynamoDB treats them as number type attributes for mathematical operations.
     *
     * Numbers may be expressed as literal strings or as JsonPath
     *
     * @experimental
     */
    static numberSetFromStrings(value) {
        return new DynamoAttributeValue({ NS: value });
    }
    /**
     * (experimental) Sets an attribute of type Binary Set.
     *
     * For example:  "BS": ["U3Vubnk=", "UmFpbnk=", "U25vd3k="]
     *
     * @experimental
     */
    static fromBinarySet(value) {
        return new DynamoAttributeValue({ BS: value });
    }
    /**
     * (experimental) Sets an attribute of type Map.
     *
     * For example:  "M": {"Name": {"S": "Joe"}, "Age": {"N": "35"}}
     *
     * @experimental
     */
    static fromMap(value) {
        return new DynamoAttributeValue({ M: utils_1.transformAttributeValueMap(value) });
    }
    /**
     * (experimental) Sets an attribute of type Map.
     *
     * For example:  "M": {"Name": {"S": "Joe"}, "Age": {"N": "35"}}
     *
     * @param value Json path that specifies state input to be used.
     * @experimental
     */
    static mapFromJsonPath(value) {
        utils_1.validateJsonPath(value);
        return new DynamoAttributeValue({ 'M.$': value });
    }
    /**
     * (experimental) Sets an attribute of type List.
     *
     * For example:  "L": [ {"S": "Cookies"} , {"S": "Coffee"}, {"N", "3.14159"}]
     *
     * @experimental
     */
    static fromList(value) {
        return new DynamoAttributeValue({ L: value.map((val) => val.toObject()) });
    }
    /**
     * (experimental) Sets an attribute of type Null.
     *
     * For example:  "NULL": true
     *
     * @experimental
     */
    static fromNull(value) {
        return new DynamoAttributeValue({ NULL: value });
    }
    /**
     * (experimental) Sets an attribute of type Boolean.
     *
     * For example:  "BOOL": true
     *
     * @experimental
     */
    static fromBoolean(value) {
        return new DynamoAttributeValue({ BOOL: value });
    }
    /**
     * (experimental) Sets an attribute of type Boolean from state input through Json path.
     *
     * For example:  "BOOL": true
     *
     * @param value Json path that specifies state input to be used.
     * @experimental
     */
    static booleanFromJsonPath(value) {
        utils_1.validateJsonPath(value);
        return new DynamoAttributeValue({ BOOL: value.toString() });
    }
    /**
     * (experimental) Returns the DynamoDB attribute value.
     *
     * @experimental
     */
    toObject() {
        return this.attributeValue;
    }
}
exports.DynamoAttributeValue = DynamoAttributeValue;
//# sourceMappingURL=data:application/json;base64,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