"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Queue = exports.QueueEncryption = void 0;
const kms = require("../../aws-kms"); // Automatically re-written from '@aws-cdk/aws-kms'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const queue_base_1 = require("./queue-base");
const sqs_generated_1 = require("./sqs.generated");
const validate_props_1 = require("./validate-props");
/**
 * (experimental) What kind of encryption to apply to this queue.
 *
 * @experimental
 */
var QueueEncryption;
(function (QueueEncryption) {
    QueueEncryption["UNENCRYPTED"] = "NONE";
    QueueEncryption["KMS_MANAGED"] = "MANAGED";
    QueueEncryption["KMS"] = "KMS";
})(QueueEncryption = exports.QueueEncryption || (exports.QueueEncryption = {}));
/**
 * (experimental) A new Amazon SQS queue.
 *
 * @experimental
 */
class Queue extends queue_base_1.QueueBase {
    /**
     * @experimental
     */
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.queueName,
        });
        /**
         * (experimental) Controls automatic creation of policy objects.
         *
         * Set by subclasses.
         *
         * @experimental
         */
        this.autoCreatePolicy = true;
        validate_props_1.validateProps(props);
        const redrivePolicy = props.deadLetterQueue
            ? {
                deadLetterTargetArn: props.deadLetterQueue.queue.queueArn,
                maxReceiveCount: props.deadLetterQueue.maxReceiveCount,
            }
            : undefined;
        const { encryptionMasterKey, encryptionProps } = _determineEncryptionProps.call(this);
        const fifoProps = this.determineFifoProps(props);
        this.fifo = fifoProps.fifoQueue || false;
        const queue = new sqs_generated_1.CfnQueue(this, 'Resource', {
            queueName: this.physicalName,
            ...fifoProps,
            ...encryptionProps,
            redrivePolicy,
            delaySeconds: props.deliveryDelay && props.deliveryDelay.toSeconds(),
            maximumMessageSize: props.maxMessageSizeBytes,
            messageRetentionPeriod: props.retentionPeriod && props.retentionPeriod.toSeconds(),
            receiveMessageWaitTimeSeconds: props.receiveMessageWaitTime && props.receiveMessageWaitTime.toSeconds(),
            visibilityTimeout: props.visibilityTimeout && props.visibilityTimeout.toSeconds(),
        });
        this.queueArn = this.getResourceArnAttribute(queue.attrArn, {
            service: 'sqs',
            resource: this.physicalName,
        });
        this.queueName = this.getResourceNameAttribute(queue.attrQueueName);
        this.encryptionMasterKey = encryptionMasterKey;
        this.queueUrl = queue.ref;
        function _determineEncryptionProps() {
            let encryption = props.encryption || QueueEncryption.UNENCRYPTED;
            if (encryption !== QueueEncryption.KMS && props.encryptionMasterKey) {
                encryption = QueueEncryption.KMS; // KMS is implied by specifying an encryption key
            }
            if (encryption === QueueEncryption.UNENCRYPTED) {
                return { encryptionProps: {} };
            }
            if (encryption === QueueEncryption.KMS_MANAGED) {
                return {
                    encryptionProps: {
                        kmsMasterKeyId: 'alias/aws/sqs',
                        kmsDataKeyReusePeriodSeconds: props.dataKeyReuse && props.dataKeyReuse.toSeconds(),
                    },
                };
            }
            if (encryption === QueueEncryption.KMS) {
                const masterKey = props.encryptionMasterKey || new kms.Key(this, 'Key', {
                    description: `Created by ${this.node.path}`,
                });
                return {
                    encryptionMasterKey: masterKey,
                    encryptionProps: {
                        kmsMasterKeyId: masterKey.keyArn,
                        kmsDataKeyReusePeriodSeconds: props.dataKeyReuse && props.dataKeyReuse.toSeconds(),
                    },
                };
            }
            throw new Error(`Unexpected 'encryptionType': ${encryption}`);
        }
    }
    /**
     * (experimental) Import an existing SQS queue provided an ARN.
     *
     * @param scope The parent creating construct.
     * @param id The construct's name.
     * @param queueArn queue ARN (i.e. arn:aws:sqs:us-east-2:444455556666:queue1).
     * @experimental
     */
    static fromQueueArn(scope, id, queueArn) {
        return Queue.fromQueueAttributes(scope, id, { queueArn });
    }
    /**
     * (experimental) Import an existing queue.
     *
     * @experimental
     */
    static fromQueueAttributes(scope, id, attrs) {
        const stack = core_1.Stack.of(scope);
        const queueName = attrs.queueName || stack.parseArn(attrs.queueArn).resource;
        const queueUrl = attrs.queueUrl || `https://sqs.${stack.region}.${stack.urlSuffix}/${stack.account}/${queueName}`;
        class Import extends queue_base_1.QueueBase {
            constructor() {
                super(...arguments);
                this.queueArn = attrs.queueArn; // arn:aws:sqs:us-east-1:123456789012:queue1
                this.queueUrl = queueUrl;
                this.queueName = queueName;
                this.encryptionMasterKey = attrs.keyArn
                    ? kms.Key.fromKeyArn(this, 'Key', attrs.keyArn)
                    : undefined;
                this.fifo = queueName.endsWith('.fifo') ? true : false;
                this.autoCreatePolicy = false;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Look at the props, see if the FIFO props agree, and return the correct subset of props
     */
    determineFifoProps(props) {
        // Check if any of the signals that we have say that this is a FIFO queue.
        let fifoQueue = props.fifo;
        const queueName = props.queueName;
        if (typeof fifoQueue === 'undefined' && queueName && !core_1.Token.isUnresolved(queueName) && queueName.endsWith('.fifo')) {
            fifoQueue = true;
        }
        if (typeof fifoQueue === 'undefined' && props.contentBasedDeduplication) {
            fifoQueue = true;
        }
        // If we have a name, see that it agrees with the FIFO setting
        if (typeof queueName === 'string') {
            if (fifoQueue && !queueName.endsWith('.fifo')) {
                throw new Error("FIFO queue names must end in '.fifo'");
            }
            if (!fifoQueue && queueName.endsWith('.fifo')) {
                throw new Error("Non-FIFO queue name may not end in '.fifo'");
            }
        }
        if (props.contentBasedDeduplication && !fifoQueue) {
            throw new Error('Content-based deduplication can only be defined for FIFO queues');
        }
        return {
            contentBasedDeduplication: props.contentBasedDeduplication,
            fifoQueue,
        };
    }
}
exports.Queue = Queue;
//# sourceMappingURL=data:application/json;base64,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