"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Asset = void 0;
const fs = require("fs");
const path = require("path");
const kms = require("../../aws-kms"); // Automatically re-written from '@aws-cdk/aws-kms'
const s3 = require("../../aws-s3"); // Automatically re-written from '@aws-cdk/aws-s3'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cxapi = require("../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
const compat_1 = require("./compat");
const ARCHIVE_EXTENSIONS = ['.zip', '.jar'];
/**
 * (experimental) An asset represents a local file or directory, which is automatically uploaded to S3 and then can be referenced within a CDK application.
 *
 * @experimental
 */
class Asset extends cdk.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        // stage the asset source (conditionally).
        const staging = new cdk.AssetStaging(this, 'Stage', {
            ...props,
            sourcePath: path.resolve(props.path),
            follow: compat_1.toSymlinkFollow(props.follow),
            assetHash: (_a = props.assetHash) !== null && _a !== void 0 ? _a : props.sourceHash,
        });
        this.assetHash = staging.assetHash;
        this.sourceHash = this.assetHash;
        const stack = cdk.Stack.of(this);
        this.assetPath = staging.relativeStagedPath(stack);
        const packaging = determinePackaging(staging.sourcePath);
        // sets isZipArchive based on the type of packaging and file extension
        this.isZipArchive = packaging === cdk.FileAssetPackaging.ZIP_DIRECTORY
            ? true
            : ARCHIVE_EXTENSIONS.some(ext => staging.sourcePath.toLowerCase().endsWith(ext));
        const location = stack.synthesizer.addFileAsset({
            packaging,
            sourceHash: this.sourceHash,
            fileName: this.assetPath,
        });
        this.s3BucketName = location.bucketName;
        this.s3ObjectKey = location.objectKey;
        this.s3ObjectUrl = location.s3ObjectUrl;
        this.httpUrl = location.httpUrl;
        this.s3Url = location.httpUrl; // for backwards compatibility
        const kmsKey = location.kmsKeyArn ? kms.Key.fromKeyArn(this, 'Key', location.kmsKeyArn) : undefined;
        this.bucket = s3.Bucket.fromBucketAttributes(this, 'AssetBucket', {
            bucketName: this.s3BucketName,
            encryptionKey: kmsKey,
        });
        for (const reader of ((_b = props.readers) !== null && _b !== void 0 ? _b : [])) {
            this.grantRead(reader);
        }
    }
    /**
     * (experimental) Adds CloudFormation template metadata to the specified resource with information that indicates which resource property is mapped to this local asset.
     *
     * This can be used by tools such as SAM CLI to provide local
     * experience such as local invocation and debugging of Lambda functions.
     *
     * Asset metadata will only be included if the stack is synthesized with the
     * "aws:cdk:enable-asset-metadata" context key defined, which is the default
     * behavior when synthesizing via the CDK Toolkit.
     *
     * @param resource The CloudFormation resource which is using this asset [disable-awslint:ref-via-interface].
     * @param resourceProperty The property name where this asset is referenced (e.g. "Code" for AWS::Lambda::Function).
     * @see https://github.com/aws/aws-cdk/issues/1432
     * @experimental
     */
    addResourceMetadata(resource, resourceProperty) {
        if (!this.node.tryGetContext(cxapi.ASSET_RESOURCE_METADATA_ENABLED_CONTEXT)) {
            return; // not enabled
        }
        // tell tools such as SAM CLI that the "Code" property of this resource
        // points to a local path in order to enable local invocation of this function.
        resource.cfnOptions.metadata = resource.cfnOptions.metadata || {};
        resource.cfnOptions.metadata[cxapi.ASSET_RESOURCE_METADATA_PATH_KEY] = this.assetPath;
        resource.cfnOptions.metadata[cxapi.ASSET_RESOURCE_METADATA_PROPERTY_KEY] = resourceProperty;
    }
    /**
     * (experimental) Grants read permissions to the principal on the assets bucket.
     *
     * @experimental
     */
    grantRead(grantee) {
        // we give permissions on all files in the bucket since we don't want to
        // accidentally revoke permission on old versions when deploying a new
        // version (for example, when using Lambda traffic shifting).
        this.bucket.grantRead(grantee);
    }
}
exports.Asset = Asset;
function determinePackaging(assetPath) {
    if (!fs.existsSync(assetPath)) {
        throw new Error(`Cannot find asset at ${assetPath}`);
    }
    if (fs.statSync(assetPath).isDirectory()) {
        return cdk.FileAssetPackaging.ZIP_DIRECTORY;
    }
    if (fs.statSync(assetPath).isFile()) {
        return cdk.FileAssetPackaging.FILE;
    }
    throw new Error(`Asset ${assetPath} is expected to be either a directory or a regular file`);
}
//# sourceMappingURL=data:application/json;base64,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