"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.makeHostedZoneArn = exports.determineFullyQualifiedDomainName = exports.validateZoneName = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * Validates a zone name is valid by Route53 specifc naming rules,
 * and that there is no trailing dot in the name.
 *
 * @param zoneName the zone name to be validated.
 * @returns +zoneName+
 * @throws ValidationError if the name is not valid.
 */
function validateZoneName(zoneName) {
    if (zoneName.endsWith('.')) {
        throw new ValidationError('zone name must not end with a trailing dot');
    }
    if (zoneName.length > 255) {
        throw new ValidationError('zone name cannot be more than 255 bytes long');
    }
    if (zoneName.split('.').find(label => label.length > 63)) {
        throw new ValidationError('zone name labels cannot be more than 63 bytes long');
    }
    if (!zoneName.match(/^[a-z0-9!"#$%&'()*+,/:;<=>?@[\\\]^_`{|}~.-]+$/i)) {
        throw new ValidationError('zone names can only contain a-z, 0-9, -, ! " # $ % & \' ( ) * + , - / : ; < = > ? @ [ \ ] ^ _ ` { | } ~ .');
    }
}
exports.validateZoneName = validateZoneName;
class ValidationError extends Error {
    constructor(message) {
        super(message);
    }
}
/**
 * Route53 requires the record names are specified as fully qualified names, but this
 * forces lots of redundant work on the user (repeating the zone name over and over).
 * This function allows the user to be lazier and offers a nicer experience, by
 * qualifying relative names appropriately:
 *
 * @param providedName the user-specified name of the record.
 * @param zoneName   the fully-qualified name of the zone the record will be created in.
 *
 * @returns <ul>
 *        <li>If +providedName+ ends with a +.+, use it as-is</li>
 *        <li>If +providedName+ ends with or equals +zoneName+, append a trailing +.+</li>
 *        <li>Otherwise, append +.+, +zoneName+ and a trailing +.+</li>
 *      </ul>
 */
function determineFullyQualifiedDomainName(providedName, hostedZone) {
    if (providedName.endsWith('.')) {
        return providedName;
    }
    const hostedZoneName = hostedZone.zoneName.endsWith('.')
        ? hostedZone.zoneName.substring(0, hostedZone.zoneName.length - 1)
        : hostedZone.zoneName;
    const suffix = `.${hostedZoneName}`;
    if (providedName.endsWith(suffix) || providedName === hostedZoneName) {
        return `${providedName}.`;
    }
    return `${providedName}${suffix}.`;
}
exports.determineFullyQualifiedDomainName = determineFullyQualifiedDomainName;
function makeHostedZoneArn(construct, hostedZoneId) {
    return core_1.Stack.of(construct).formatArn({
        account: '',
        region: '',
        service: 'route53',
        resource: 'hostedzone',
        resourceName: hostedZoneId,
    });
}
exports.makeHostedZoneArn = makeHostedZoneArn;
//# sourceMappingURL=data:application/json;base64,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