"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OptionGroup = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const rds_generated_1 = require("./rds.generated");
/**
 * (experimental) An option group.
 *
 * @experimental
 */
class OptionGroup extends core_1.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        /**
         * (experimental) The connections object for the options.
         *
         * @experimental
         */
        this.optionConnections = {};
        this.configurations = [];
        const majorEngineVersion = (_a = props.engine.engineVersion) === null || _a === void 0 ? void 0 : _a.majorVersion;
        if (!majorEngineVersion) {
            throw new Error("OptionGroup cannot be used with an engine that doesn't specify a version");
        }
        props.configurations.forEach(config => this.addConfiguration(config));
        const optionGroup = new rds_generated_1.CfnOptionGroup(this, 'Resource', {
            engineName: props.engine.engineType,
            majorEngineVersion,
            optionGroupDescription: props.description || `Option group for ${props.engine.engineType} ${majorEngineVersion}`,
            optionConfigurations: core_1.Lazy.anyValue({ produce: () => this.renderConfigurations(this.configurations) }),
        });
        this.optionGroupName = optionGroup.ref;
    }
    /**
     * (experimental) Import an existing option group.
     *
     * @experimental
     */
    static fromOptionGroupName(scope, id, optionGroupName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.optionGroupName = optionGroupName;
            }
            addConfiguration(_) { return false; }
        }
        return new Import(scope, id);
    }
    /**
     * (experimental) Adds a configuration to this OptionGroup.
     *
     * This method is a no-op for an imported OptionGroup.
     *
     * @experimental
     */
    addConfiguration(configuration) {
        this.configurations.push(configuration);
        if (configuration.port) {
            if (!configuration.vpc) {
                throw new Error('`port` and `vpc` must be specified together.');
            }
            const securityGroups = configuration.securityGroups && configuration.securityGroups.length > 0
                ? configuration.securityGroups
                : [new ec2.SecurityGroup(this, `SecurityGroup${configuration.name}`, {
                        description: `Security group for ${configuration.name} option`,
                        vpc: configuration.vpc,
                    })];
            this.optionConnections[configuration.name] = new ec2.Connections({
                securityGroups: securityGroups,
                defaultPort: ec2.Port.tcp(configuration.port),
            });
        }
        return true;
    }
    /**
     * Renders the option configurations specifications.
     */
    renderConfigurations(configurations) {
        const configs = [];
        for (const config of configurations) {
            const securityGroups = config.vpc
                ? this.optionConnections[config.name].securityGroups.map(sg => sg.securityGroupId)
                : undefined;
            configs.push({
                optionName: config.name,
                optionSettings: config.settings && Object.entries(config.settings).map(([name, value]) => ({ name, value })),
                optionVersion: config.version,
                port: config.port,
                vpcSecurityGroupMemberships: securityGroups,
            });
        }
        return configs;
    }
}
exports.OptionGroup = OptionGroup;
//# sourceMappingURL=data:application/json;base64,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