"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LayerVersion = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const lambda_generated_1 = require("./lambda.generated");
const runtime_1 = require("./runtime");
/**
 * A reference to a Lambda Layer version.
 */
class LayerVersionBase extends core_1.Resource {
    /**
     * (experimental) Add permission for this layer version to specific entities.
     *
     * Usage within
     * the same account where the layer is defined is always allowed and does not
     * require calling this method. Note that the principal that creates the
     * Lambda function using the layer (for example, a CloudFormation changeset
     * execution role) also needs to have the ``lambda:GetLayerVersion``
     * permission on the layer version.
     *
     * @experimental
     */
    addPermission(id, permission) {
        if (permission.organizationId != null && permission.accountId !== '*') {
            throw new Error(`OrganizationId can only be specified if AwsAccountId is '*', but it is ${permission.accountId}`);
        }
        new lambda_generated_1.CfnLayerVersionPermission(this, id, {
            action: 'lambda:GetLayerVersion',
            layerVersionArn: this.layerVersionArn,
            principal: permission.accountId,
            organizationId: permission.organizationId,
        });
    }
}
/**
 * (experimental) Defines a new Lambda Layer version.
 *
 * @experimental
 */
class LayerVersion extends LayerVersionBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.layerVersionName,
        });
        if (props.compatibleRuntimes && props.compatibleRuntimes.length === 0) {
            throw new Error('Attempted to define a Lambda layer that supports no runtime!');
        }
        if (props.code.isInline) {
            throw new Error('Lambda layers cannot be created from inline code');
        }
        // Allow usage of the code in this context...
        const code = props.code.bind(this);
        if (code.inlineCode) {
            throw new Error('Inline code is not supported for AWS Lambda layers');
        }
        if (!code.s3Location) {
            throw new Error('Code must define an S3 location');
        }
        const resource = new lambda_generated_1.CfnLayerVersion(this, 'Resource', {
            compatibleRuntimes: props.compatibleRuntimes && props.compatibleRuntimes.map(r => r.name),
            content: {
                s3Bucket: code.s3Location.bucketName,
                s3Key: code.s3Location.objectKey,
                s3ObjectVersion: code.s3Location.objectVersion,
            },
            description: props.description,
            layerName: this.physicalName,
            licenseInfo: props.license,
        });
        props.code.bindToResource(resource, {
            resourceProperty: 'Content',
        });
        this.layerVersionArn = resource.ref;
        this.compatibleRuntimes = props.compatibleRuntimes;
    }
    /**
     * (experimental) Imports a layer version by ARN.
     *
     * Assumes it is compatible with all Lambda runtimes.
     *
     * @experimental
     */
    static fromLayerVersionArn(scope, id, layerVersionArn) {
        return LayerVersion.fromLayerVersionAttributes(scope, id, {
            layerVersionArn,
            compatibleRuntimes: runtime_1.Runtime.ALL,
        });
    }
    /**
     * (experimental) Imports a Layer that has been defined externally.
     *
     * @param scope the parent Construct that will use the imported layer.
     * @param id the id of the imported layer in the construct tree.
     * @param attrs the properties of the imported layer.
     * @experimental
     */
    static fromLayerVersionAttributes(scope, id, attrs) {
        if (attrs.compatibleRuntimes && attrs.compatibleRuntimes.length === 0) {
            throw new Error('Attempted to import a Lambda layer that supports no runtime!');
        }
        class Import extends LayerVersionBase {
            constructor() {
                super(...arguments);
                this.layerVersionArn = attrs.layerVersionArn;
                this.compatibleRuntimes = attrs.compatibleRuntimes;
            }
        }
        return new Import(scope, id);
    }
}
exports.LayerVersion = LayerVersion;
//# sourceMappingURL=data:application/json;base64,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