"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Alias = void 0;
const appscaling = require("../../aws-applicationautoscaling"); // Automatically re-written from '@aws-cdk/aws-applicationautoscaling'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const function_base_1 = require("./function-base");
const lambda_version_1 = require("./lambda-version");
const lambda_generated_1 = require("./lambda.generated");
const scalable_function_attribute_1 = require("./private/scalable-function-attribute");
/**
 * (experimental) A new alias to a particular version of a Lambda function.
 *
 * @experimental
 */
class Alias extends function_base_1.QualifiedFunctionBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.aliasName,
        });
        /**
         * (experimental) Whether the addPermission() call adds any permissions.
         *
         * True for new Lambdas, false for version $LATEST and imported Lambdas
         * from different accounts.
         *
         * @experimental
         */
        this.canCreatePermissions = true;
        this.lambda = props.version.lambda;
        this.aliasName = this.physicalName;
        this.version = props.version;
        const alias = new lambda_generated_1.CfnAlias(this, 'Resource', {
            name: this.aliasName,
            description: props.description,
            functionName: this.version.lambda.functionName,
            functionVersion: props.version.version,
            routingConfig: this.determineRoutingConfig(props),
            provisionedConcurrencyConfig: this.determineProvisionedConcurrency(props),
        });
        // Use a Service Linked Role
        // https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-service-linked-roles.html
        this.scalingRole = iam.Role.fromRoleArn(this, 'ScalingRole', this.stack.formatArn({
            service: 'iam',
            region: '',
            resource: 'role/aws-service-role/lambda.application-autoscaling.amazonaws.com',
            resourceName: 'AWSServiceRoleForApplicationAutoScaling_LambdaConcurrency',
        }));
        this.functionArn = this.getResourceArnAttribute(alias.ref, {
            service: 'lambda',
            resource: 'function',
            resourceName: `${this.lambda.functionName}:${this.physicalName}`,
            sep: ':',
        });
        this.qualifier = lambda_version_1.extractQualifierFromArn(alias.ref);
        if (props.onFailure || props.onSuccess || props.maxEventAge || props.retryAttempts !== undefined) {
            this.configureAsyncInvoke({
                onFailure: props.onFailure,
                onSuccess: props.onSuccess,
                maxEventAge: props.maxEventAge,
                retryAttempts: props.retryAttempts,
            });
        }
        // ARN parsing splits on `:`, so we can only get the function's name from the ARN as resourceName...
        // And we're parsing it out (instead of using the underlying function directly) in order to have use of it incur
        // an implicit dependency on the resource.
        this.functionName = `${this.stack.parseArn(this.functionArn, ':').resourceName}:${this.aliasName}`;
    }
    /**
     * @experimental
     */
    static fromAliasAttributes(scope, id, attrs) {
        class Imported extends function_base_1.QualifiedFunctionBase {
            constructor() {
                super(...arguments);
                this.aliasName = attrs.aliasName;
                this.version = attrs.aliasVersion;
                this.lambda = attrs.aliasVersion.lambda;
                this.functionArn = `${attrs.aliasVersion.lambda.functionArn}:${attrs.aliasName}`;
                this.functionName = `${attrs.aliasVersion.lambda.functionName}:${attrs.aliasName}`;
                this.grantPrincipal = attrs.aliasVersion.grantPrincipal;
                this.role = attrs.aliasVersion.role;
                this.canCreatePermissions = this._isStackAccount();
                this.qualifier = attrs.aliasName;
            }
        }
        return new Imported(scope, id);
    }
    /**
     * (experimental) The principal this Lambda Function is running as.
     *
     * @experimental
     */
    get grantPrincipal() {
        return this.version.grantPrincipal;
    }
    /**
     * (experimental) The IAM role associated with this function.
     *
     * Undefined if the function was imported without a role.
     *
     * @experimental
     */
    get role() {
        return this.version.role;
    }
    /**
     * (experimental) Return the given named metric for this Function.
     *
     * @experimental
     */
    metric(metricName, props = {}) {
        // Metrics on Aliases need the "bare" function name, and the alias' ARN, this differes from the base behavior.
        return super.metric(metricName, {
            dimensions: {
                FunctionName: this.lambda.functionName,
                // construct the name from the underlying lambda so that alarms on an alias
                // don't cause a circular dependency with CodeDeploy
                // see: https://github.com/aws/aws-cdk/issues/2231
                Resource: `${this.lambda.functionName}:${this.aliasName}`,
            },
            ...props,
        });
    }
    /**
     * (experimental) Configure provisioned concurrency autoscaling on a function alias.
     *
     * Returns a scalable attribute that can call
     * `scaleOnUtilization()` and `scaleOnSchedule()`.
     *
     * @param options Autoscaling options.
     * @experimental
     */
    addAutoScaling(options) {
        var _a;
        if (this.scalableAlias) {
            throw new Error('AutoScaling already enabled for this alias');
        }
        return this.scalableAlias = new scalable_function_attribute_1.ScalableFunctionAttribute(this, 'AliasScaling', {
            minCapacity: (_a = options.minCapacity) !== null && _a !== void 0 ? _a : 1,
            maxCapacity: options.maxCapacity,
            resourceId: `function:${this.functionName}`,
            dimension: 'lambda:function:ProvisionedConcurrency',
            serviceNamespace: appscaling.ServiceNamespace.LAMBDA,
            role: this.scalingRole,
        });
    }
    /**
     * Calculate the routingConfig parameter from the input props
     */
    determineRoutingConfig(props) {
        if (!props.additionalVersions || props.additionalVersions.length === 0) {
            return undefined;
        }
        this.validateAdditionalWeights(props.additionalVersions);
        return {
            additionalVersionWeights: props.additionalVersions.map(vw => {
                return {
                    functionVersion: vw.version.version,
                    functionWeight: vw.weight,
                };
            }),
        };
    }
    /**
     * Validate that the additional version weights make sense
     *
     * We validate that they are positive and add up to something <= 1.
     */
    validateAdditionalWeights(weights) {
        const total = weights.map(w => {
            if (w.weight < 0 || w.weight > 1) {
                throw new Error(`Additional version weight must be between 0 and 1, got: ${w.weight}`);
            }
            return w.weight;
        }).reduce((a, x) => a + x);
        if (total > 1) {
            throw new Error(`Sum of additional version weights must not exceed 1, got: ${total}`);
        }
    }
    /**
     * Validate that the provisionedConcurrentExecutions makes sense
     *
     * Member must have value greater than or equal to 1
     */
    determineProvisionedConcurrency(props) {
        if (!props.provisionedConcurrentExecutions) {
            return undefined;
        }
        if (props.provisionedConcurrentExecutions <= 0) {
            throw new Error('provisionedConcurrentExecutions must have value greater than or equal to 1');
        }
        return { provisionedConcurrentExecutions: props.provisionedConcurrentExecutions };
    }
}
exports.Alias = Alias;
//# sourceMappingURL=data:application/json;base64,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