"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.hasDependencies = exports.bundle = exports.BUNDLER_DEPENDENCIES_CACHE = exports.DEPENDENCY_EXCLUDES = void 0;
const fs = require("fs");
const path = require("path");
const lambda = require("../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * Dependency files to exclude from the asset hash.
 */
exports.DEPENDENCY_EXCLUDES = ['*.pyc'];
/**
 * The location in the image that the bundler image caches dependencies.
 */
exports.BUNDLER_DEPENDENCIES_CACHE = '/var/dependencies';
/**
 * Produce bundled Lambda asset code
 */
function bundle(options) {
    const { entry, runtime, outputPathSuffix } = options;
    const hasDeps = hasDependencies(entry);
    const depsCommand = chain([
        hasDeps ? `rsync -r ${exports.BUNDLER_DEPENDENCIES_CACHE}/. ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR}/${outputPathSuffix}` : '',
        `rsync -r . ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR}/${outputPathSuffix}`,
    ]);
    // Determine which dockerfile to use. When dependencies are present, we use a
    // Dockerfile that can create a cacheable layer. We can't use this Dockerfile
    // if there aren't dependencies or the Dockerfile will complain about missing
    // sources.
    const dockerfile = hasDeps
        ? 'Dockerfile.dependencies'
        : 'Dockerfile';
    const image = cdk.BundlingDockerImage.fromAsset(entry, {
        buildArgs: {
            IMAGE: runtime.bundlingDockerImage.image,
        },
        file: path.join(__dirname, dockerfile),
    });
    return lambda.Code.fromAsset(entry, {
        assetHashType: cdk.AssetHashType.BUNDLE,
        exclude: exports.DEPENDENCY_EXCLUDES,
        bundling: {
            image,
            command: ['bash', '-c', depsCommand],
        },
    });
}
exports.bundle = bundle;
/**
 * Checks to see if the `entry` directory contains a type of dependency that
 * we know how to install.
 */
function hasDependencies(entry) {
    if (fs.existsSync(path.join(entry, 'Pipfile'))) {
        return true;
    }
    if (fs.existsSync(path.join(entry, 'requirements.txt'))) {
        return true;
    }
    return false;
}
exports.hasDependencies = hasDependencies;
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
//# sourceMappingURL=data:application/json;base64,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