"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NodejsFunction = void 0;
const fs = require("fs");
const path = require("path");
const lambda = require("../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const bundling_1 = require("./bundling");
const package_json_manager_1 = require("./package-json-manager");
const util_1 = require("./util");
/**
 * (experimental) A Node.js Lambda function bundled using Parcel.
 *
 * @experimental
 */
class NodejsFunction extends lambda.Function {
    /**
     * @experimental
     */
    constructor(scope, id, props = {}) {
        var _a, _b, _c;
        if (props.runtime && props.runtime.family !== lambda.RuntimeFamily.NODEJS) {
            throw new Error('Only `NODEJS` runtimes are supported.');
        }
        // Entry and defaults
        const entry = path.resolve(findEntry(id, props.entry));
        const handler = (_a = props.handler) !== null && _a !== void 0 ? _a : 'handler';
        const defaultRunTime = util_1.nodeMajorVersion() >= 12
            ? lambda.Runtime.NODEJS_12_X
            : lambda.Runtime.NODEJS_10_X;
        const runtime = (_b = props.runtime) !== null && _b !== void 0 ? _b : defaultRunTime;
        // Look for the closest package.json starting in the directory of the entry
        // file. We need to restore it after bundling.
        const packageJsonManager = new package_json_manager_1.PackageJsonManager(path.dirname(entry));
        try {
            super(scope, id, {
                ...props,
                runtime,
                code: bundling_1.Bundling.parcel({
                    ...props,
                    entry,
                    runtime,
                }),
                handler: `index.${handler}`,
            });
            // Enable connection reuse for aws-sdk
            if ((_c = props.awsSdkConnectionReuse) !== null && _c !== void 0 ? _c : true) {
                this.addEnvironment('AWS_NODEJS_CONNECTION_REUSE_ENABLED', '1', { removeInEdge: true });
            }
        }
        finally {
            // We can only restore after the code has been bound to the function
            packageJsonManager.restore();
        }
    }
}
exports.NodejsFunction = NodejsFunction;
/**
 * Searches for an entry file. Preference order is the following:
 * 1. Given entry file
 * 2. A .ts file named as the defining file with id as suffix (defining-file.id.ts)
 * 3. A .js file name as the defining file with id as suffix (defining-file.id.js)
 */
function findEntry(id, entry) {
    if (entry) {
        if (!/\.(jsx?|tsx?)$/.test(entry)) {
            throw new Error('Only JavaScript or TypeScript entry files are supported.');
        }
        if (!fs.existsSync(entry)) {
            throw new Error(`Cannot find entry file at ${entry}`);
        }
        return entry;
    }
    const definingFile = findDefiningFile();
    const extname = path.extname(definingFile);
    const tsHandlerFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}.ts`);
    if (fs.existsSync(tsHandlerFile)) {
        return tsHandlerFile;
    }
    const jsHandlerFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}.js`);
    if (fs.existsSync(jsHandlerFile)) {
        return jsHandlerFile;
    }
    throw new Error('Cannot find entry file.');
}
/**
 * Finds the name of the file where the `NodejsFunction` is defined
 */
function findDefiningFile() {
    const stackTrace = util_1.parseStackTrace();
    const functionIndex = stackTrace.findIndex(s => /NodejsFunction/.test(s.methodName || ''));
    if (functionIndex === -1 || !stackTrace[functionIndex + 1]) {
        throw new Error('Cannot find defining file.');
    }
    return stackTrace[functionIndex + 1].file;
}
//# sourceMappingURL=data:application/json;base64,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