"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StreamEncryption = exports.Stream = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const kms = require("../../aws-kms"); // Automatically re-written from '@aws-cdk/aws-kms'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const kinesis_generated_1 = require("./kinesis.generated");
const READ_OPERATIONS = [
    'kinesis:DescribeStreamSummary',
    'kinesis:GetRecords',
    'kinesis:GetShardIterator',
    'kinesis:ListShards',
    'kinesis:SubscribeToShard',
];
const WRITE_OPERATIONS = [
    'kinesis:ListShards',
    'kinesis:PutRecord',
    'kinesis:PutRecords',
];
/**
 * Represents a Kinesis Stream.
 */
class StreamBase extends core_1.Resource {
    /**
     * (experimental) Grant write permissions for this stream and its contents to an IAM principal (Role/Group/User).
     *
     * If an encryption key is used, permission to ues the key to decrypt the
     * contents of the stream will also be granted.
     *
     * @experimental
     */
    grantRead(grantee) {
        const ret = this.grant(grantee, ...READ_OPERATIONS);
        if (this.encryptionKey) {
            this.encryptionKey.grantDecrypt(grantee);
        }
        return ret;
    }
    /**
     * (experimental) Grant read permissions for this stream and its contents to an IAM principal (Role/Group/User).
     *
     * If an encryption key is used, permission to ues the key to decrypt the
     * contents of the stream will also be granted.
     *
     * @experimental
     */
    grantWrite(grantee) {
        var _a;
        const ret = this.grant(grantee, ...WRITE_OPERATIONS);
        (_a = this.encryptionKey) === null || _a === void 0 ? void 0 : _a.grantEncrypt(grantee);
        return ret;
    }
    /**
     * (experimental) Grants read/write permissions for this stream and its contents to an IAM principal (Role/Group/User).
     *
     * If an encryption key is used, permission to use the key for
     * encrypt/decrypt will also be granted.
     *
     * @experimental
     */
    grantReadWrite(grantee) {
        var _a;
        const ret = this.grant(grantee, ...Array.from(new Set([...READ_OPERATIONS, ...WRITE_OPERATIONS])));
        (_a = this.encryptionKey) === null || _a === void 0 ? void 0 : _a.grantEncryptDecrypt(grantee);
        return ret;
    }
    /**
     * (experimental) Grant the indicated permissions on this stream to the given IAM principal (Role/Group/User).
     *
     * @experimental
     */
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [this.streamArn],
            scope: this,
        });
    }
}
/**
 * (experimental) A Kinesis stream.
 *
 * Can be encrypted with a KMS key.
 *
 * @experimental
 */
class Stream extends StreamBase {
    /**
     * @experimental
     */
    constructor(scope, id, props = {}) {
        var _a, _b;
        super(scope, id, {
            physicalName: props.streamName,
        });
        const shardCount = props.shardCount || 1;
        const retentionPeriodHours = (_b = (_a = props.retentionPeriod) === null || _a === void 0 ? void 0 : _a.toHours()) !== null && _b !== void 0 ? _b : 24;
        if (!core_1.Token.isUnresolved(retentionPeriodHours)) {
            if (retentionPeriodHours < 24 || retentionPeriodHours > 168) {
                throw new Error(`retentionPeriod must be between 24 and 168 hours. Received ${retentionPeriodHours}`);
            }
        }
        const { streamEncryption, encryptionKey } = this.parseEncryption(props);
        this.stream = new kinesis_generated_1.CfnStream(this, 'Resource', {
            name: this.physicalName,
            retentionPeriodHours,
            shardCount,
            streamEncryption,
        });
        this.streamArn = this.getResourceArnAttribute(this.stream.attrArn, {
            service: 'kinesis',
            resource: 'stream',
            resourceName: this.physicalName,
        });
        this.streamName = this.getResourceNameAttribute(this.stream.ref);
        this.encryptionKey = encryptionKey;
    }
    /**
     * (experimental) Import an existing Kinesis Stream provided an ARN.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param streamArn Stream ARN (i.e. arn:aws:kinesis:<region>:<account-id>:stream/Foo).
     * @experimental
     */
    static fromStreamArn(scope, id, streamArn) {
        return Stream.fromStreamAttributes(scope, id, { streamArn });
    }
    /**
     * (experimental) Creates a Stream construct that represents an external stream.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs Stream import properties.
     * @experimental
     */
    static fromStreamAttributes(scope, id, attrs) {
        class Import extends StreamBase {
            constructor() {
                super(...arguments);
                this.streamArn = attrs.streamArn;
                this.streamName = core_1.Stack.of(scope).parseArn(attrs.streamArn).resourceName;
                this.encryptionKey = attrs.encryptionKey;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Set up key properties and return the Stream encryption property from the
     * user's configuration.
     */
    parseEncryption(props) {
        var _a;
        // if encryption properties are not set, default to KMS in regions where KMS is available
        if (!props.encryption && !props.encryptionKey) {
            const conditionName = 'AwsCdkKinesisEncryptedStreamsUnsupportedRegions';
            const existing = core_1.Stack.of(this).node.tryFindChild(conditionName);
            // create a single condition for the Stack
            if (!existing) {
                new core_1.CfnCondition(core_1.Stack.of(this), conditionName, {
                    expression: core_1.Fn.conditionOr(core_1.Fn.conditionEquals(core_1.Aws.REGION, 'cn-north-1'), core_1.Fn.conditionEquals(core_1.Aws.REGION, 'cn-northwest-1')),
                });
            }
            return {
                streamEncryption: core_1.Fn.conditionIf(conditionName, core_1.Aws.NO_VALUE, { EncryptionType: 'KMS', KeyId: 'alias/aws/kinesis' }),
            };
        }
        // default based on whether encryption key is specified
        const encryptionType = (_a = props.encryption) !== null && _a !== void 0 ? _a : (props.encryptionKey ? StreamEncryption.KMS : StreamEncryption.UNENCRYPTED);
        // if encryption key is set, encryption must be set to KMS.
        if (encryptionType !== StreamEncryption.KMS && props.encryptionKey) {
            throw new Error(`encryptionKey is specified, so 'encryption' must be set to KMS (value: ${encryptionType})`);
        }
        if (encryptionType === StreamEncryption.UNENCRYPTED) {
            return {};
        }
        if (encryptionType === StreamEncryption.MANAGED) {
            const encryption = { encryptionType: 'KMS', keyId: 'alias/aws/kinesis' };
            return { streamEncryption: encryption };
        }
        if (encryptionType === StreamEncryption.KMS) {
            const encryptionKey = props.encryptionKey || new kms.Key(this, 'Key', {
                description: `Created by ${this.node.path}`,
            });
            const streamEncryption = {
                encryptionType: 'KMS',
                keyId: encryptionKey.keyArn,
            };
            return { encryptionKey, streamEncryption };
        }
        throw new Error(`Unexpected 'encryptionType': ${encryptionType}`);
    }
}
exports.Stream = Stream;
/**
 * (experimental) What kind of server-side encryption to apply to this stream.
 *
 * @experimental
 */
var StreamEncryption;
(function (StreamEncryption) {
    StreamEncryption["UNENCRYPTED"] = "NONE";
    StreamEncryption["KMS"] = "KMS";
    StreamEncryption["MANAGED"] = "MANAGED";
})(StreamEncryption = exports.StreamEncryption || (exports.StreamEncryption = {}));
//# sourceMappingURL=data:application/json;base64,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