"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AcceleratorSecurityGroup = void 0;
const aws_ec2_1 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const custom_resources_1 = require("../../custom-resources"); // Automatically re-written from '@aws-cdk/custom-resources'
/**
 * (experimental) The security group used by a Global Accelerator to send traffic to resources in a VPC.
 *
 * @experimental
 */
class AcceleratorSecurityGroup {
    /**
     * (experimental) Lookup the Global Accelerator security group at CloudFormation deployment time.
     *
     * As of this writing, Global Accelerators (AGA) create a single security group per VPC. AGA security groups are shared
     * by all AGAs in an account. Additionally, there is no CloudFormation mechanism to reference the AGA security groups.
     *
     * This makes creating security group rules which allow traffic from an AGA complicated in CDK. This lookup will identify
     * the AGA security group for a given VPC at CloudFormation deployment time, and lets you create rules for traffic from AGA
     * to other resources created by CDK.
     *
     * @experimental
     */
    static fromVpc(scope, id, vpc, endpointGroup) {
        // The security group name is always 'GlobalAccelerator'
        const globalAcceleratorSGName = 'GlobalAccelerator';
        // How to reference the security group name in the response from EC2
        const ec2ResponseSGIdField = 'SecurityGroups.0.GroupId';
        // The AWS Custom Resource that make a call to EC2 to get the security group ID, for the given VPC
        const lookupAcceleratorSGCustomResource = new custom_resources_1.AwsCustomResource(scope, id + 'CustomResource', {
            onCreate: {
                service: 'EC2',
                action: 'describeSecurityGroups',
                parameters: {
                    Filters: [
                        {
                            Name: 'group-name',
                            Values: [
                                globalAcceleratorSGName,
                            ],
                        },
                        {
                            Name: 'vpc-id',
                            Values: [
                                vpc.vpcId,
                            ],
                        },
                    ],
                },
                // We get back a list of responses, but the list should be of length 0 or 1
                // Getting no response means no resources have been linked to the AGA
                physicalResourceId: custom_resources_1.PhysicalResourceId.fromResponse(ec2ResponseSGIdField),
            },
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                resources: custom_resources_1.AwsCustomResourcePolicy.ANY_RESOURCE,
            }),
        });
        // Look up the security group ID
        const sg = aws_ec2_1.SecurityGroup.fromSecurityGroupId(scope, id, lookupAcceleratorSGCustomResource.getResponseField(ec2ResponseSGIdField));
        // We add a dependency on the endpoint group, guaranteeing that CloudFormation won't
        // try and look up the SG before AGA creates it. The SG is created when a VPC resource
        // is associated with an AGA
        lookupAcceleratorSGCustomResource.node.addDependency(endpointGroup);
        return sg;
    }
    constructor() { }
}
exports.AcceleratorSecurityGroup = AcceleratorSecurityGroup;
//# sourceMappingURL=data:application/json;base64,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