"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.addLambdaPermission = exports.singletonEventRole = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * Obtain the Role for the EventBridge event
 *
 * If a role already exists, it will be returned. This ensures that if multiple
 * events have the same target, they will share a role.
 */
function singletonEventRole(scope, policyStatements) {
    const id = 'EventsRole';
    const existing = scope.node.tryFindChild(id);
    if (existing) {
        return existing;
    }
    const role = new iam.Role(scope, id, {
        assumedBy: new iam.ServicePrincipal('events.amazonaws.com'),
    });
    policyStatements.forEach(role.addToPolicy.bind(role));
    return role;
}
exports.singletonEventRole = singletonEventRole;
/**
 * Allows a Lambda function to be called from a rule
 */
function addLambdaPermission(rule, handler) {
    let scope;
    let node = handler.permissionsNode;
    if (rule instanceof core_1.Construct) {
        // Place the Permission resource in the same stack as Rule rather than the Function
        // This is to reduce circular dependency when the lambda handler and the rule are across stacks.
        scope = rule;
        node = rule.node;
    }
    const permissionId = `AllowEventRule${core_1.Names.nodeUniqueId(rule.node)}`;
    if (!node.tryFindChild(permissionId)) {
        handler.addPermission(permissionId, {
            scope,
            action: 'lambda:InvokeFunction',
            principal: new iam.ServicePrincipal('events.amazonaws.com'),
            sourceArn: rule.ruleArn,
        });
    }
}
exports.addLambdaPermission = addLambdaPermission;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidXRpbC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInV0aWwudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQ0EscUNBQXFDLENBQUMsbURBQW1EO0FBRXpGLHFDQUF5RSxDQUFDLGdEQUFnRDtBQUMxSDs7Ozs7R0FLRztBQUNILFNBQWdCLGtCQUFrQixDQUFDLEtBQWlCLEVBQUUsZ0JBQXVDO0lBQ3pGLE1BQU0sRUFBRSxHQUFHLFlBQVksQ0FBQztJQUN4QixNQUFNLFFBQVEsR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxFQUFFLENBQWMsQ0FBQztJQUMxRCxJQUFJLFFBQVEsRUFBRTtRQUNWLE9BQU8sUUFBUSxDQUFDO0tBQ25CO0lBQ0QsTUFBTSxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQWtCLEVBQUUsRUFBRSxFQUFFO1FBQzlDLFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxzQkFBc0IsQ0FBQztLQUM5RCxDQUFDLENBQUM7SUFDSCxnQkFBZ0IsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQztJQUN0RCxPQUFPLElBQUksQ0FBQztBQUNoQixDQUFDO0FBWEQsZ0RBV0M7QUFDRDs7R0FFRztBQUNILFNBQWdCLG1CQUFtQixDQUFDLElBQWtCLEVBQUUsT0FBeUI7SUFDN0UsSUFBSSxLQUE0QixDQUFDO0lBQ2pDLElBQUksSUFBSSxHQUFrQixPQUFPLENBQUMsZUFBZSxDQUFDO0lBQ2xELElBQUksSUFBSSxZQUFZLGdCQUFTLEVBQUU7UUFDM0IsbUZBQW1GO1FBQ25GLGdHQUFnRztRQUNoRyxLQUFLLEdBQUcsSUFBSSxDQUFDO1FBQ2IsSUFBSSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUM7S0FDcEI7SUFDRCxNQUFNLFlBQVksR0FBRyxpQkFBaUIsWUFBSyxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQztJQUN0RSxJQUFJLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxZQUFZLENBQUMsRUFBRTtRQUNsQyxPQUFPLENBQUMsYUFBYSxDQUFDLFlBQVksRUFBRTtZQUNoQyxLQUFLO1lBQ0wsTUFBTSxFQUFFLHVCQUF1QjtZQUMvQixTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsc0JBQXNCLENBQUM7WUFDM0QsU0FBUyxFQUFFLElBQUksQ0FBQyxPQUFPO1NBQzFCLENBQUMsQ0FBQztLQUNOO0FBQ0wsQ0FBQztBQWxCRCxrREFrQkMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBldmVudHMgZnJvbSBcIi4uLy4uL2F3cy1ldmVudHNcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1ldmVudHMnXG5pbXBvcnQgKiBhcyBpYW0gZnJvbSBcIi4uLy4uL2F3cy1pYW1cIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nXG5pbXBvcnQgKiBhcyBsYW1iZGEgZnJvbSBcIi4uLy4uL2F3cy1sYW1iZGFcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1sYW1iZGEnXG5pbXBvcnQgeyBDb25zdHJ1Y3QsIENvbnN0cnVjdE5vZGUsIElDb25zdHJ1Y3QsIE5hbWVzIH0gZnJvbSBcIi4uLy4uL2NvcmVcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2NvcmUnXG4vKipcbiAqIE9idGFpbiB0aGUgUm9sZSBmb3IgdGhlIEV2ZW50QnJpZGdlIGV2ZW50XG4gKlxuICogSWYgYSByb2xlIGFscmVhZHkgZXhpc3RzLCBpdCB3aWxsIGJlIHJldHVybmVkLiBUaGlzIGVuc3VyZXMgdGhhdCBpZiBtdWx0aXBsZVxuICogZXZlbnRzIGhhdmUgdGhlIHNhbWUgdGFyZ2V0LCB0aGV5IHdpbGwgc2hhcmUgYSByb2xlLlxuICovXG5leHBvcnQgZnVuY3Rpb24gc2luZ2xldG9uRXZlbnRSb2xlKHNjb3BlOiBJQ29uc3RydWN0LCBwb2xpY3lTdGF0ZW1lbnRzOiBpYW0uUG9saWN5U3RhdGVtZW50W10pOiBpYW0uSVJvbGUge1xuICAgIGNvbnN0IGlkID0gJ0V2ZW50c1JvbGUnO1xuICAgIGNvbnN0IGV4aXN0aW5nID0gc2NvcGUubm9kZS50cnlGaW5kQ2hpbGQoaWQpIGFzIGlhbS5JUm9sZTtcbiAgICBpZiAoZXhpc3RpbmcpIHtcbiAgICAgICAgcmV0dXJuIGV4aXN0aW5nO1xuICAgIH1cbiAgICBjb25zdCByb2xlID0gbmV3IGlhbS5Sb2xlKHNjb3BlIGFzIENvbnN0cnVjdCwgaWQsIHtcbiAgICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2V2ZW50cy5hbWF6b25hd3MuY29tJyksXG4gICAgfSk7XG4gICAgcG9saWN5U3RhdGVtZW50cy5mb3JFYWNoKHJvbGUuYWRkVG9Qb2xpY3kuYmluZChyb2xlKSk7XG4gICAgcmV0dXJuIHJvbGU7XG59XG4vKipcbiAqIEFsbG93cyBhIExhbWJkYSBmdW5jdGlvbiB0byBiZSBjYWxsZWQgZnJvbSBhIHJ1bGVcbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGFkZExhbWJkYVBlcm1pc3Npb24ocnVsZTogZXZlbnRzLklSdWxlLCBoYW5kbGVyOiBsYW1iZGEuSUZ1bmN0aW9uKTogdm9pZCB7XG4gICAgbGV0IHNjb3BlOiBDb25zdHJ1Y3QgfCB1bmRlZmluZWQ7XG4gICAgbGV0IG5vZGU6IENvbnN0cnVjdE5vZGUgPSBoYW5kbGVyLnBlcm1pc3Npb25zTm9kZTtcbiAgICBpZiAocnVsZSBpbnN0YW5jZW9mIENvbnN0cnVjdCkge1xuICAgICAgICAvLyBQbGFjZSB0aGUgUGVybWlzc2lvbiByZXNvdXJjZSBpbiB0aGUgc2FtZSBzdGFjayBhcyBSdWxlIHJhdGhlciB0aGFuIHRoZSBGdW5jdGlvblxuICAgICAgICAvLyBUaGlzIGlzIHRvIHJlZHVjZSBjaXJjdWxhciBkZXBlbmRlbmN5IHdoZW4gdGhlIGxhbWJkYSBoYW5kbGVyIGFuZCB0aGUgcnVsZSBhcmUgYWNyb3NzIHN0YWNrcy5cbiAgICAgICAgc2NvcGUgPSBydWxlO1xuICAgICAgICBub2RlID0gcnVsZS5ub2RlO1xuICAgIH1cbiAgICBjb25zdCBwZXJtaXNzaW9uSWQgPSBgQWxsb3dFdmVudFJ1bGUke05hbWVzLm5vZGVVbmlxdWVJZChydWxlLm5vZGUpfWA7XG4gICAgaWYgKCFub2RlLnRyeUZpbmRDaGlsZChwZXJtaXNzaW9uSWQpKSB7XG4gICAgICAgIGhhbmRsZXIuYWRkUGVybWlzc2lvbihwZXJtaXNzaW9uSWQsIHtcbiAgICAgICAgICAgIHNjb3BlLFxuICAgICAgICAgICAgYWN0aW9uOiAnbGFtYmRhOkludm9rZUZ1bmN0aW9uJyxcbiAgICAgICAgICAgIHByaW5jaXBhbDogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdldmVudHMuYW1hem9uYXdzLmNvbScpLFxuICAgICAgICAgICAgc291cmNlQXJuOiBydWxlLnJ1bGVBcm4sXG4gICAgICAgIH0pO1xuICAgIH1cbn1cbiJdfQ==