"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.validateNetworkProtocol = exports.ifUndefined = exports.determineProtocolAndPort = exports.defaultProtocolForPort = exports.defaultPortForProtocol = exports.renderAttributes = void 0;
const enums_1 = require("./enums");
/**
 * Render an attribute dict to a list of { key, value } pairs
 */
function renderAttributes(attributes) {
    const ret = [];
    for (const [key, value] of Object.entries(attributes)) {
        if (value !== undefined) {
            ret.push({ key, value });
        }
    }
    return ret;
}
exports.renderAttributes = renderAttributes;
/**
 * Return the appropriate default port for a given protocol
 */
function defaultPortForProtocol(proto) {
    switch (proto) {
        case enums_1.ApplicationProtocol.HTTP: return 80;
        case enums_1.ApplicationProtocol.HTTPS: return 443;
        default:
            throw new Error(`Unrecognized protocol: ${proto}`);
    }
}
exports.defaultPortForProtocol = defaultPortForProtocol;
/**
 * Return the appropriate default protocol for a given port
 */
function defaultProtocolForPort(port) {
    switch (port) {
        case 80:
        case 8000:
        case 8008:
        case 8080:
            return enums_1.ApplicationProtocol.HTTP;
        case 443:
        case 8443:
            return enums_1.ApplicationProtocol.HTTPS;
        default:
            throw new Error(`Don't know default protocol for port: ${port}; please supply a protocol`);
    }
}
exports.defaultProtocolForPort = defaultProtocolForPort;
/**
 * Given a protocol and a port, try to guess the other one if it's undefined
 */
// eslint-disable-next-line max-len
function determineProtocolAndPort(protocol, port) {
    if (protocol === undefined && port === undefined) {
        return [undefined, undefined];
    }
    if (protocol === undefined) {
        protocol = defaultProtocolForPort(port);
    }
    if (port === undefined) {
        port = defaultPortForProtocol(protocol);
    }
    return [protocol, port];
}
exports.determineProtocolAndPort = determineProtocolAndPort;
/**
 * Helper function to default undefined input props
 */
function ifUndefined(x, def) {
    return x !== undefined ? x : def;
}
exports.ifUndefined = ifUndefined;
/**
 * Helper function for ensuring network listeners and target groups only accept valid
 * protocols.
 */
function validateNetworkProtocol(protocol) {
    const NLB_PROTOCOLS = [enums_1.Protocol.TCP, enums_1.Protocol.TLS, enums_1.Protocol.UDP, enums_1.Protocol.TCP_UDP];
    if (NLB_PROTOCOLS.indexOf(protocol) === -1) {
        throw new Error(`The protocol must be one of ${NLB_PROTOCOLS.join(', ')}. Found ${protocol}`);
    }
}
exports.validateNetworkProtocol = validateNetworkProtocol;
//# sourceMappingURL=data:application/json;base64,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