"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NetworkTargetGroup = void 0;
const cloudwatch = require("../../../aws-cloudwatch"); // Automatically re-written from '@aws-cdk/aws-cloudwatch'
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const base_target_group_1 = require("../shared/base-target-group");
const enums_1 = require("../shared/enums");
const imported_1 = require("../shared/imported");
const util_1 = require("../shared/util");
/**
 * (experimental) Define a Network Target Group.
 *
 * @experimental
 */
class NetworkTargetGroup extends base_target_group_1.TargetGroupBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        const proto = props.protocol || enums_1.Protocol.TCP;
        util_1.validateNetworkProtocol(proto);
        super(scope, id, props, {
            protocol: proto,
            port: props.port,
        });
        this.listeners = [];
        if (props.proxyProtocolV2 != null) {
            this.setAttribute('proxy_protocol_v2.enabled', props.proxyProtocolV2 ? 'true' : 'false');
        }
        this.addTarget(...(props.targets || []));
    }
    /**
     * (experimental) Import an existing target group.
     *
     * @experimental
     */
    static fromTargetGroupAttributes(scope, id, attrs) {
        return new ImportedNetworkTargetGroup(scope, id, attrs);
    }
    /**
     * (deprecated) Import an existing listener.
     *
     * @deprecated Use `fromTargetGroupAttributes` instead
     */
    static import(scope, id, props) {
        return NetworkTargetGroup.fromTargetGroupAttributes(scope, id, props);
    }
    /**
     * (experimental) Add a load balancing target to this target group.
     *
     * @experimental
     */
    addTarget(...targets) {
        for (const target of targets) {
            const result = target.attachToNetworkTargetGroup(this);
            this.addLoadBalancerTarget(result);
        }
    }
    /**
     * (experimental) Register a listener that is load balancing to this target group.
     *
     * Don't call this directly. It will be called by listeners.
     *
     * @experimental
     */
    registerListener(listener) {
        this.loadBalancerAttachedDependencies.add(listener);
        this.listeners.push(listener);
    }
    /**
     * (experimental) The number of targets that are considered healthy.
     *
     * @default Average over 5 minutes
     * @experimental
     */
    metricHealthyHostCount(props) {
        return this.metric('HealthyHostCount', {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * (experimental) The number of targets that are considered unhealthy.
     *
     * @default Average over 5 minutes
     * @experimental
     */
    metricUnHealthyHostCount(props) {
        return this.metric('UnHealthyHostCount', {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * (experimental) Full name of first load balancer.
     *
     * @experimental
     */
    get firstLoadBalancerFullName() {
        if (this.listeners.length === 0) {
            throw new Error('The TargetGroup needs to be attached to a LoadBalancer before you can call this method');
        }
        return base_target_group_1.loadBalancerNameFromListenerArn(this.listeners[0].listenerArn);
    }
    /**
     * (experimental) Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @experimental
     */
    validate() {
        const ret = super.validate();
        const healthCheck = this.healthCheck || {};
        const allowedIntervals = [10, 30];
        if (healthCheck.interval) {
            const seconds = healthCheck.interval.toSeconds();
            if (!cdk.Token.isUnresolved(seconds) && !allowedIntervals.includes(seconds)) {
                ret.push(`Health check interval '${seconds}' not supported. Must be one of the following values '${allowedIntervals.join(',')}'.`);
            }
        }
        if (healthCheck.healthyThresholdCount) {
            const thresholdCount = healthCheck.healthyThresholdCount;
            if (thresholdCount < 2 || thresholdCount > 10) {
                ret.push(`Healthy Threshold Count '${thresholdCount}' not supported. Must be a number between 2 and 10.`);
            }
        }
        if (healthCheck.unhealthyThresholdCount) {
            const thresholdCount = healthCheck.unhealthyThresholdCount;
            if (thresholdCount < 2 || thresholdCount > 10) {
                ret.push(`Unhealthy Threshold Count '${thresholdCount}' not supported. Must be a number between 2 and 10.`);
            }
        }
        if (healthCheck.healthyThresholdCount && healthCheck.unhealthyThresholdCount &&
            healthCheck.healthyThresholdCount !== healthCheck.unhealthyThresholdCount) {
            ret.push([
                `Healthy and Unhealthy Threshold Counts must be the same: ${healthCheck.healthyThresholdCount}`,
                `is not equal to ${healthCheck.unhealthyThresholdCount}.`,
            ].join(' '));
        }
        if (!healthCheck.protocol) {
            return ret;
        }
        if (!NLB_HEALTH_CHECK_PROTOCOLS.includes(healthCheck.protocol)) {
            ret.push(`Health check protocol '${healthCheck.protocol}' is not supported. Must be one of [${NLB_HEALTH_CHECK_PROTOCOLS.join(', ')}]`);
        }
        if (healthCheck.path && !NLB_PATH_HEALTH_CHECK_PROTOCOLS.includes(healthCheck.protocol)) {
            ret.push([
                `'${healthCheck.protocol}' health checks do not support the path property.`,
                `Must be one of [${NLB_PATH_HEALTH_CHECK_PROTOCOLS.join(', ')}]`,
            ].join(' '));
        }
        if (healthCheck.timeout && healthCheck.timeout.toSeconds() !== NLB_HEALTH_CHECK_TIMEOUTS[healthCheck.protocol]) {
            ret.push([
                'Custom health check timeouts are not supported for Network Load Balancer health checks.',
                `Expected ${NLB_HEALTH_CHECK_TIMEOUTS[healthCheck.protocol]} seconds for ${healthCheck.protocol}, got ${healthCheck.timeout.toSeconds()}`,
            ].join(' '));
        }
        return ret;
    }
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/NetworkELB',
            metricName,
            dimensions: { LoadBalancer: this.firstLoadBalancerFullName, TargetGroup: this.targetGroupFullName },
            ...props,
        }).attachTo(this);
    }
}
exports.NetworkTargetGroup = NetworkTargetGroup;
/**
 * An imported network target group
 */
class ImportedNetworkTargetGroup extends imported_1.ImportedTargetGroupBase {
    registerListener(_listener) {
        // Nothing to do, we know nothing of our members
    }
    addTarget(...targets) {
        for (const target of targets) {
            const result = target.attachToNetworkTargetGroup(this);
            if (result.targetJson !== undefined) {
                throw new Error('Cannot add a non-self registering target to an imported TargetGroup. Create a new TargetGroup instead.');
            }
        }
    }
}
const NLB_HEALTH_CHECK_PROTOCOLS = [enums_1.Protocol.HTTP, enums_1.Protocol.HTTPS, enums_1.Protocol.TCP];
const NLB_PATH_HEALTH_CHECK_PROTOCOLS = [enums_1.Protocol.HTTP, enums_1.Protocol.HTTPS];
const NLB_HEALTH_CHECK_TIMEOUTS = {
    [enums_1.Protocol.HTTP]: 6,
    [enums_1.Protocol.HTTPS]: 10,
    [enums_1.Protocol.TCP]: 10,
};
//# sourceMappingURL=data:application/json;base64,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